package ai.cheq.sst.android.models.advertising

import ai.cheq.sst.android.core.ContextProvider
import ai.cheq.sst.android.core.models.Constants.UNKNOWN
import ai.cheq.sst.android.core.models.Model
import ai.cheq.sst.android.core.models.ModelContext
import com.fasterxml.jackson.annotation.JsonCreator
import com.fasterxml.jackson.annotation.JsonProperty
import com.google.android.gms.ads.identifier.AdvertisingIdClient
import kotlinx.coroutines.Job
import kotlinx.coroutines.cancelAndJoin
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.cancellable
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.launch
import kotlin.time.Duration.Companion.seconds

/**
 * Model for collecting advertising ID and related information such as the limit ad tracking setting.
 *
 * @sample samples.advertising.AdvertisingModel.Usage.model
 */
class AdvertisingModel : Model<AdvertisingModel.Data> {
    private val advertisingIdLoaderProvider: (contextProvider: ContextProvider) -> AdvertisingIdLoader
    private lateinit var collector: AdvertisingIdCollector
    private lateinit var collectorFlow: Flow<Data>
    private lateinit var collectorJob: Job
    private lateinit var data: Data

    /**
     * Constructs a new instance of the model.
     */
    constructor() : this({ GooglePlayServicesAdvertisingIdLoader(it) })

    internal constructor(
        advertisingIdLoaderProvider: (context: ContextProvider) -> AdvertisingIdLoader = {
            GooglePlayServicesAdvertisingIdLoader(
                it
            )
        }
    ) : super(
        Data::class, BuildConfig.LIBRARY_NAME, BuildConfig.LIBRARY_VERSION
    ) {
        this.advertisingIdLoaderProvider = advertisingIdLoaderProvider
    }

    /**
     * Initializes the model with the provided context.
     *
     * @param modelContext The context for managing models.
     */
    override fun initialize(modelContext: ModelContext) {
        if (!this::collector.isInitialized) {
            collector =
                AdvertisingIdCollector(advertisingIdLoaderProvider(modelContext.contextProvider))
        }
        if (!this::data.isInitialized) {
            modelContext.modelScope.launch(modelContext.modelDispatcher) {
                data = collector.collect(modelContext)
            }
        }
        if (!this::collectorFlow.isInitialized) {
            collectorFlow = flow {
                while (true) {
                    delay(30.seconds)
                    emit(collector.collect(modelContext))
                }
            }.cancellable()
            collectorJob = collectorFlow.onEach { data = it }.launchIn(modelContext.modelScope)
        }
    }

    /**
     * Retrieves the data associated with this model.
     *
     * @param modelContext The context in which the data is being retrieved.
     * @return The data associated with the model, or null if no data is found.
     */
    override suspend fun get(modelContext: ModelContext): Data? {
        if (!this::data.isInitialized) {
            return null
        }
        return data
    }

    internal suspend fun stop() {
        if (this::collectorJob.isInitialized) {
            collectorJob.cancelAndJoin()
        }
    }

    /**
     * Data class representing the advertising model data.
     *
     * @property advertisingId The advertising ID.
     * @property enabled Indicates whether ad tracking is enabled.
     */
    class Data @JsonCreator internal constructor(
        @param:JsonProperty("advertisingId") @get:JsonProperty("advertisingId") val advertisingId: String,
        @param:JsonProperty("enabled") @get:JsonProperty("enabled") val enabled: Boolean
    ) : Model.Data()

    private class GooglePlayServicesAdvertisingIdLoader(contextProvider: ContextProvider) :
        AdvertisingIdLoader(contextProvider) {
        override fun getAdvertisingIdInfo(): AdvertisingIdClient.Info {
            return AdvertisingIdClient.getAdvertisingIdInfo(contextProvider.context())
        }
    }

    private enum class CollectionStatus {
        ERROR, WARN, SUCCESS
    }

    private data class CollectionResult(
        val status: CollectionStatus,
        val message: String,
        val advertisingId: String,
        val enabled: Boolean,
        val throwable: Throwable? = null
    )

    private class AdvertisingIdCollector(private val loader: AdvertisingIdLoader) {
        private lateinit var result: CollectionResult

        fun collect(modelContext: ModelContext): Data {
            val newResult = collectInternal()
            if (!this::result.isInitialized || result != newResult) {
                when (newResult.status) {
                    CollectionStatus.ERROR -> modelContext.log.e(
                        newResult.message, newResult.throwable!!
                    )

                    CollectionStatus.WARN -> modelContext.log.w(newResult.message)
                    CollectionStatus.SUCCESS -> modelContext.log.i(newResult.message)
                }
                result = newResult
            }
            return Data(result.advertisingId, result.enabled)
        }

        private fun collectInternal(): CollectionResult {
            try {
                val advertisingInfo = loader.getAdvertisingIdInfo()
                val isLimitAdTrackingEnabled = advertisingInfo.isLimitAdTrackingEnabled
                if (isLimitAdTrackingEnabled) {
                    return CollectionResult(
                        CollectionStatus.WARN,
                        "Not collecting advertising ID because limit Ad tracking is enabled",
                        UNKNOWN,
                        false
                    )
                }
                if (advertisingInfo.id == null) {
                    return CollectionResult(
                        CollectionStatus.WARN,
                        "Not collecting advertising ID because it was not returned",
                        UNKNOWN,
                        true
                    )
                }
                return CollectionResult(
                    CollectionStatus.SUCCESS,
                    "Advertising ID collected",
                    advertisingInfo.id!!,
                    true
                )
            } catch (e: Exception) {
                return CollectionResult(
                    CollectionStatus.ERROR, "Failed to get advertising ID", UNKNOWN, true, e
                )
            }
        }
    }
}