package ai.cheq.sst.android.models.advertising

import ai.cheq.sst.android.core.ContextProvider
import ai.cheq.sst.android.core.models.Constants.UNKNOWN
import ai.cheq.sst.android.core.models.Model
import ai.cheq.sst.android.core.models.ModelContext
import com.fasterxml.jackson.annotation.JsonCreator
import com.fasterxml.jackson.annotation.JsonProperty
import com.google.android.gms.ads.identifier.AdvertisingIdClient
import kotlinx.coroutines.Job
import kotlinx.coroutines.channels.BufferOverflow
import kotlinx.coroutines.channels.Channel
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.cancellable
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.flow.receiveAsFlow
import kotlinx.coroutines.launch
import kotlin.time.Duration.Companion.seconds

/**
 * Model for collecting advertising ID and related information such as the limit ad tracking setting.
 *
 * @sample samples.advertising.AdvertisingModel.Usage.model
 */
class AdvertisingModel : Model<AdvertisingModel.Data> {
    private val advertisingIdLoaderProvider: (contextProvider: ContextProvider) -> AdvertisingIdLoader
    private lateinit var collector: AdvertisingIdCollector
    private lateinit var collectorJob: Job

    @Volatile
    private lateinit var data: Data

    /**
     * Constructs a new instance of the model.
     */
    constructor() : this({ GooglePlayServicesAdvertisingIdLoader(it) })

    internal constructor(
        advertisingIdLoaderProvider: (context: ContextProvider) -> AdvertisingIdLoader = {
            GooglePlayServicesAdvertisingIdLoader(
                it
            )
        }
    ) : super(
        Data::class, BuildConfig.LIBRARY_NAME, BuildConfig.LIBRARY_VERSION
    ) {
        this.advertisingIdLoaderProvider = advertisingIdLoaderProvider
    }

    /**
     * Sets up the model with the specified [ModelContext].
     *
     * @param modelContext The context in which the model is being set up.
     */
    override suspend fun initialize(modelContext: ModelContext) {
        super.initialize(modelContext)
        if (!this::collector.isInitialized) {
            collector =
                AdvertisingIdCollector(advertisingIdLoaderProvider(modelContext.contextProvider))
        }
        if (!this::data.isInitialized) {
            val advertisingId =
                AdvertisingRepository.readAdvertisingId(modelContext.contextProvider)
            data = if (!advertisingId.isNullOrEmpty()) {
                Data(advertisingId, true)
            } else {
                Data(UNKNOWN, false)
            }
        }
        if (!this::collectorJob.isInitialized) {
            val serializeChannel = Channel<Data?>(1, BufferOverflow.DROP_LATEST)
            collectorJob = modelContext.modelScope.launch(modelContext.modelDispatcher) {
                serializeChannel.receiveAsFlow().cancellable().onEach {
                        if (it == null) {
                            modelContext.log.i("Clearing advertising ID")
                            AdvertisingRepository.clearAdvertisingId(modelContext.contextProvider)
                        } else {
                            modelContext.log.i("Storing advertising ID: ${it.advertisingId}")
                            AdvertisingRepository.storeAdvertisingId(
                                modelContext.contextProvider, it.advertisingId
                            )
                        }
                    }.launchIn(this)

                while (true) {
                    val priorAdvertisingId = data.advertisingId
                    val newData = collector.collect(modelContext)
                    data = newData
                    if (newData.advertisingId != priorAdvertisingId) {
                        if (newData.enabled && newData.advertisingId != UNKNOWN) {
                            serializeChannel.send(newData)
                        } else {
                            serializeChannel.send(null)
                        }
                    }

                    delay(30.seconds)
                }
            }
        }
    }

    /**
     * Retrieves the data associated with this model.
     *
     * @param modelContext The context in which the data is being retrieved.
     * @return The data associated with the model, or null if no data is found.
     */
    override suspend fun get(modelContext: ModelContext): Data? {
        if (!this::data.isInitialized) {
            return null
        }
        return data
    }

    /**
     * Stops the model and releases any resources it holds.
     *
     * @param modelContext The context in which the data is being retrieved.
     */
    override suspend fun stop(modelContext: ModelContext) {
        if (this::collectorJob.isInitialized) {
            collectorJob.cancel()
        }
    }

    /**
     * Data class representing the advertising model data.
     *
     * @property advertisingId The advertising ID.
     * @property enabled Indicates whether ad tracking is enabled.
     */
    class Data @JsonCreator internal constructor(
        @param:JsonProperty("advertisingId") @get:JsonProperty("advertisingId") val advertisingId: String,
        @param:JsonProperty("enabled") @get:JsonProperty("enabled") val enabled: Boolean
    ) : Model.Data()

    private class GooglePlayServicesAdvertisingIdLoader(contextProvider: ContextProvider) :
        AdvertisingIdLoader(contextProvider) {
        override fun getAdvertisingIdInfo(): AdvertisingIdClient.Info {
            return AdvertisingIdClient.getAdvertisingIdInfo(contextProvider.context())
        }
    }

    private enum class CollectionStatus {
        ERROR, WARN, SUCCESS
    }

    private data class CollectionResult(
        val status: CollectionStatus,
        val message: String,
        val advertisingId: String,
        val enabled: Boolean,
        val throwable: Throwable? = null
    )

    private class AdvertisingIdCollector(private val loader: AdvertisingIdLoader) {
        private lateinit var result: CollectionResult

        fun collect(modelContext: ModelContext): Data {
            val newResult = collectInternal()
            if (!this::result.isInitialized || result != newResult) {
                when (newResult.status) {
                    CollectionStatus.ERROR -> modelContext.log.e(
                        newResult.message, newResult.throwable!!
                    )

                    CollectionStatus.WARN -> modelContext.log.w(newResult.message)
                    CollectionStatus.SUCCESS -> modelContext.log.i(newResult.message)
                }
                result = newResult
            }
            return Data(result.advertisingId, result.enabled)
        }

        private fun collectInternal(): CollectionResult {
            try {
                val advertisingInfo = loader.getAdvertisingIdInfo()
                val isLimitAdTrackingEnabled = advertisingInfo.isLimitAdTrackingEnabled
                if (isLimitAdTrackingEnabled) {
                    return CollectionResult(
                        CollectionStatus.WARN,
                        "Not collecting advertising ID because limit Ad tracking is enabled",
                        UNKNOWN,
                        false
                    )
                }
                if (advertisingInfo.id == null) {
                    return CollectionResult(
                        CollectionStatus.WARN,
                        "Not collecting advertising ID because it was not returned",
                        UNKNOWN,
                        true
                    )
                }
                return CollectionResult(
                    CollectionStatus.SUCCESS,
                    "Advertising ID collected",
                    advertisingInfo.id!!,
                    true
                )
            } catch (e: Exception) {
                return CollectionResult(
                    CollectionStatus.ERROR, "Failed to get advertising ID", UNKNOWN, true, e
                )
            }
        }
    }
}