package ai.cheq.sst.android.core.models

import ai.cheq.sst.android.core.BuildConfig
import ai.cheq.sst.android.core.ContextProvider
import ai.cheq.sst.android.core.models.DeviceModel.Data.Os
import ai.cheq.sst.android.core.models.DeviceModel.Data.Screen
import ai.cheq.sst.android.core.settings.SettingsRepository
import android.content.Context
import android.content.res.Configuration
import android.graphics.Point
import android.hardware.display.DisplayManager
import android.os.Build
import android.view.Display
import android.view.Surface
import android.view.WindowManager
import android.webkit.WebSettings
import com.fasterxml.jackson.annotation.JsonAutoDetect
import com.fasterxml.jackson.annotation.JsonProperty
import com.google.android.gms.appset.AppSet
import com.google.android.gms.appset.AppSetIdClient
import kotlinx.coroutines.TimeoutCancellationException
import kotlinx.coroutines.tasks.await
import java.time.Clock
import java.util.Locale

/**
 * Represents the device model that collects and exposes device-related data.
 */
class DeviceModel internal constructor() :
    Model<DeviceModel.Data>(Data::class, Constants.MODEL_KEY_DEVICE, BuildConfig.LIBRARY_VERSION) {
    private lateinit var builder: Builder
    private lateinit var windowManager: WindowManager
    private lateinit var orientationProvider: OrientationProvider
    private lateinit var appSetIdClient: AppSetIdClient
    private lateinit var userAgent: String

    override fun initialize(modelContext: ModelContext) {
        super.initialize(modelContext)
        windowManager = modelContext.contextProvider.context()
            .getSystemService(Context.WINDOW_SERVICE) as WindowManager
        orientationProvider = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
            OrientationProviderImplR(modelContext.contextProvider)
        } else {
            OrientationProviderImpl(modelContext.contextProvider)
        }
        appSetIdClient = AppSet.getClient(modelContext.contextProvider.context())
        userAgent = WebSettings.getDefaultUserAgent(modelContext.contextProvider.context())
    }

    /**
     * Retrieves the data associated with this model.
     *
     * @param modelContext The context in which the data is being retrieved.
     * @return The data associated with the model.
     */
    override suspend fun get(modelContext: ModelContext): Data {
        if (!this::builder.isInitialized) {
            builder = Builder(
                modelContext.clock, windowManager, orientationProvider, appSetIdClient, userAgent
            )
        }
        return builder.build(modelContext.contextProvider)
    }

    private class Builder(
        private val clock: () -> Clock,
        private val windowManager: WindowManager,
        private val orientationProvider: OrientationProvider,
        private val appSetIdClient: AppSetIdClient,
        private val userAgent: String
    ) {
        private val resolution: Pair<Int, Int>
            get() = when (Build.VERSION.SDK_INT) {
                in Build.VERSION_CODES.R..Int.MAX_VALUE -> Pair(
                    windowManager.currentWindowMetrics.bounds.width(),
                    windowManager.currentWindowMetrics.bounds.height()
                )

                else -> {
                    val size = Point()
                    @Suppress("DEPRECATION") windowManager.defaultDisplay.getSize(size)
                    Pair(size.x, size.y)
                }
            }

        private val orientation: String
            get() = try {
                orientationProvider.getOrientation()
            } catch (ex: Exception) {
                Constants.UNKNOWN
            }

        private val screen: Screen
            get() = Screen(resolution.first, resolution.second, orientation)

        private suspend fun id(contextProvider: ContextProvider): String {
            val deviceId = SettingsRepository.readDeviceUuid(contextProvider)
            if (deviceId != null) {
                return deviceId
            }
            try {
                val newDeviceId = appSetIdClient.appSetIdInfo.await().id
                SettingsRepository.storeDeviceUuid(contextProvider, newDeviceId)
                return newDeviceId
            } catch (ex: TimeoutCancellationException) {
                return Constants.UNKNOWN
            }
        }

        private val language: String get() = Locale.getDefault().toLanguageTag()

        private val timezone: String get() = clock().zone.id

        suspend fun build(contextProvider: ContextProvider): Data {
            return Data(id(contextProvider), screen, userAgent, language, timezone)
        }
    }

    /**
     * Data class representing the device model data.
     *
     * This class holds various information about a device and its environment,
     * including device ID, screen details, and OS information.
     *
     * @property id The unique identifier of the device.
     * @property screen Information about the device's screen (see [Screen]).
     * @property manufacturer The device manufacturer (e.g., "Samsung").
     * @property model The device model (e.g., "Galaxy S21").
     * @property architecture The device's CPU architecture (e.g., "arm64-v8a").
     * @property os Information about the operating system (see [Os]).
     */
    class Data internal constructor(
        @field:JsonProperty val id: String, @field:JsonProperty val screen: Screen
    ) : Model.Data() {
        internal constructor(
            id: String, screen: Screen, userAgent: String, language: String, timezone: String
        ) : this(id, screen) {
            this.userAgent = userAgent
            this.language = language
            this.timezone = timezone
        }

        @JsonProperty
        val manufacturer: String = Build.MANUFACTURER

        @JsonProperty
        val model: String = Build.MODEL

        @JsonProperty
        val architecture: String = System.getProperty("os.arch") ?: Constants.UNKNOWN

        @JsonProperty
        val os = Os()

        internal var userAgent: String? = null; private set

        internal var language: String? = null; private set

        internal var timezone: String? = null; private set

        /**
         * Represents information about the operating system.
         *
         * @property name The name of the operating system (always "Android").
         * @property version The version of the operating system (e.g., "12").
         */
        @JsonAutoDetect(
            fieldVisibility = JsonAutoDetect.Visibility.NONE,
            setterVisibility = JsonAutoDetect.Visibility.NONE,
            getterVisibility = JsonAutoDetect.Visibility.NONE,
            isGetterVisibility = JsonAutoDetect.Visibility.NONE,
            creatorVisibility = JsonAutoDetect.Visibility.NONE
        )
        class Os internal constructor() {
            @JsonProperty
            val name = Constants.ANDROID

            @JsonProperty
            val version: String = Build.VERSION.RELEASE
        }

        /**
         * Represents information about the device's screen.
         *
         * @property width The width of the screen in pixels.
         * @property height The height of the screen in pixels.
         * @property orientation The screen orientation (e.g., "portrait" or "landscape").
         * @property depth The color depth of the screen (always 24).
         */
        @JsonAutoDetect(
            fieldVisibility = JsonAutoDetect.Visibility.NONE,
            setterVisibility = JsonAutoDetect.Visibility.NONE,
            getterVisibility = JsonAutoDetect.Visibility.NONE,
            isGetterVisibility = JsonAutoDetect.Visibility.NONE,
            creatorVisibility = JsonAutoDetect.Visibility.NONE
        )
        @ConsistentCopyVisibility
        data class Screen internal constructor(
            @field:JsonProperty val width: Int,
            @field:JsonProperty val height: Int,
            @field:JsonProperty val orientation: String,
        ) {
            val depth: Int = 24
        }
    }

    private interface OrientationProvider {
        fun getOrientation(): String
    }

    private class OrientationProviderImpl(contextProvider: ContextProvider) : OrientationProvider {
        private val resources = contextProvider.context().resources

        override fun getOrientation(): String {
            return when (resources.configuration.orientation) {
                Configuration.ORIENTATION_PORTRAIT -> Constants.SCREEN_ORIENTATION_PORTRAIT
                Configuration.ORIENTATION_LANDSCAPE -> Constants.SCREEN_ORIENTATION_LANDSCAPE
                else -> Constants.UNKNOWN
            }
        }
    }

    private class OrientationProviderImplR(contextProvider: ContextProvider) : OrientationProvider {
        private val displayManager =
            contextProvider.context().getSystemService(Context.DISPLAY_SERVICE) as DisplayManager

        override fun getOrientation(): String {
            return when (displayManager.getDisplay(Display.DEFAULT_DISPLAY)?.rotation ?: 0) {
                Surface.ROTATION_0 -> Constants.SCREEN_ORIENTATION_PORTRAIT
                Surface.ROTATION_90 -> Constants.SCREEN_ORIENTATION_LANDSCAPE_RIGHT
                Surface.ROTATION_180 -> Constants.SCREEN_ORIENTATION_PORTRAIT_UPSIDE_DOWN
                Surface.ROTATION_270 -> Constants.SCREEN_ORIENTATION_LANDSCAPE_LEFT
                else -> Constants.UNKNOWN
            }
        }
    }
}
