package ai.cheq.sst.android.core.models

import ai.cheq.sst.android.core.Config
import ai.cheq.sst.android.core.ContextProvider
import ai.cheq.sst.android.core.Utils
import ai.cheq.sst.android.core.exceptions.DuplicateModelException
import ai.cheq.sst.android.core.exceptions.InvalidModelException
import com.fasterxml.jackson.core.JsonGenerator
import com.fasterxml.jackson.core.JsonParser
import com.fasterxml.jackson.core.JsonToken
import com.fasterxml.jackson.databind.DeserializationContext
import com.fasterxml.jackson.databind.JsonNode
import com.fasterxml.jackson.databind.SerializerProvider
import com.fasterxml.jackson.databind.annotation.JsonSerialize
import com.fasterxml.jackson.databind.deser.std.StdDeserializer
import com.fasterxml.jackson.databind.ser.std.StdSerializer
import kotlin.reflect.KClass

/**
 * The Models class provides a way to manage and access models that collect and expose data.
 * Custom models can be added to the collection as long as they do not conflict with base models.
 * The following base models are always included and cannot be overridden:
 *   * [AppModel]
 *   * [DeviceModel]
 *   * [LibraryModel]
 *
 * @sample samples.core.Models.Usage.defaultModels
 * @sample samples.core.Models.Usage.customModels
 */
class Models {
    /**
     * Creates a new collection of models including the specified custom models.
     *
     * @param customModels Custom models to be added to the collection.
     * @throws InvalidModelException If a base model with the same key already exists.
     * @throws DuplicateModelException If custom model with the same key or type has already been added.
     *
     * @sample samples.core.Models.Usage.defaultModels
     * @sample samples.core.Models.Usage.customModels
     */
    @Throws(InvalidModelException::class, DuplicateModelException::class)
    @Suppress("ConvertSecondaryConstructorToPrimary")
    constructor(vararg customModels: Model<*>) {
        customModels.forEach {
            this.add(it)
        }
    }

    private val baseModelSet: ModelSet = ModelSet(AppModel(), LibraryModel(), DeviceModel())
    private val customModelSet: ModelSet = ModelSet()

    internal fun initialize(config: Config, contextProvider: ContextProvider) {
        val modelContext = getModelContext(config, contextProvider)
        baseModelSet.all().forEach {
            it.initialize(modelContext)
        }
        customModelSet.all().forEach {
            it.initialize(modelContext)
        }
    }

    /**
     * Retrieves a model of the specified class.
     *
     * @param T The type of the model to retrieve.
     * @param clazz The class, which is type [T], of the model to retrieve.
     * @return The model of the specified class, or `null` if not found.
     */
    fun <T : Model<*>> get(clazz: Class<T>): T? =
        baseModelSet.get(clazz) ?: customModelSet.get(clazz)

    /**
     * Retrieves a model of the specified class.
     *
     * @param T The type of the model to retrieve.
     * @return The model of the specified type, or `null` if not found.
     */
    inline fun <reified T : Model<*>> get(): T? = get(T::class.java)

    /**
     * Retrieves all models in the collection.
     *
     * @return A list of all models.
     */
    fun all() = baseModelSet.all() + customModelSet.all()

    /**
     * Adds a custom model to the collection.
     *
     * @param T The type of the model to add.
     * @param model The custom model to add.
     *
     * @throws InvalidModelException If a base model with the same key already exists.
     * @throws DuplicateModelException If custom model with the same key or type has already been added.
     */
    @Throws(InvalidModelException::class, DuplicateModelException::class)
    operator fun <T : Model<*>> plusAssign(model: T) {
        add(model)
    }

    /**
     * Adds a custom model to the collection.
     *
     * @param T The type of the model to add.
     * @param model The custom model to add.
     *
     * @throws InvalidModelException If a base model with the same key already exists.
     * @throws DuplicateModelException If a custom model with the same key or type has already been added.
     */
    @Throws(InvalidModelException::class, DuplicateModelException::class)
    fun <T : Model<*>> add(model: T) {
        if (baseModelSet.containsKey(model.identifier.key) || baseModelSet.containsModel(model::class.java)) {
            throw InvalidModelException("A base model with the key '${model.identifier.key}' already exists and cannot be overridden")
        }
        customModelSet.add(model)
    }

    /**
     * Removes a model of the specified type from the collection.
     *
     * @param T The type of the model to remove.
     */
    inline fun <reified T : Model<*>> remove() = remove(T::class.java)

    /**
     * Removes a model of the specified type from the collection.
     *
     * @param T The type of the model to remove.
     * @param clazz The class, which is type [T], of the model to remove.
     */
    fun <T : Model<*>> remove(clazz: KClass<T>) = customModelSet.remove(clazz.java)

    /**
     * Removes a model of the specified type from the collection.
     *
     * @param T The type of the model to remove.
     * @param clazz The class, which is type [T], of the model to remove.
     */
    fun <T : Model<*>> remove(clazz: Class<T>) = customModelSet.remove(clazz)

    /**
     * Removes a model of the specified type from the collection.
     *
     * @param T The type of the model to remove.
     */
    operator fun <T : Model<*>> minusAssign(model: KClass<T>) {
        remove(model)
    }

    internal suspend fun collect(config: Config, contextProvider: ContextProvider): Data {
        val data = Data()
        val modelContext = getModelContext(config, contextProvider)
        (baseModelSet.all() + customModelSet.all()).forEach {
            val modelData = it.get(modelContext)
            if (modelData != null) {
                data.add(it.identifier.key, it.dataClass, modelData)
            }
        }
        return data
    }

    @JvmName("collectOne")
    internal suspend inline fun <reified T : Model<D>, reified D : Model.Data> collect(
        config: Config, contextProvider: ContextProvider
    ): D? {
        val modelContext = getModelContext(config, contextProvider)
        return get<T>()?.get(modelContext)
    }

    private fun getModelContext(config: Config, contextProvider: ContextProvider): ModelContext {
        val baseModels = baseModelSet.all()
        val customModels = customModelSet.all()
        return ModelContext(
            config.clock,
            contextProvider,
            config.modelScope,
            config.modelDispatcher,
            ModelIdentifiers(baseModels.map { it.identifier }, customModels.map { it.identifier })
        )
    }

    @JsonSerialize(using = Data.Serializer::class)
    internal class Data {
        private val models: HashMap<Class<out Model.Data>, Pair<String, Model.Data>> = HashMap()

        internal fun <T : Model.Data> add(key: String, clazz: Class<T>, data: Model.Data) {
            models[clazz] = Pair(key, data)
        }

        fun <T : Model.Data> get(clazz: Class<T>): T? {
            val model = models[clazz]
            if (model == null) {
                return model
            }
            return clazz.cast(model.second)
        }

        inline fun <reified T : Model.Data> get(): T? = get(T::class.java)

        internal fun getAll() = models.values.map { it.second }

        fun toMap() = models.values.associate {
            it.first to it.second.toMap()
        }

        class Serializer : StdSerializer<Data>(Data::class.java, false) {
            override fun serialize(
                value: Data?, gen: JsonGenerator?, provider: SerializerProvider?
            ) {
                gen!!.writeStartObject()
                value!!.models.toSortedMap { l, r -> l.name.compareTo(r.name) }.forEach {
                    gen.writeFieldName(it.value.first)
                    gen.writeObject(it.value.second)
                }
                gen.writeEndObject()
            }
        }

        class Deserializer(private vararg val models: Model<*>) :
            StdDeserializer<Data>(Data::class.java) {
            override fun deserialize(p: JsonParser?, ctxt: DeserializationContext?): Data {
                val data = Data()
                if (p != null && p.codec != null && ctxt != null) {
                    if (p.currentToken == JsonToken.FIELD_NAME) {
                        p.nextToken()
                    }
                    val node: JsonNode? = p.codec.readTree(p)
                    if (node != null) {
                        models.forEach {
                            val modelData = Utils.jsonMapper.readValue(
                                node.get(it.identifier.key).traverse(p.codec), it.dataClass
                            )
                            data.add(it.identifier.key, it.dataClass, modelData)
                        }
                    }
                    p.nextToken()
                }
                return data
            }
        }
    }

    private class ModelSet(vararg models: Model<*>) {
        private val keyedModels: HashMap<String, Model<*>> = HashMap()
        private val models: HashMap<Class<out Model<*>>, Model<*>> = HashMap()

        init {
            models.forEach {
                add(it)
            }
        }

        fun <T : Model<*>> get(clazz: Class<T>): T? {
            val model = models[clazz]
            if (model == null) {
                return model
            }
            return clazz.cast(model)
        }

        fun all() = models.values

        fun <T : Model<*>> add(model: T) {
            var existingModel = keyedModels[model.identifier.key]
            if (existingModel != null) {
                throw DuplicateModelException("Model ${existingModel::class.qualifiedName} already exists with key ${model.identifier.key}.")
            }
            existingModel = models[model::class.java]
            if (existingModel != null) {
                throw DuplicateModelException("Model ${existingModel::class.qualifiedName} already exists.  Cannot add duplicate model class.")
            }
            keyedModels[model.identifier.key] = model
            models[model::class.java] = model
        }

        fun <T : Model<*>> remove(clazz: Class<T>) {
            val model = models.remove(clazz)
            if (model != null) {
                keyedModels.remove(model.identifier.key)
            }
        }

        fun containsKey(key: String): Boolean = keyedModels.contains(key)

        fun <T : Model<*>> containsModel(clazz: Class<T>): Boolean = models.contains(clazz)
    }
}