package ai.cheq.sst.android.core.models

import ai.cheq.sst.android.core.BuildConfig
import ai.cheq.sst.android.core.models.Constants.UNKNOWN
import android.content.pm.PackageInfo
import android.content.pm.PackageManager
import android.os.Build
import com.fasterxml.jackson.annotation.JsonProperty

/**
 * Represents the application model that collects and exposes app-related data.
 */
class AppModel internal constructor() :
    Model<AppModel.Data>(Data::class, Constants.MODEL_KEY_APP, BuildConfig.LIBRARY_VERSION) {
    private lateinit var data: Data
    private lateinit var packageName: String
    private lateinit var packageManager: PackageManager

    override fun modelType(): ModelType = ModelType.REQUIRED

    override fun initialize(modelContext: ModelContext) {
        super.initialize(modelContext)
        packageName = modelContext.contextProvider.context().packageName
        packageManager = modelContext.contextProvider.context().packageManager
    }

    /**
     * Retrieves the data associated with this model.
     *
     * @param modelContext The context in which the data is being retrieved.
     * @return The data associated with the model.
     */
    override suspend fun get(modelContext: ModelContext): Data {
        if (!this::data.isInitialized) {
            data = Builder(packageName, packageManager).build()
        }
        return data
    }

    private class Builder(
        private val packageName: String, private val packageManager: PackageManager
    ) {
        private val name: String =
            packageManager.getApplicationLabel(packageInfo().applicationInfo).toString()

        private val version: String = packageInfo().versionName ?: UNKNOWN

        @Suppress("DEPRECATION")
        private val build: String = when (Build.VERSION.SDK_INT) {
            in Build.VERSION_CODES.P..Int.MAX_VALUE -> packageInfo().longVersionCode.toString()
            else -> packageInfo().versionCode.toString()
        }

        private val namespace: String = packageName

        private fun packageInfo(): PackageInfo = packageManager.getPackageInfo(packageName, 0)

        fun build() = Data(name, version, build, namespace)
    }

    /**
     * Data class representing the application model data.
     *
     * This class holds information about a model, including its name, version, build, and namespace.
     *
     * @property name The name of the app.
     * @property version The version of the app.
     * @property build The build identifier of the app.
     * @property namespace The namespace associated with the app.
     */
    class Data internal constructor(
        @field:JsonProperty val name: String,
        @field:JsonProperty val version: String,
        @field:JsonProperty val build: String,
        @field:JsonProperty val namespace: String
    ) : Model.Data()
}