package ai.cheq.sst.android.core

import ai.cheq.sst.android.core.exceptions.NotConfiguredException
import ai.cheq.sst.android.core.internal.EventBus
import ai.cheq.sst.android.core.internal.lateVar
import ai.cheq.sst.android.core.monitoring.Error
import ai.cheq.sst.android.protobuf.dataLayer.DataLayer
import android.content.Context
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.coroutines.future.future
import java.util.concurrent.CompletableFuture
import kotlin.coroutines.CoroutineContext
import kotlin.coroutines.EmptyCoroutineContext

/**
 * A data layer that stores key-value pairs to persistent storage.
 *
 * The data layer is a simple key-value store that can be used to store data that needs to
 * be included in every [Sst.trackEvent] call. The data layer is stored in the application's
 * data store, and can be accessed using the [add], [all], [clear], [contains], [get],
 * and [remove] methods.
 *
 * The data layer is implemented as a singleton, and is accessed using the [Sst.dataLayer] property.
 */
class DataLayer internal constructor() {
    private var contextProvider: ContextProvider by lateVar {
        NotConfiguredException("Sst.configure must be called before using the data layer")
    }
    private lateinit var log: Log
    private lateinit var eventBus: EventBus

    internal fun configure(contextProvider: ContextProvider, log: Log, eventBus: EventBus) {
        this.contextProvider = contextProvider
        this.log = log
        this.eventBus = eventBus
    }

    /**
     * Associates the specified [value] with the specified [key] in the data layer.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @param T The type of the value.
     * @param key The key to add.
     * @param value The value to add.
     * @param coroutineScope The coroutine scope to use for the operation.
     * @param coroutineContext The coroutine context to use for the operation.
     * @return A [CompletableFuture] that completes when the operation is done.
     *
     * @exception NotConfiguredException If [Sst.configure] has not been called.
     *
     * @suppress
     */
    @JvmOverloads
    fun <T : Any> add(
        key: String,
        value: T,
        coroutineScope: CoroutineScope,
        coroutineContext: CoroutineContext = EmptyCoroutineContext
    ): CompletableFuture<Void?> {
        val context = contextProvider.context()
        return coroutineScope.future(coroutineContext) {
            add(context, key, value)
            null
        }
    }

    /**
     * Associates the specified [value] with the specified [key] in the data layer.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @param T The type of the value.
     * @param key The key to add.
     * @param value The value to add.
     *
     * @exception NotConfiguredException If [Sst.configure] has not been called.
     *
     * @sample samples.core.DataLayer.Usage.add
     */
    @JvmSynthetic
    suspend fun <T : Any> add(key: String, value: T) {
        return add(contextProvider.context(), key, value)
    }

    /**
     * Returns a map of all key-value pairs in the data layer.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @param coroutineScope The [CoroutineScope] to use for the operation.
     * @param coroutineContext The [CoroutineContext] to use for the operation.
     * @return A [CompletableFuture] that completes when the operation is done.  If successful, the result is a map of all key-value pairs in the data layer.
     *
     * @exception NotConfiguredException If [Sst.configure] has not been called.
     *
     * @suppress
     */
    @Throws(NotConfiguredException::class)
    fun all(
        coroutineScope: CoroutineScope, coroutineContext: CoroutineContext = EmptyCoroutineContext
    ): CompletableFuture<Map<String, Any?>> {
        val context = contextProvider.context()
        return coroutineScope.future(coroutineContext) {
            all(context)
        }
    }

    /**
     * Returns a map of all key-value pairs in the data layer.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @return A map of all key-value pairs in the data layer.
     *
     * @exception NotConfiguredException If [Sst.configure] has not been called.
     *
     * @sample samples.core.DataLayer.Usage.all
     */
    @JvmSynthetic
    suspend fun all(): Map<String, Any?> {
        return all(contextProvider.context())
    }

    /**
     * Clears all key-value pairs from the data layer.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @param coroutineScope The [CoroutineScope] to use for the operation.
     * @param coroutineContext The [CoroutineContext] to use for the operation.
     * @return A [CompletableFuture] that completes when the operation is done.
     *
     * @exception NotConfiguredException If [Sst.configure] has not been called.
     *
     * @suppress
     */
    @JvmOverloads
    @Throws(NotConfiguredException::class)
    fun clear(
        coroutineScope: CoroutineScope, coroutineContext: CoroutineContext = EmptyCoroutineContext
    ): CompletableFuture<Void?> {
        val context = contextProvider.context()
        return coroutineScope.future(coroutineContext) {
            clear(context)
            null
        }
    }

    /**
     * Clears all key-value pairs from the data layer.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @exception NotConfiguredException If [Sst.configure] has not been called.
     *
     * @sample samples.core.DataLayer.Usage.clear
     */
    @JvmSynthetic
    suspend fun clear() {
        clear(contextProvider.context())
    }

    /**
     * Determines whether the data layer contains the specified key.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @param coroutineScope The [CoroutineScope] to use for the operation.
     * @param coroutineContext The [CoroutineContext] to use for the operation.
     * @param key The key to check.
     * @return A [CompletableFuture] that completes when the operation is done.  If successful, the result is `true` if the data layer contains the key, `false` otherwise.
     *
     * @exception NotConfiguredException If [Sst.configure] has not been called.
     *
     * @suppress
     */
    @JvmOverloads
    @Throws(NotConfiguredException::class)
    fun contains(
        coroutineScope: CoroutineScope,
        coroutineContext: CoroutineContext = EmptyCoroutineContext,
        key: String
    ): CompletableFuture<Boolean> {
        val context = contextProvider.context()
        return coroutineScope.future(coroutineContext) {
            contains(context, key)
        }
    }

    /**
     * Determines whether the data layer contains the specified key.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @param key The key to check.
     * @return `true` if the data layer contains the key, `false` otherwise.
     *
     * @exception NotConfiguredException If [Sst.configure] has not been called.
     *
     * @sample samples.core.DataLayer.Usage.contains
     */
    @JvmSynthetic
    suspend fun contains(key: String): Boolean {
        return contains(contextProvider.context(), key)
    }

    /**
     * Returns the value associated with the specified [key], or `null` if the key is not in the data layer.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @param T The type of the value.
     * @param key The key to get.
     * @return The value associated with the specified key, or `null` if the key is not in the data layer
     *
     * @exception NotConfiguredException If [Sst.configure] has not been called.
     *
     * @sample samples.core.DataLayer.Usage.get
     */
    @JvmSynthetic
    suspend inline fun <reified T : Any> get(key: String): T? {
        return get(key, T::class.javaObjectType)
    }

    /**
     * Returns the value associated with the specified [key], or `null` if the key is not in the data layer.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @param coroutineScope The [CoroutineScope] to use for the operation.
     * @param coroutineContext The [CoroutineContext] to use for the operation.
     * @param T The type of the value.
     * @param key The key to get.
     * @param valueClass The class of the value.
     * @return A [CompletableFuture] that completes when the operation is done.  If successful, the result is the value associated with the specified key, or `null` if the key is not in the data layer
     *
     * @exception NotConfiguredException If [Sst.configure] has not been called.
     *
     * @suppress
     */
    @JvmOverloads
    @Throws(NotConfiguredException::class)
    fun <T : Any> get(
        coroutineScope: CoroutineScope,
        coroutineContext: CoroutineContext = EmptyCoroutineContext,
        key: String,
        valueClass: Class<T>
    ): CompletableFuture<T?> {
        val context = contextProvider.context()
        return dataLayerMap(coroutineScope, coroutineContext, context).thenCompose<T?> {
            CompletableFuture.completedFuture(parseSerializedValue(key, it?.get(key), valueClass))
        }
    }

    /**
     * Removes the specified key and its corresponding value from the data layer.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @param coroutineScope The [CoroutineScope] to use for the operation.
     * @param coroutineContext The [CoroutineContext] to use for the operation.
     * @param key The key to remove.
     * @return `true` if the key was removed, `false` if the key was not in the data layer.
     *
     * @exception NotConfiguredException If [Sst.configure] has not been called.
     *
     * @suppress
     */
    @JvmOverloads
    @Throws(NotConfiguredException::class)
    fun remove(
        coroutineScope: CoroutineScope,
        coroutineContext: CoroutineContext = EmptyCoroutineContext,
        key: String
    ): CompletableFuture<Boolean> {
        val context = contextProvider.context()
        return contains(coroutineScope, coroutineContext, key).thenComposeAsync {
            if (!it) {
                CompletableFuture.completedFuture(false)
            } else {
                remove(coroutineScope, coroutineContext, context, key)
            }
        }
    }

    /**
     * Removes the specified key and its corresponding value from the data layer.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @param key The key to remove.
     * @return `true` if the key was removed, `false` if the key was not in the data layer.
     *
     * @exception NotConfiguredException If [Sst.configure] has not been called.
     *
     * @sample samples.core.DataLayer.Usage.remove
     */
    @JvmSynthetic
    suspend fun remove(key: String): Boolean {
        return remove(contextProvider.context(), key)
    }

    @JvmSynthetic
    @PublishedApi
    internal suspend fun <T : Any> get(key: String, valueClass: Class<T>): T? {
        val valueSerialized = dataLayerMap(contextProvider.context())?.get(key)
        return parseSerializedValue(key, valueSerialized, valueClass)
    }

    @JvmSynthetic
    internal suspend fun raw(): Map<String, String?> {
        return raw(contextProvider.context())
    }

    private suspend fun <T : Any> add(context: Context, key: String, value: T) {
        try {
            context.dataLayerDataStore.updateData {
                it.toBuilder().putDataLayer(key, Utils.jsonMapper.writeValueAsString(value)).build()
            }
        } catch (e: Exception) {
            eventBus.publish(
                Error(
                    "SerializationError", "Sst.dataLayer.add", "Key $key, details ${e.message}"
                )
            )
        }
    }

    private suspend fun all(context: Context): Map<String, Any?> {
        return raw(context).mapValues { entry ->
            try {
                Utils.jsonMapper.readValue(entry.value, Any::class.java)
            } catch (e: Exception) {
                if (log.isErrorLoggable()) {
                    log.e("CHEQ SST failed to get ${entry.key} from data layer: ${e.message}")
                }
                null
            }
        }
    }

    private suspend fun clear(context: Context) {
        context.dataLayerDataStore.updateData {
            it.toBuilder().clearDataLayer().build()
        }
    }

    private suspend fun contains(context: Context, key: String): Boolean {
        return dataLayerMap(context)?.containsKey(key) ?: false
    }

    private fun <T : Any> parseSerializedValue(
        key: String, valueSerialized: String?, valueClass: Class<T>
    ): T? {
        return valueSerialized?.let {
            try {
                Utils.jsonMapper.readValue(it, valueClass) ?: null
            } catch (e: Exception) {
                if (log.isErrorLoggable()) {
                    log.e("CHEQ SST failed to get $key from data layer: ${e.message}")
                }
                null
            }
        }
    }

    private fun remove(
        coroutineScope: CoroutineScope,
        coroutineContext: CoroutineContext,
        context: Context,
        key: String
    ): CompletableFuture<Boolean> {
        return coroutineScope.future(coroutineContext) {
            remove(context, key)
        }
    }

    private suspend fun remove(context: Context, key: String): Boolean {
        if (!contains(context, key)) {
            return false
        }
        context.dataLayerDataStore.updateData {
            it.toBuilder().removeDataLayer(key).build()
        }
        return true
    }

    private suspend fun raw(context: Context): Map<String, String?> {
        return dataLayerMap(context) ?: emptyMap()
    }

    private fun dataLayerMap(
        coroutineScope: CoroutineScope, coroutineContext: CoroutineContext, context: Context
    ): CompletableFuture<Map<String, String>?> {
        return coroutineScope.future(coroutineContext) {
            dataLayerMap(context)
        }
    }

    private suspend fun dataLayerMap(context: Context): Map<String, String>? {
        val dataLayer =
            context.dataLayerDataStore.data.firstOrNull() ?: DataLayer.getDefaultInstance()
        return dataLayer.dataLayerMap
    }
}