package ai.cheq.sst.android.core.internal

import ai.cheq.sst.android.core.monitoring.Error
import ai.cheq.sst.android.core.monitoring.Event
import ai.cheq.sst.android.core.monitoring.HttpRequest
import ai.cheq.sst.android.core.monitoring.HttpResponse
import kotlinx.coroutines.ensureActive
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.filterIsInstance
import kotlin.coroutines.coroutineContext
import kotlin.reflect.KClass

internal class EventBus {
    private val flow: MutableSharedFlow<in Event> = MutableSharedFlow()
    private val enableState = mutableMapOf<KClass<out Event>, Boolean>()

    init {
        Event::class.sealedSubclasses.forEach { enableState[it] = false }
    }

    inline fun <reified T : Event> enabled(): Boolean {
        return when (T::class) {
            HttpRequest::class -> enableState[HttpRequest::class]
            HttpResponse::class -> enableState[HttpResponse::class]
            Error::class -> enableState[Error::class]
            else -> null
        } ?: false
    }

    inline fun <reified T : Event> enable() {
        enableState[T::class] = true
    }

    inline fun <reified T : Event> disable() {
        enableState[T::class] = false
    }

    suspend inline fun <reified T : Event> publish(event: T) {
        if (enabled<T>()) {
            when (T::class) {
                HttpRequest::class -> flow.emit(event)
                HttpResponse::class -> flow.emit(event)
                Error::class -> flow.emit(event)
            }
        }
    }

    suspend inline fun <reified T : Event> subscribe(noinline action: suspend (value: T) -> Unit) {
        flow.filterIsInstance<T>().collectLatest {
            coroutineContext.ensureActive()
            action(it)
        }
    }
}