package ai.cheq.sst.android.core.settings

import ai.cheq.sst.android.core.ContextProvider
import ai.cheq.sst.android.protobuf.settings.Settings
import android.content.Context
import androidx.datastore.core.DataStore
import androidx.datastore.dataStore
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.coroutines.flow.map
import java.util.UUID

internal val Context.settingsDataStore: DataStore<Settings> by dataStore(
    fileName = "settings.pb", serializer = SettingsSerializer
)

internal object SettingsRepository {
    suspend fun readCheqUuid(contextProvider: ContextProvider): String? {
        return readUuid(contextProvider) { settings ->
            settings.cheqUuid ?: null
        }
    }

    suspend fun storeCheqUuid(contextProvider: ContextProvider, uuid: String?) {
        storeUuid(contextProvider, uuid) { l: Settings.Builder, r: String -> l.setCheqUuid(r) }
    }

    suspend fun clearCheqUuid(contextProvider: ContextProvider) {
        try {
            contextProvider.context().settingsDataStore.updateData {
                it.toBuilder().clearCheqUuid().build()
            }
        } catch (_: Exception) {
        }
    }

    suspend fun readDeviceUuid(contextProvider: ContextProvider): String? {
        return readUuid(contextProvider) { settings ->
            settings.deviceUuid ?: null
        }
    }

    suspend fun storeDeviceUuid(contextProvider: ContextProvider, uuid: String?) {
        storeUuid(contextProvider, uuid) { l: Settings.Builder, r: String -> l.setDeviceUuid(r) }
    }

    suspend fun readUuid(
        contextProvider: ContextProvider, accessor: suspend (value: Settings) -> String?
    ): String? {
        val uuid = read(contextProvider, accessor)
        if (uuid.isNullOrEmpty()) {
            return null
        }
        return try {
            UUID.fromString(uuid).toString()
        } catch (_: Exception) {
            null
        }
    }

    suspend fun storeUuid(
        contextProvider: ContextProvider,
        uuid: String?,
        setter: suspend (t: Settings.Builder, v: String) -> Settings.Builder
    ) {
        if (!uuid.isNullOrBlank()) {
            try {
                val validatedUuid = UUID.fromString(uuid).toString()
                store(contextProvider, validatedUuid, setter)
            } catch (_: Exception) {
            }
        }
    }

    suspend fun <R> read(
        contextProvider: ContextProvider, accessor: suspend (value: Settings) -> R
    ): R? {
        return contextProvider.context().settingsDataStore.data.map(accessor).firstOrNull()
    }

    suspend fun <R> store(
        contextProvider: ContextProvider,
        value: R,
        setter: suspend (settings: Settings.Builder, value: R) -> Settings.Builder
    ) {
        contextProvider.context().settingsDataStore.updateData {
            setter(it.toBuilder(), value).build()
        }
    }
}