package ai.cheq.sst.android.core

import ai.cheq.sst.android.core.Utils.Companion.truncate
import ai.cheq.sst.android.core.domain.Data
import ai.cheq.sst.android.core.exceptions.NotConfiguredException
import ai.cheq.sst.android.core.internal.EventBus
import ai.cheq.sst.android.core.internal.HttpClient
import ai.cheq.sst.android.core.internal.HttpResult
import ai.cheq.sst.android.core.internal.lateVar
import ai.cheq.sst.android.core.models.AppModel
import ai.cheq.sst.android.core.models.DeviceModel
import ai.cheq.sst.android.core.monitoring.Error
import ai.cheq.sst.android.core.monitoring.Event
import ai.cheq.sst.android.core.monitoring.EventHandler
import ai.cheq.sst.android.core.monitoring.HttpRequest
import ai.cheq.sst.android.core.monitoring.HttpResponse
import ai.cheq.sst.android.core.settings.SettingsRepository
import ai.cheq.sst.android.core.storage.Storage
import io.ktor.client.request.cookie
import io.ktor.client.request.header
import io.ktor.http.HttpHeaders
import io.ktor.http.HttpMethod
import io.ktor.http.URLBuilder
import io.ktor.http.URLProtocol
import io.ktor.http.setCookie
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.future.future
import kotlinx.coroutines.launch
import java.util.concurrent.CompletableFuture
import kotlin.coroutines.CoroutineContext
import kotlin.coroutines.EmptyCoroutineContext
import kotlin.reflect.KClass

private typealias SstEvent = ai.cheq.sst.android.core.Event

private const val CHEQ_UUID_COOKIE_NAME = "uuid"

/**
 * The main entry point for the CHEQ SST SDK.
 *
 * The `Sst` object is the main entry point for the CHEQ SST SDK. It provides methods for configuring
 * the SDK, tracking events, and accessing the data layer.
 *
 * The Event and included model information is encoded according to
 * the [SST Protocol](https://help.ensighten.com/hc/en-us/articles/22990062690577-SST-Protocol-Reference)
 * before being sent to SST.
 *
 * __Links__
 *
 * More information about parameters that can be passed to the CHEQ SST service, see [Getting Started with Data Collection Pixels](https://help.ensighten.com/hc/en-us/articles/22962356688273-Getting-Started-with-Data-Collection-Pixels)
 *
 * @sample samples.core.Sst.Usage.trackEvent
 * @sample samples.core.Sst.Usage.trackEventWithParameters
 */
object Sst {
    private var contextProvider: ContextProvider by lateVar()
    private var client: HttpClient by lateVar()
    private var config: Config by lateVar { NotConfiguredException("CHEQ SST not configured") }
    private lateinit var eventBus: EventBus

    /**
     * The [DataLayer] that stores key-value pairs to persistent storage for automatic inclusion in all [Sst.trackEvent] calls.
     *
     * **[Sst.configure] must be called once before methods on this object are called.**
     *
     * @sample samples.core.Sst.Usage.dataLayer
     */
    @JvmStatic
    val dataLayer = DataLayer()

    /**
     * [Storage] provides read/write access to cookies, local data, and session data for automatic inclusion in all [Sst.trackEvent] calls.
     *
     * **[Sst.configure] must be called once before methods on this object are called.**
     *
     * @sample samples.core.Sst.Usage.storageCookies
     * @sample samples.core.Sst.Usage.storageLocal
     * @sample samples.core.Sst.Usage.storageSession
     */
    @JvmStatic
    val storage = Storage()

    /**
     * Configures the CHEQ SST SDK with the specified [contextProvider] and [config].
     *
     * **This must be called before any other methods in the SDK are used.**
     *
     * @param config The [Config] object to use for the SDK.
     * @param contextProvider The Android [ContextProvider] to use for the SDK.
     *
     * @sample samples.core.Sst.Usage.configure
     * @sample samples.core.Sst.Usage.configureOverrideDefaults
     * @sample samples.core.Sst.Usage.configureWithCustomModel
     * @sample samples.core.Sst.Usage.configureWithoutDeviceModelAndOverriddenUserAgent
     */
    @JvmStatic
    fun configure(config: Config, contextProvider: ContextProvider) {
        this.contextProvider = contextProvider
        this.config = config
        configureEventBus(config)
        dataLayer.configure(this.contextProvider, config.log, eventBus)
        storage.configure(this.contextProvider, config.log, eventBus)
        client = HttpClient(config, eventBus)
        config.models.initialize(config, contextProvider)
        config.log.i("CHEQ SST configured")
    }

    /**
     * Returns the current configuration of the SDK.
     *
     * @return The [Config] object or `null` if the [configure] has not been called.
     */
    @JvmStatic
    fun config() = config

    /**
     * Registers a callback to be invoked when a [HttpRequest] is emitted.
     *
     * @param action The callback to be invoked when a [HttpRequest] is emitted.
     */
    @JvmStatic
    suspend fun onRequest(action: EventHandler<HttpRequest>) {
        eventBus.subscribe<HttpRequest>(action::handle)
    }

    /**
     * Registers a callback to be invoked when a [HttpResponse] is emitted.
     *
     * @param action The callback to be invoked when a [HttpResponse] is emitted.
     */
    @JvmStatic
    suspend fun onResponse(action: EventHandler<HttpResponse>) {
        eventBus.subscribe<HttpResponse>(action::handle)
    }

    /**
     * Registers a callback to be invoked when an [Error] is emitted.
     *
     * @param action The callback to be invoked when an [Error] is emitted.
     */
    @JvmStatic
    suspend fun onError(action: EventHandler<Error>) {
        eventBus.subscribe<Error>(action::handle)
    }

    /**
     * Sends the specified [event] to the CHEQ SST server, including data from all configured models
     * and the key-value pairs stored in the [dataLayer].
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @param coroutineScope The [CoroutineScope] to use for the operation.
     * @param coroutineContext The [CoroutineContext] to use for the operation.
     * @param event The [Event] to track.
     *
     * @throws NotConfiguredException If the SDK has not been configured.
     *
     * @suppress
     */
    @JvmStatic
    @JvmOverloads
    @Throws(NotConfiguredException::class)
    fun trackEvent(
        coroutineScope: CoroutineScope,
        coroutineContext: CoroutineContext = EmptyCoroutineContext,
        event: ai.cheq.sst.android.core.Event
    ): CompletableFuture<Void?> {
        return coroutineScope.future(coroutineContext) {
            trackEvent(event)
            return@future null
        }
    }

    /**
     * Sends the specified [event] to the CHEQ SST server, including data from all configured models
     * and the key-value pairs stored in the [dataLayer].
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @param event The [Event] to track.
     *
     * @throws NotConfiguredException If the SDK has not been configured.
     *
     * __Links__
     *
     * More information about parameters that can be passed to the CHEQ SST service, see [Getting Started with Data Collection Pixels](https://help.ensighten.com/hc/en-us/articles/22962356688273-Getting-Started-with-Data-Collection-Pixels)
     *
     * @sample samples.core.Sst.Usage.trackEvent
     * @sample samples.core.Sst.Usage.trackEventWithParameters
     */
    @JvmSynthetic
    suspend fun trackEvent(event: SstEvent) {
        val data = Data(
            config, event, config.models.collect(config, contextProvider), dataLayer.raw(), storage.raw()
        )
        val url = trackEventUrl()
        val cheqUuid = SettingsRepository.readCheqUuid(contextProvider)
        val userAgent = config.virtualBrowser.userAgent
            ?: data.dataLayer.mobileData?.get<DeviceModel.Data>(includeAll = true)?.userAgent
        val result = client.sendHttp(
            url, HttpMethod.Post, data = data, parameters = event.parameters, userAgent = userAgent
        ) {
            if (cheqUuid != null) {
                cookie(
                    CHEQ_UUID_COOKIE_NAME,
                    cheqUuid.toString(),
                    domain = config.domain,
                    secure = true,
                    httpOnly = true
                )
            }
        }
        when (result) {
            is HttpResult.Success -> {
                SettingsRepository.storeCheqUuid(
                    contextProvider,
                    result.response.setCookie()
                        .firstOrNull { it.name == CHEQ_UUID_COOKIE_NAME }?.value
                )
            }

            is HttpResult.Error.HttpError -> handleHttpResultError(result)
            is HttpResult.Error.NetworkError -> handleHttpResultError(result)
            is HttpResult.Error.SerializationError -> handleHttpResultError(result)
            is HttpResult.Error.UnknownError -> handleHttpResultError(result)
        }
    }

    /**
     * Gets the CHEQ UUID (universally unique identifier) from persistent storage.
     * The CHEQ UUID is automatically set by the SDK when an [Sst.trackEvent] call is made.
     *
     * The CHEQ UUID is used to group together events from the same device and application across multiple sessions.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @param coroutineScope The [CoroutineScope] to use for the operation.
     * @param coroutineContext The [CoroutineContext] to use for the operation.
     *
     * @return The CHEQ UUID or `null` if it does not exist.
     *
     * @throws NotConfiguredException If the SDK has not been configured.
     *
     * @suppress
     */
    @JvmStatic
    @JvmOverloads
    @Throws(NotConfiguredException::class)
    fun getCheqUuid(
        coroutineScope: CoroutineScope, coroutineContext: CoroutineContext = EmptyCoroutineContext
    ): CompletableFuture<String?> {
        val ctxProvider = contextProvider
        return coroutineScope.future(coroutineContext) {
            getCheqUuid(ctxProvider)
        }
    }

    /**
     * Gets the CHEQ UUID (universally unique identifier) from persistent storage.
     * The CHEQ UUID is automatically set by the SDK when an [Sst.trackEvent] call is made.
     *
     * The CHEQ UUID is used to group together events from the same device and application across multiple sessions.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @return The CHEQ UUID or `null` if it does not exist.
     *
     * @throws NotConfiguredException If the SDK has not been configured.
     *
     * @sample samples.core.Sst.Usage.getCheqUuid
     */
    @JvmSynthetic
    suspend fun getCheqUuid(): String? {
        return getCheqUuid(contextProvider)
    }

    /**
     * Clears the CHEQ UUID (universally unique identifier) from persistent storage.
     * The CHEQ UUID is automatically set by the SDK when an [Sst.trackEvent] call is made.
     *
     * The CHEQ UUID is used to group together events from the same device and application across multiple sessions.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @param coroutineScope The [CoroutineScope] to use for the operation
     * @param coroutineContext The [CoroutineContext] to use for the operation
     *
     * @return A [CompletableFuture] that completes when the operation is done.
     *
     * @throws NotConfiguredException If the SDK has not been configured.
     *
     * @suppress
     */
    @JvmStatic
    @JvmOverloads
    @Throws(NotConfiguredException::class)
    fun clearCheqUuid(
        coroutineScope: CoroutineScope, coroutineContext: CoroutineContext = EmptyCoroutineContext
    ): CompletableFuture<Void?> {
        val ctxProvider = contextProvider
        return coroutineScope.future(coroutineContext) {
            clearCheqUuid(ctxProvider)
            null
        }
    }

    /**
     * Clears the CHEQ UUID (universally unique identifier) from persistent storage.
     * The CHEQ UUID is automatically set by the SDK when an [Sst.trackEvent] call is made.
     *
     * The CHEQ UUID is used to group together events from the same device and application across multiple sessions.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @throws NotConfiguredException If the SDK has not been configured.
     *
     * @sample samples.core.Sst.Usage.clearCheqUuid
     */
    @JvmSynthetic
    suspend fun clearCheqUuid() {
        clearCheqUuid(contextProvider)
    }

    /**
     * Gets the mobile data collected by the SDK.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @return A map of the mobile data collected by the SDK.
     *
     * @throws NotConfiguredException If the SDK has not been configured.
     *
     * @sample samples.core.Sst.Usage.getMobileData
     */
    @JvmSynthetic
    suspend fun getMobileData(): Map<String, Any> {
        return getMobileData(contextProvider)
    }

    /**
     * Gets the mobile data collected by the SDK.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @param coroutineScope The [CoroutineScope] to use for the operation.
     * @param coroutineContext The [CoroutineContext] to use for the operation.
     *
     * @return A [CompletableFuture] that completes when the operation is done.  If successful, the result is a map of the mobile data collected by the SDK.
     *
     * @throws NotConfiguredException If the SDK has not been configured.
     *
     * @suppress
     */
    @JvmStatic
    @JvmOverloads
    @Throws(NotConfiguredException::class)
    fun getMobileData(
        coroutineScope: CoroutineScope, coroutineContext: CoroutineContext = EmptyCoroutineContext
    ): CompletableFuture<Map<String, Any>> {
        val ctxProvider = contextProvider
        return coroutineScope.future(coroutineContext) {
            getMobileData(ctxProvider)
        }
    }

    internal suspend fun storeCheqUuid(cheqUuid: String) {
        SettingsRepository.storeCheqUuid(contextProvider, cheqUuid)
    }

    private suspend fun getMobileData(contextProvider: ContextProvider): Map<String, Any> {
        return config.models.collect(config, contextProvider).toMap()
    }

    private fun trackEventUrl(): URLBuilder {
        val url = URLBuilder(
            URLProtocol.HTTPS,
            config.domain,
            pathSegments = listOf("pc", config.clientName, "sst"),
        )
        url.parameters.append("sstVersion", "1.0.0")
        url.parameters.append("sstOrigin", "mobile")
        return url
    }

    private suspend fun handleError(name: String, function: String, message: String) {
        var sourceFile = "$function ${BuildConfig.LIBRARY_NAME}:${BuildConfig.LIBRARY_VERSION}"
        try {
            val appModel = config.models.collect<AppModel, AppModel.Data>(config, contextProvider)
            if (appModel != null) {
                sourceFile += " ${appModel.name}:${appModel.version}"
            }
        } catch (_: Exception) {
        }

        val url = URLBuilder(
            URLProtocol.HTTPS, config.nexusHost, pathSegments = listOf("error", "e.gif")
        )
        url.parameters.append("msg", message.truncate(1024))
        url.parameters.append("fn", sourceFile.truncate(256))
        url.parameters.append("client", config.clientName.truncate(256))
        url.parameters.append("publishPath", config.publishPath.truncate(256))
        url.parameters.append("errorName", name.truncate(256))
        url.parameters
        var userAgent: String? = null
        try {
            userAgent = config.virtualBrowser.userAgent
                ?: config.models.collect<DeviceModel, DeviceModel.Data>(
                    config,
                    contextProvider
                )?.userAgent
        } catch (_: Exception) {
        }
        client.sendHttp<Any>(url, HttpMethod.Get, userAgent = userAgent) {
            header(HttpHeaders.Referrer, trackEventUrl().buildString())
        }
    }

    private suspend fun handleHttpResultError(result: HttpResult.Error.HttpError) {
        handleHttpResultError(
            result::class, "${result.response.status.description}(${result.response.status.value}"
        )
    }

    private suspend fun handleHttpResultError(result: HttpResult.Error.NetworkError) {
        handleHttpResultError(result::class, result.throwable.message ?: "Unknown")
    }

    private suspend fun handleHttpResultError(result: HttpResult.Error.SerializationError) {
        handleHttpResultError(result::class, result.throwable.message ?: "Unknown")
    }

    private suspend fun handleHttpResultError(result: HttpResult.Error.UnknownError) {
        handleHttpResultError(result::class, result.throwable.message ?: "Unknown")
    }

    private suspend fun <T : HttpResult.Error> handleHttpResultError(
        errorClass: KClass<T>, message: String
    ) {
        val errorName = errorClass.simpleName ?: "Error"
        if (config.log.isErrorLoggable()) {
            config.log.e("CHEQ SST TrackEvent $errorName: $message")
        }
        eventBus.publish(Error(errorName, "Sst.trackEvent", message))
    }

    private fun configureEventBus(config: Config) {
        if (!Sst::eventBus.isInitialized) {
            eventBus = EventBus()
            config.monitoringScope.launch(config.monitoringDispatcher) {
                eventBus.subscribe<Error> {
                    handleError(it.name, it.function, it.message)
                }
            }
        }
        if (config.debug) {
            eventBus.enable<HttpRequest>()
            eventBus.enable<HttpResponse>()
        } else {
            eventBus.disable<HttpRequest>()
            eventBus.disable<HttpResponse>()
        }
        eventBus.enable<Error>()
    }

    private suspend fun clearCheqUuid(contextProvider: ContextProvider) {
        SettingsRepository.clearCheqUuid(contextProvider)
    }

    private suspend fun getCheqUuid(contextProvider: ContextProvider): String? {
        return SettingsRepository.readCheqUuid(contextProvider)
    }
}
