package ai.cheq.sst.android.core.internal

import ai.cheq.sst.android.core.Constants.TAG
import android.util.Log
import org.apache.commons.text.StringEscapeUtils

internal class AndroidLog internal constructor() : ai.cheq.sst.android.core.Log {
    /**
     * Check if debug logging is enabled.
     */
    override fun isDebugLoggable(): Boolean {
        return Log.isLoggable(TAG, Log.DEBUG)
    }

    /**
     * Send a debug log message.
     *
     * @param message The message you would like logged.
     */
    override fun d(message: String) {
        if (isDebugLoggable()) {
            Log.d(TAG, sanitizeMessage(message))
        }
    }

    /**
     * Check if info logging is enabled.
     */
    override fun isInfoLoggable(): Boolean {
        return Log.isLoggable(TAG, Log.INFO)
    }

    /**
     * Send a information log message.
     *
     * @param message The message you would like logged.
     */
    override fun i(message: String) {
        if (isInfoLoggable()) {
            Log.i(TAG, sanitizeMessage(message))
        }
    }

    /**
     * Check if warn logging is enabled.
     */
    override fun isWarnLoggable(): Boolean {
        return Log.isLoggable(TAG, Log.WARN)
    }

    /**
     * Send a warning log message.
     *
     * @param message The message you would like logged.
     */
    override fun w(message: String) {
        if (isWarnLoggable()) {
            Log.w(TAG, sanitizeMessage(message))
        }
    }

    /**
     * Check if error logging is enabled.
     */
    override fun isErrorLoggable(): Boolean {
        return Log.isLoggable(TAG, Log.INFO)
    }

    /**
     * Send an error log message.
     *
     * @param message The message you would like logged.
     */
    override fun e(message: String) {
        if (isErrorLoggable()) {
            Log.e(TAG, sanitizeMessage(message))
        }
    }

    /**
     * Send an error log message and log the exception.

     * @param message The message you would like logged.
     * @param throwable An exception to log.
     */
    override fun e(message: String, throwable: Throwable) {
        if (isErrorLoggable()) {
            Log.e(
                TAG, sanitizeMessage(message), throwable
            )
        }
    }

    private fun sanitizeMessage(message: String): String {
        return StringEscapeUtils.escapeJava(message)
    }
}