package ai.cheq.sst.android.core.internal

import ai.cheq.sst.android.core.ContextProvider
import ai.cheq.sst.android.core.Log
import ai.cheq.sst.android.core.Utils
import ai.cheq.sst.android.core.exceptions.NotConfiguredException
import ai.cheq.sst.android.core.monitoring.Error
import android.content.Context
import androidx.datastore.core.DataStore
import com.google.protobuf.GeneratedMessageLite
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.coroutines.future.future
import java.util.concurrent.CompletableFuture
import kotlin.coroutines.CoroutineContext
import kotlin.coroutines.EmptyCoroutineContext

internal class DataStoreCollection<T : GeneratedMessageLite<S, B>, S: T, B: GeneratedMessageLite.Builder<S, B>> internal constructor(
    private val config: Config<T, S, B>,
    private val contextProvider: ContextProvider,
    private val log: Log,
    private val eventBus: EventBus
) {
    internal class Config<T : GeneratedMessageLite<S, B>, S: T, B: GeneratedMessageLite.Builder<S, B>>(
        val type: String,
        val dataStoreFunction: (Context) -> DataStore<S>,
        val putFunction: (B, String, String) -> B,
        val clearFunction: (B) -> B,
        val removeFunction: (B, String) -> B,
        val mapFunction: (S) -> MutableMap<String, String>?,
        val defaultFunction: () -> S
    )

    @JvmOverloads
    @Throws(NotConfiguredException::class)
    fun <T : Any> add(
        coroutineScope: CoroutineScope,
        coroutineContext: CoroutineContext = EmptyCoroutineContext,
        key: String,
        value: T
    ): CompletableFuture<Void?> {
        val context = contextProvider.context()
        return coroutineScope.future(coroutineContext) {
            add(context, key, value)
            null
        }
    }

    @JvmSynthetic
    suspend fun <T : Any> add(key: String, value: T) {
        return add(contextProvider.context(), key, value)
    }

    @Throws(NotConfiguredException::class)
    fun all(
        coroutineScope: CoroutineScope, coroutineContext: CoroutineContext = EmptyCoroutineContext
    ): CompletableFuture<Map<String, Any?>> {
        val context = contextProvider.context()
        return coroutineScope.future(coroutineContext) {
            all(context)
        }
    }

    @JvmSynthetic
    suspend fun all(): Map<String, Any?> {
        return all(contextProvider.context())
    }

    @JvmOverloads
    @Throws(NotConfiguredException::class)
    fun clear(
        coroutineScope: CoroutineScope, coroutineContext: CoroutineContext = EmptyCoroutineContext
    ): CompletableFuture<Void?> {
        val context = contextProvider.context()
        return coroutineScope.future(coroutineContext) {
            clear(context)
            null
        }
    }

    @JvmSynthetic
    suspend fun clear() {
        clear(contextProvider.context())
    }

    @JvmOverloads
    @Throws(NotConfiguredException::class)
    fun contains(
        coroutineScope: CoroutineScope,
        coroutineContext: CoroutineContext = EmptyCoroutineContext,
        key: String
    ): CompletableFuture<Boolean> {
        val context = contextProvider.context()
        return coroutineScope.future(coroutineContext) {
            contains(context, key)
        }
    }

    @JvmSynthetic
    suspend fun contains(key: String): Boolean {
        return contains(contextProvider.context(), key)
    }

    @JvmOverloads
    @Throws(NotConfiguredException::class)
    fun <T : Any> get(
        coroutineScope: CoroutineScope,
        coroutineContext: CoroutineContext = EmptyCoroutineContext,
        key: String,
        valueClass: Class<T>
    ): CompletableFuture<T?> {
        val context = contextProvider.context()
        return dataMap(coroutineScope, coroutineContext, context).thenCompose<T?> {
            CompletableFuture.completedFuture(deserialize(key, it?.get(key), valueClass))
        }
    }

    @JvmOverloads
    @Throws(NotConfiguredException::class)
    fun remove(
        coroutineScope: CoroutineScope,
        coroutineContext: CoroutineContext = EmptyCoroutineContext,
        key: String
    ): CompletableFuture<Boolean> {
        val context = contextProvider.context()
        return contains(coroutineScope, coroutineContext, key).thenComposeAsync {
            if (!it) {
                CompletableFuture.completedFuture(false)
            } else {
                remove(coroutineScope, coroutineContext, context, key)
            }
        }
    }

    @JvmSynthetic
    suspend fun remove(key: String): Boolean {
        return remove(contextProvider.context(), key)
    }

    @JvmSynthetic
    @PublishedApi
    internal suspend fun <T : Any> get(key: String, valueClass: Class<T>): T? {
        val valueSerialized = dataMap(contextProvider.context())?.get(key)
        return deserialize(key, valueSerialized, valueClass)
    }

    @JvmSynthetic
    internal suspend fun raw(): Map<String, String?> {
        return raw(contextProvider.context())
    }

    private suspend fun <T : Any> add(context: Context, key: String, value: T) {
        try {
            config.dataStoreFunction(context).updateData {
                config.putFunction(it.toBuilder(), key, serialize(value)).build()
            }
        } catch (e: Exception) {
            eventBus.publish(
                Error(
                    "SerializationError", "Sst.${config.type}.add", "Key $key, details ${e.message}"
                )
            )
        }
    }

    private suspend fun all(context: Context): Map<String, Any?> {
        return raw(context).mapValues { entry ->
            try {
                deserialize(entry.key, entry.value, Any::class.java)
            } catch (e: Exception) {
                if (log.isErrorLoggable()) {
                    log.e("CHEQ SST failed to get ${entry.key} from ${config.type}: ${e.message}")
                }
                null
            }
        }
    }

    private suspend fun clear(context: Context) {
        config.dataStoreFunction(context).updateData {
            config.clearFunction(it.toBuilder()).build()
        }
    }

    private suspend fun contains(context: Context, key: String): Boolean {
        return dataMap(context)?.containsKey(key) ?: false
    }

    private fun remove(
        coroutineScope: CoroutineScope,
        coroutineContext: CoroutineContext,
        context: Context,
        key: String
    ): CompletableFuture<Boolean> {
        return coroutineScope.future(coroutineContext) {
            remove(context, key)
        }
    }

    private suspend fun remove(context: Context, key: String): Boolean {
        if (!contains(context, key)) {
            return false
        }
        config.dataStoreFunction(context).updateData {
            config.removeFunction(it.toBuilder(), key).build()
        }
        return true
    }

    private suspend fun raw(context: Context): Map<String, String?> {
        return dataMap(context) ?: emptyMap()
    }

    private fun dataMap(
        coroutineScope: CoroutineScope, coroutineContext: CoroutineContext, context: Context
    ): CompletableFuture<Map<String, String>?> {
        return coroutineScope.future(coroutineContext) {
            dataMap(context)
        }
    }

    private suspend fun dataMap(context: Context): Map<String, String>? {
        val dataLayer = config.dataStoreFunction(context).data.firstOrNull() ?: config.defaultFunction()
        return config.mapFunction(dataLayer)?.toMap()
    }

    private fun serialize(value: Any): String = Utils.jsonMapper.writeValueAsString(value)

    private fun <T : Any> deserialize(key: String, valueSerialized: String?, valueClass: Class<T>): T? {
        return valueSerialized?.let {
            try {
                Utils.jsonMapper.readValue(it, valueClass) ?: null
            } catch (e: Exception) {
                if (log.isErrorLoggable()) {
                    log.e("CHEQ SST failed to get $key from ${config.type}: ${e.message}")
                }
                null
            }
        }
    }
}