package ai.cheq.sst.android.core.internal

import java.util.concurrent.atomic.AtomicReference
import kotlin.properties.ReadWriteProperty
import kotlin.reflect.KProperty

internal fun <T : Any> lateVar(uninitializedMessage: String? = null): LateVar<T> =
    LateVarImpl(uninitializedMessage)

internal fun <T : Any> lateVar(error: () -> Throwable): LateVar<T> = LateVarImpl(error)

internal interface LateVar<T> : ReadWriteProperty<Any?, T> {
    val isInitialized: Boolean
    val value: T
}

private class LateVarImpl<T : Any> private constructor(private val error: (String) -> Throwable) :
    LateVar<T> {
    private var name: String? = null
    private var valueRef = AtomicReference<T>()

    constructor(uninitializedMessage: String?) : this({ defaultMessage ->
        IllegalStateException(
            uninitializedMessage ?: defaultMessage
        )
    })

    constructor(error: () -> Throwable) : this({ _: String -> error() })

    override val value: T
        get() {
            return valueRef.get() ?: throw error("$name has not been initialized")
        }

    override fun getValue(thisRef: Any?, property: KProperty<*>): T {
        if (name == null) {
            name = property.name
        }
        return value
    }

    override fun setValue(thisRef: Any?, property: KProperty<*>, value: T) {
        if (valueRef.compareAndSet(null, value)) {
            this.name = property.name
        } else {
            valueRef.set(value)
        }
    }

    override val isInitialized: Boolean
        get() = valueRef.get() != null

    override fun toString(): String {
        val name = name ?: return "LateVal(not initialized)"
        val value = valueRef.get()
        return "LateVal(${if (value != null) "$name=$value" else "$name not initialized yet"})"
    }
}
