package ai.cheq.sst.android.core.storage

import ai.cheq.sst.android.core.exceptions.NotConfiguredException
import kotlinx.coroutines.CoroutineScope
import java.util.concurrent.CompletableFuture
import kotlin.coroutines.CoroutineContext
import kotlin.coroutines.EmptyCoroutineContext

/**
 * A collection of [Cookie] objects that can be stored and retrieved from persistent storage.
 */
class Cookies internal constructor(private val layer: DataStoreStorageCollection<Cookie>) : StorageCollection<Cookie> by layer {
    /**
     * Associates the specified [value] to the storage layer.
     *
     * **[ai.cheq.sst.android.core.Sst.configure] must be called once before this method is called.**
     *
     * @param coroutineScope The coroutine scope to use for the operation.
     * @param coroutineContext The coroutine context to use for the operation.
     * @param name The name of the cookie.
     * @param value The value of the cookie.
     * @return A [CompletableFuture] that completes when the operation is done.
     *
     * @exception NotConfiguredException If [ai.cheq.sst.android.core.Sst.configure] has not been called.
     *
     * @suppress
     */
    @JvmOverloads
    @Throws(NotConfiguredException::class)
    fun add(
        coroutineScope: CoroutineScope,
        coroutineContext: CoroutineContext = EmptyCoroutineContext,
        name: String,
        value: String
    ): CompletableFuture<Void?> {
        return add(coroutineScope, coroutineContext, Cookie(name, value))
    }

    @Throws(NotConfiguredException::class)
    override fun add(coroutineScope: CoroutineScope, value: Cookie): CompletableFuture<Void?> {
        return layer.add(coroutineScope, value)
    }

    @Throws(NotConfiguredException::class)
    override fun add(
        coroutineScope: CoroutineScope, coroutineContext: CoroutineContext, value: Cookie
    ): CompletableFuture<Void?> {
        return layer.add(coroutineScope, coroutineContext, value)
    }

    /**
     * Associates the specified [value] to the storage layer.
     *
     * **[ai.cheq.sst.android.core.Sst.configure] must be called once before this method is called.**
     *
     * @param name The name of the cookie.
     * @param value The value of the cookie.
     *
     * @exception NotConfiguredException If [ai.cheq.sst.android.core.Sst.configure] has not been called.
     *
     * @sample samples.core.StorageLayer.Usage.add
     */
    @JvmSynthetic
    suspend fun add(name: String, value: String) {
        return add(Cookie(name, value))
    }

    @Throws(NotConfiguredException::class)
    override fun all(coroutineScope: CoroutineScope): CompletableFuture<Map<String, Any?>> {
        return layer.all(coroutineScope)
    }

    @Throws(NotConfiguredException::class)
    override fun all(
        coroutineScope: CoroutineScope, coroutineContext: CoroutineContext
    ): CompletableFuture<Map<String, Any?>> {
        return layer.all(coroutineScope, coroutineContext)
    }

    @Throws(NotConfiguredException::class)
    override fun clear(coroutineScope: CoroutineScope): CompletableFuture<Void?> {
        return layer.clear(coroutineScope)
    }

    @Throws(NotConfiguredException::class)
    override fun clear(
        coroutineScope: CoroutineScope, coroutineContext: CoroutineContext
    ): CompletableFuture<Void?> {
        return layer.clear(coroutineScope, coroutineContext)
    }

    @Throws(NotConfiguredException::class)
    override fun contains(coroutineScope: CoroutineScope, key: String): CompletableFuture<Boolean> {
        return layer.contains(coroutineScope, key)
    }

    @Throws(NotConfiguredException::class)
    override fun contains(
        coroutineScope: CoroutineScope, coroutineContext: CoroutineContext, key: String
    ): CompletableFuture<Boolean> {
        return layer.contains(coroutineScope, coroutineContext, key)
    }

    @Throws(NotConfiguredException::class)
    override fun get(coroutineScope: CoroutineScope, key: String): CompletableFuture<Cookie?> {
        return layer.get(coroutineScope, key)
    }

    @Throws(NotConfiguredException::class)
    override fun get(
        coroutineScope: CoroutineScope, coroutineContext: CoroutineContext, key: String
    ): CompletableFuture<Cookie?> {
        return layer.get(coroutineScope, coroutineContext, key)
    }

    @Throws(NotConfiguredException::class)
    override fun remove(coroutineScope: CoroutineScope, key: String): CompletableFuture<Boolean> {
        return layer.remove(coroutineScope, key)
    }

    @Throws(NotConfiguredException::class)
    override fun remove(
        coroutineScope: CoroutineScope, coroutineContext: CoroutineContext, key: String
    ): CompletableFuture<Boolean> {
        return layer.remove(coroutineScope, coroutineContext, key)
    }

    @JvmSynthetic
    internal suspend fun raw(): List<String> {
        return layer.raw()
    }
}