package ai.cheq.sst.android.core.storage

import ai.cheq.sst.android.core.ContextProvider
import ai.cheq.sst.android.core.Log
import ai.cheq.sst.android.core.exceptions.NotConfiguredException
import ai.cheq.sst.android.core.internal.DataStoreCollection
import ai.cheq.sst.android.core.internal.EventBus
import ai.cheq.sst.android.protobuf.storage.Storage
import kotlinx.coroutines.CoroutineScope
import java.util.concurrent.CompletableFuture
import kotlin.coroutines.CoroutineContext
import kotlin.coroutines.EmptyCoroutineContext
import kotlin.reflect.KClass

internal class DataStoreStorageCollection<T : Identifiable> private constructor(
    private val type : KClass<T>,
    layerConfig: DataStoreCollection.Config<Storage, Storage, Storage.Builder>,
    contextProvider: ContextProvider,
    log: Log,
    eventBus: EventBus
) : StorageCollection<T> {
    private var dataStoreCollection: DataStoreCollection<Storage, Storage, Storage.Builder> = DataStoreCollection(
        layerConfig, contextProvider, log, eventBus
    )

    @Throws(NotConfiguredException::class)
    override fun add(coroutineScope: CoroutineScope, value: T): CompletableFuture<Void?> {
        return add(coroutineScope, EmptyCoroutineContext, value)
    }

    @Throws(NotConfiguredException::class)
    override fun add(
        coroutineScope: CoroutineScope,
        coroutineContext: CoroutineContext,
        value: T
    ): CompletableFuture<Void?> {
        return dataStoreCollection.add(coroutineScope, coroutineContext, value.id(), value)
    }

    @JvmSynthetic
    override suspend fun add(value: T) {
        return dataStoreCollection.add(value.id(), value)
    }

    @Throws(NotConfiguredException::class)
    override fun all(coroutineScope: CoroutineScope): CompletableFuture<Map<String, Any?>> {
        return all(coroutineScope, EmptyCoroutineContext)
    }

    @Throws(NotConfiguredException::class)
    override fun all(
        coroutineScope: CoroutineScope,
        coroutineContext: CoroutineContext
    ): CompletableFuture<Map<String, Any?>> {
        return dataStoreCollection.all(coroutineScope, coroutineContext)
    }

    @JvmSynthetic
    override suspend fun all(): Map<String, Any?> {
        return dataStoreCollection.all()
    }

    override fun clear(coroutineScope: CoroutineScope): CompletableFuture<Void?> {
        return clear(coroutineScope, EmptyCoroutineContext)
    }

    @Throws(NotConfiguredException::class)
    override fun clear(
        coroutineScope: CoroutineScope,
        coroutineContext: CoroutineContext
    ): CompletableFuture<Void?> {
        return dataStoreCollection.clear(coroutineScope, coroutineContext)
    }

    @JvmSynthetic
    override suspend fun clear() {
        dataStoreCollection.clear()
    }

    @Throws(NotConfiguredException::class)
    override fun contains(coroutineScope: CoroutineScope, key: String): CompletableFuture<Boolean> {
        return contains(coroutineScope, EmptyCoroutineContext, key)
    }

    @Throws(NotConfiguredException::class)
    override fun contains(
        coroutineScope: CoroutineScope,
        coroutineContext: CoroutineContext,
        key: String
    ): CompletableFuture<Boolean> {
        return dataStoreCollection.contains(coroutineScope, coroutineContext, key)
    }

    @JvmSynthetic
    override suspend fun contains(key: String): Boolean {
        return dataStoreCollection.contains(key)
    }

    override suspend fun get(key: String): T? {
        return dataStoreCollection.get(key, type.javaObjectType)
    }

    @Throws(NotConfiguredException::class)
    override fun get(coroutineScope: CoroutineScope, key: String): CompletableFuture<T?> {
        return get(coroutineScope, EmptyCoroutineContext, key)
    }

    @Throws(NotConfiguredException::class)
    override fun get(
        coroutineScope: CoroutineScope,
        coroutineContext: CoroutineContext,
        key: String
    ): CompletableFuture<T?> {
        return dataStoreCollection.get(coroutineScope, coroutineContext, key, type.javaObjectType)
    }

    @Throws(NotConfiguredException::class)
    override fun remove(coroutineScope: CoroutineScope, key: String): CompletableFuture<Boolean> {
        return remove(coroutineScope, EmptyCoroutineContext, key)
    }

    @Throws(NotConfiguredException::class)
    override fun remove(
        coroutineScope: CoroutineScope,
        coroutineContext: CoroutineContext,
        key: String
    ): CompletableFuture<Boolean> {
        return dataStoreCollection.remove(coroutineScope, coroutineContext, key)
    }

    @JvmSynthetic
    override suspend fun remove(key: String): Boolean {
        return dataStoreCollection.remove(key)
    }

    @JvmSynthetic
    internal suspend fun raw(): List<String> {
        return dataStoreCollection.raw().values.filterNotNull()
    }

    companion object{
        internal inline fun <reified T: Identifiable> create(layerConfig: DataStoreCollection.Config<Storage, Storage, Storage.Builder>,
                                                             contextProvider: ContextProvider,
                                                             log: Log,
                                                             eventBus: EventBus
        ) = DataStoreStorageCollection(T::class, layerConfig, contextProvider, log, eventBus)
    }
}