package ai.cheq.sst.android.core.storage

import ai.cheq.sst.android.core.exceptions.NotConfiguredException
import kotlinx.coroutines.CoroutineScope
import java.util.concurrent.CompletableFuture
import kotlin.coroutines.CoroutineContext

/**
 * A generic collection of values that can be stored and retrieved from persistent storage.
 *
 * @param T the type of elements contained in the collection.
 */
interface StorageCollection<T : Identifiable> {
    /**
     * Associates the specified [value] to the storage layer.
     *
     * **[ai.cheq.sst.android.core.Sst.configure] must be called once before this method is called.**
     *
     * @param coroutineScope The coroutine scope to use for the operation.
     * @param value The value to add.
     * @return A [CompletableFuture] that completes when the operation is done.
     *
     * @exception NotConfiguredException If [ai.cheq.sst.android.core.Sst.configure] has not been called.
     *
     * @suppress
     */
    fun add(
        coroutineScope: CoroutineScope,
        value: T
    ): CompletableFuture<Void?>

    /**
     * Associates the specified [value] to the storage layer.
     *
     * **[ai.cheq.sst.android.core.Sst.configure] must be called once before this method is called.**
     *
     * @param coroutineScope The coroutine scope to use for the operation.
     * @param coroutineContext The coroutine context to use for the operation.
     * @param value The value to add.
     * @return A [CompletableFuture] that completes when the operation is done.
     *
     * @exception NotConfiguredException If [ai.cheq.sst.android.core.Sst.configure] has not been called.
     *
     * @suppress
     */
    fun add(
        coroutineScope: CoroutineScope,
        coroutineContext: CoroutineContext,
        value: T
    ): CompletableFuture<Void?>

    /**
     * Associates the specified [value] to the storage layer.
     *
     * **[ai.cheq.sst.android.core.Sst.configure] must be called once before this method is called.**
     *
     * @param value The value to add.
     *
     * @exception NotConfiguredException If [ai.cheq.sst.android.core.Sst.configure] has not been called.
     *
     * @sample samples.core.StorageLayer.Usage.add
     */
    @JvmSynthetic
    suspend fun add(value: T)

    /**
     * Returns a map of all key-value pairs in the storage layer.
     *
     * **[ai.cheq.sst.android.core.Sst.configure] must be called once before this method is called.**
     *
     * @param coroutineScope The [CoroutineScope] to use for the operation.
     * @return A [CompletableFuture] that completes when the operation is done.  If successful, the result is a map of all key-value pairs in the storage layer.
     *
     * @exception NotConfiguredException If [ai.cheq.sst.android.core.Sst.configure] has not been called.
     *
     * @suppress
     */
    @Throws(NotConfiguredException::class)
    fun all(
        coroutineScope: CoroutineScope
    ): CompletableFuture<Map<String, Any?>>

    /**
     * Returns a map of all key-value pairs in the storage layer.
     *
     * **[ai.cheq.sst.android.core.Sst.configure] must be called once before this method is called.**
     *
     * @param coroutineScope The [CoroutineScope] to use for the operation.
     * @param coroutineContext The [CoroutineContext] to use for the operation.
     * @return A [CompletableFuture] that completes when the operation is done.  If successful, the result is a map of all key-value pairs in the storage layer.
     *
     * @exception NotConfiguredException If [ai.cheq.sst.android.core.Sst.configure] has not been called.
     *
     * @suppress
     */
    @Throws(NotConfiguredException::class)
    fun all(
        coroutineScope: CoroutineScope, coroutineContext: CoroutineContext
    ): CompletableFuture<Map<String, Any?>>

    /**
     * Returns a map of all key-value pairs in the storage layer.
     *
     * **[ai.cheq.sst.android.core.Sst.configure] must be called once before this method is called.**
     *
     * @return A map of all key-value pairs in the storage layer.
     *
     * @exception NotConfiguredException If [ai.cheq.sst.android.core.Sst.configure] has not been called.
     *
     * @sample samples.core.StorageLayer.Usage.all
     */
    @JvmSynthetic
    suspend fun all(): Map<String, Any?>

    /**
     * Clears all key-value pairs from the storage layer.
     *
     * **[ai.cheq.sst.android.core.Sst.configure] must be called once before this method is called.**
     *
     * @param coroutineScope The [CoroutineScope] to use for the operation.
     * @return A [CompletableFuture] that completes when the operation is done.
     *
     * @exception NotConfiguredException If [ai.cheq.sst.android.core.Sst.configure] has not been called.
     *
     * @suppress
     */
    @Throws(NotConfiguredException::class)
    fun clear(
        coroutineScope: CoroutineScope
    ): CompletableFuture<Void?>

    /**
     * Clears all key-value pairs from the storage layer.
     *
     * **[ai.cheq.sst.android.core.Sst.configure] must be called once before this method is called.**
     *
     * @param coroutineScope The [CoroutineScope] to use for the operation.
     * @param coroutineContext The [CoroutineContext] to use for the operation.
     * @return A [CompletableFuture] that completes when the operation is done.
     *
     * @exception NotConfiguredException If [ai.cheq.sst.android.core.Sst.configure] has not been called.
     *
     * @suppress
     */
    @Throws(NotConfiguredException::class)
    fun clear(
        coroutineScope: CoroutineScope, coroutineContext: CoroutineContext
    ): CompletableFuture<Void?>

    /**
     * Clears all key-value pairs from the storage layer.
     *
     * **[ai.cheq.sst.android.core.Sst.configure] must be called once before this method is called.**
     *
     * @exception NotConfiguredException If [ai.cheq.sst.android.core.Sst.configure] has not been called.
     *
     * @sample samples.core.StorageLayer.Usage.clear
     */
    @JvmSynthetic
    suspend fun clear()

    /**
     * Determines whether the storage layer contains the specified key.
     *
     * **[ai.cheq.sst.android.core.Sst.configure] must be called once before this method is called.**
     *
     * @param coroutineScope The [CoroutineScope] to use for the operation.
     * @param key The key to check.
     * @return A [CompletableFuture] that completes when the operation is done.  If successful, the result is `true` if the storage layer contains the key, `false` otherwise.
     *
     * @exception NotConfiguredException If [ai.cheq.sst.android.core.Sst.configure] has not been called.
     *
     * @suppress
     */
    @Throws(NotConfiguredException::class)
    fun contains(
        coroutineScope: CoroutineScope,
        key: String
    ): CompletableFuture<Boolean>

    /**
     * Determines whether the storage layer contains the specified key.
     *
     * **[ai.cheq.sst.android.core.Sst.configure] must be called once before this method is called.**
     *
     * @param coroutineScope The [CoroutineScope] to use for the operation.
     * @param coroutineContext The [CoroutineContext] to use for the operation.
     * @param key The key to check.
     * @return A [CompletableFuture] that completes when the operation is done.  If successful, the result is `true` if the storage layer contains the key, `false` otherwise.
     *
     * @exception NotConfiguredException If [ai.cheq.sst.android.core.Sst.configure] has not been called.
     *
     * @suppress
     */
    @Throws(NotConfiguredException::class)
    fun contains(
        coroutineScope: CoroutineScope,
        coroutineContext: CoroutineContext,
        key: String
    ): CompletableFuture<Boolean>

    /**
     * Determines whether the storage layer contains the specified key.
     *
     * **[ai.cheq.sst.android.core.Sst.configure] must be called once before this method is called.**
     *
     * @param key The key to check.
     * @return `true` if the storage layer contains the key, `false` otherwise.
     *
     * @exception NotConfiguredException If [ai.cheq.sst.android.core.Sst.configure] has not been called.
     *
     * @sample samples.core.StorageLayer.Usage.contains
     */
    @JvmSynthetic
    suspend fun contains(key: String): Boolean

    /**
     * Returns the value associated with the specified [key], or `null` if the key is not in the storage layer.
     *
     * **[ai.cheq.sst.android.core.Sst.configure] must be called once before this method is called.**
     *
     * @param key The key to get.
     * @return The value associated with the specified key, or `null` if the key is not in the storage layer
     *
     * @exception NotConfiguredException If [ai.cheq.sst.android.core.Sst.configure] has not been called.
     *
     * @sample samples.core.StorageLayer.Usage.get
     */
    suspend fun get(key: String): T?

    /**
     * Returns the value associated with the specified [key], or `null` if the key is not in the storage layer.
     *
     * **[ai.cheq.sst.android.core.Sst.configure] must be called once before this method is called.**
     *
     * @param coroutineScope The [CoroutineScope] to use for the operation.
     * @param key The key to get.
     * @return A [CompletableFuture] that completes when the operation is done.  If successful, the result is the value associated with the specified key, or `null` if the key is not in the storage layer
     *
     * @exception NotConfiguredException If [ai.cheq.sst.android.core.Sst.configure] has not been called.
     *
     * @suppress
     */
    @Throws(NotConfiguredException::class)
    fun get(
        coroutineScope: CoroutineScope,
        key: String
    ): CompletableFuture<T?>

    /**
     * Returns the value associated with the specified [key], or `null` if the key is not in the storage layer.
     *
     * **[ai.cheq.sst.android.core.Sst.configure] must be called once before this method is called.**
     *
     * @param coroutineScope The [CoroutineScope] to use for the operation.
     * @param coroutineContext The [CoroutineContext] to use for the operation.
     * @param key The key to get.
     * @return A [CompletableFuture] that completes when the operation is done.  If successful, the result is the value associated with the specified key, or `null` if the key is not in the storage layer
     *
     * @exception NotConfiguredException If [ai.cheq.sst.android.core.Sst.configure] has not been called.
     *
     * @suppress
     */
    @Throws(NotConfiguredException::class)
    fun get(
        coroutineScope: CoroutineScope,
        coroutineContext: CoroutineContext,
        key: String
    ): CompletableFuture<T?>

    /**
     * Removes the specified key and its corresponding value from the storage layer.
     *
     * **[ai.cheq.sst.android.core.Sst.configure] must be called once before this method is called.**
     *
     * @param coroutineScope The [CoroutineScope] to use for the operation.
     * @param key The key to remove.
     * @return `true` if the key was removed, `false` if the key was not in the storage layer.
     *
     * @exception NotConfiguredException If [ai.cheq.sst.android.core.Sst.configure] has not been called.
     *
     * @suppress
     */
    @Throws(NotConfiguredException::class)
    fun remove(
        coroutineScope: CoroutineScope,
        key: String
    ): CompletableFuture<Boolean>

    /**
     * Removes the specified key and its corresponding value from the storage layer.
     *
     * **[ai.cheq.sst.android.core.Sst.configure] must be called once before this method is called.**
     *
     * @param coroutineScope The [CoroutineScope] to use for the operation.
     * @param coroutineContext The [CoroutineContext] to use for the operation.
     * @param key The key to remove.
     * @return `true` if the key was removed, `false` if the key was not in the storage layer.
     *
     * @exception NotConfiguredException If [ai.cheq.sst.android.core.Sst.configure] has not been called.
     *
     * @suppress
     */
    @Throws(NotConfiguredException::class)
    fun remove(
        coroutineScope: CoroutineScope,
        coroutineContext: CoroutineContext,
        key: String
    ): CompletableFuture<Boolean>

    /**
     * Removes the specified key and its corresponding value from the storage layer.
     *
     * **[ai.cheq.sst.android.core.Sst.configure] must be called once before this method is called.**
     *
     * @param key The key to remove.
     * @return `true` if the key was removed, `false` if the key was not in the storage layer.
     *
     * @exception NotConfiguredException If [ai.cheq.sst.android.core.Sst.configure] has not been called.
     *
     * @sample samples.core.StorageLayer.Usage.remove
     */
    @JvmSynthetic
    suspend fun remove(key: String): Boolean
}