package ai.cheq.sst.android.core

import ai.cheq.sst.android.core.internal.AndroidLog
import ai.cheq.sst.android.core.config.VirtualBrowser
import ai.cheq.sst.android.core.internal.Clock
import ai.cheq.sst.android.core.models.Models
import io.ktor.client.engine.HttpClientEngine
import io.ktor.client.engine.android.Android
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.asCoroutineDispatcher
import java.util.concurrent.Executors

private const val defaultDomain = "t.nc0.co"

private const val defaultPublishPath = "sst"

private const val defaultNexusHost = "nexus.ensighten.com"

private const val defaultDebug = false

private const val defaultDataLayerName = "digitalData"

/**
 * Configuration object for the CHEQ SST service.
 *
 * @sample samples.core.Config.Usage.createConfig
 * @sample samples.core.Config.Usage.createConfigWithAllParameters
 */
class Config {
    /**
     * The name of the client is used to identify the client for the CHEQ SST service.
     */
    val clientName: String

    /**
     * The domain used to build the URL to the CHEQ SST service.  Should be your
     * first-party domain or “t.nc0.co” if using the provided third-party domain.
     */
    val domain: String

    /**
     * The Server-Side Tagging publish path if not using the default "sst".
     */
    val publishPath: String

    /**
     * The alternative Ensighten domain for loading tags from a source other than
     * nexus.ensighten.com within the SST instance. Typically only used with a
     * value of “nexus-test.ensighten.com” when testing tags in a committed state.
     */
    val nexusHost: String

    /**
     * The name of the data layer used to store the data layer in the CHEQ SST service.
     */
    val dataLayerName: String

    /**
     * The debug flag is used to enable debug logging and event callbacks in the CHEQ SST SDK.
     */
    val debug: Boolean

    /**
     * The virtual browser configuration for the CHEQ SST service.
     */
    val virtualBrowser: VirtualBrowser

    internal val models: Models
    internal val httpClientEngine: HttpClientEngine
    internal val clock: () -> Clock
    internal var modelScope: CoroutineScope
        private set
    internal var modelDispatcher: CoroutineDispatcher
        private set
    internal var monitoringScope: CoroutineScope
        private set
    internal var monitoringDispatcher: CoroutineDispatcher
        private set
    internal var sstScope: CoroutineScope
        private set
    internal var sstDispatcher: CoroutineDispatcher
        private set
    internal val log: Log

    /**
     * Create a new configuration object for the CHEQ SST service.
     *
     * @param clientName The name of the client is used to identify the client for the CHEQ SST service.
     * @param domain The domain used to build the URL to the CHEQ SST service.  Should be your first-party domain or “t.nc0.co” if using the provided third-party domain. Defaults to “t.nc0.co”.
     * @param publishPath The Server-Side Tagging publish path if not using the default "sst".
     * @param nexusHost The alternative Ensighten domain for loading tags from a source other than nexus.ensighten.com within the SST instance. Typically only used with a value of “nexus-test.ensighten.com” when testing tags in a committed state. Defaults to “nexus.ensighten.com”.
     * @param dataLayerName The name of the data layer used to store the data layer in the CHEQ SST service. Defaults to “digitalData”.
     * @param virtualBrowser The virtual browser configuration for the CHEQ SST service.
     * @param models The models used by the CHEQ SST SDK to collect information that is included automatically in events. Defaults to the base models: [ai.cheq.sst.android.core.models.AppModel], [ai.cheq.sst.android.core.models.DeviceModel], [ai.cheq.sst.android.core.models.LibraryModel].
     * @param debug The debug flag is used to enable debug logging in the CHEQ SST service. Defaults to false.
     *
     * @sample samples.core.Config.Usage.createConfig
     * @sample samples.core.Config.Usage.createConfigWithAllParameters
     */
    @JvmOverloads
    constructor(
        clientName: String,
        domain: String = defaultDomain,
        publishPath: String = defaultPublishPath,
        nexusHost: String = defaultNexusHost,
        dataLayerName: String = defaultDataLayerName,
        virtualBrowser: VirtualBrowser = VirtualBrowser(),
        models: Models = Models.default(),
        debug: Boolean = defaultDebug
    ) : this(clientName,
        domain,
        publishPath,
        nexusHost,
        dataLayerName,
        virtualBrowser,
        models,
        debug,
        Android.create(),
        { Clock.system() })

    internal constructor(
        clientName: String,
        domain: String = defaultDomain,
        publishPath: String = defaultPublishPath,
        nexusHost: String = defaultNexusHost,
        dataLayerName: String = defaultDataLayerName,
        virtualBrowser: VirtualBrowser = VirtualBrowser(),
        models: Models = Models.default(),
        debug: Boolean = defaultDebug,
        httpClientEngine: HttpClientEngine = Android.create(),
        clock: () -> Clock = { Clock.system() },
        log: Log = AndroidLog(),
        modelScope: CoroutineScope = CoroutineScope(
            SupervisorJob() + exceptionHandler(
                log,
                "model"
            )
        ),
        modelDispatcher: CoroutineDispatcher = Executors.newCachedThreadPool()
            .asCoroutineDispatcher(),
        monitoringScope: CoroutineScope = CoroutineScope(
            SupervisorJob() + exceptionHandler(
                log,
                "monitoring"
            )
        ),
        monitoringDispatcher: CoroutineDispatcher = Executors.newCachedThreadPool()
            .asCoroutineDispatcher(),
        sstScope: CoroutineScope = CoroutineScope(SupervisorJob() + exceptionHandler(log, "sst")),
        sstDispatcher: CoroutineDispatcher = Executors.newCachedThreadPool()
            .asCoroutineDispatcher(),
    ) {
        this.clientName = clientName
        this.domain = domain
        this.publishPath = publishPath
        this.nexusHost = nexusHost
        this.dataLayerName = dataLayerName
        this.virtualBrowser = virtualBrowser
        this.debug = debug
        this.models = models.finalize()
        this.httpClientEngine = httpClientEngine
        this.clock = clock
        this.modelScope = modelScope
        this.modelDispatcher = modelDispatcher
        this.monitoringScope = monitoringScope
        this.monitoringDispatcher = monitoringDispatcher
        this.sstScope = sstScope
        this.sstDispatcher = sstDispatcher
        this.log = log
    }

    internal companion object {
        fun exceptionHandler(log: Log, name: String) = CoroutineExceptionHandler { _, t ->
            log.e("CHEQ SST caught exception in $name scope", t)
        }
    }
}