package ai.cheq.sst.android.core

import ai.cheq.sst.android.core.exceptions.NotConfiguredException
import ai.cheq.sst.android.core.internal.DataStoreCollection
import ai.cheq.sst.android.core.internal.EventBus
import ai.cheq.sst.android.core.internal.lateVar
import ai.cheq.sst.android.core.serializers.dataLayerDataStore
import ai.cheq.sst.android.protobuf.dataLayer.DataLayer
import android.content.Context
import kotlinx.coroutines.CoroutineScope
import java.util.concurrent.CompletableFuture
import kotlin.coroutines.CoroutineContext
import kotlin.coroutines.EmptyCoroutineContext

/**
 * A data layer that stores key-value pairs to persistent storage.
 *
 * The data layer is a simple key-value store that can be used to store data that needs to
 * be included in every [Sst.trackEvent] call. The data layer is stored in the application's
 * data store, and can be accessed using the [add], [all], [clear], [contains], [get],
 * and [remove] methods.
 *
 * The data layer is implemented as a singleton, and is accessed using the [Sst.dataLayer] property.
 */
class DataLayer internal constructor() {
    private var contextProvider: ContextProvider by lateVar {
        NotConfiguredException("Sst.configure must be called before using the data layer")
    }
    private lateinit var log: Log
    private lateinit var eventBus: EventBus
    private lateinit var dataStoreCollection: DataStoreCollection<DataLayer, DataLayer, DataLayer.Builder>

    internal fun configure(contextProvider: ContextProvider, log: Log, eventBus: EventBus) {
        this.contextProvider = contextProvider
        this.log = log
        this.eventBus = eventBus
        this.dataStoreCollection = DataStoreCollection(
            DataStoreCollection.Config<DataLayer, DataLayer, DataLayer.Builder>(
                "dataLayer",
                Context::dataLayerDataStore,
                DataLayer.Builder::putDataLayer,
                DataLayer.Builder::clearDataLayer,
                DataLayer.Builder::removeDataLayer,
                DataLayer::getDataLayerMap,
                DataLayer::getDefaultInstance
            ),
            contextProvider,
            log,
            eventBus
        )
    }

    /**
     * Associates the specified [value] with the specified [key] in the data layer.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @param T The type of the value.
     * @param coroutineScope The coroutine scope to use for the operation.
     * @param coroutineContext The coroutine context to use for the operation.
     * @param key The key to add.
     * @param value The value to add.
     * @return A [CompletableFuture] that completes when the operation is done.
     *
     * @exception NotConfiguredException If [Sst.configure] has not been called.
     *
     * @suppress
     */
    @JvmOverloads
    @Throws(NotConfiguredException::class)
    fun <T : Any> add(
        coroutineScope: CoroutineScope,
        coroutineContext: CoroutineContext = EmptyCoroutineContext,
        key: String,
        value: T
    ): CompletableFuture<Void?> {
        return dataStoreCollection.add(coroutineScope, coroutineContext, key, value)
    }

    /**
     * Associates the specified [value] with the specified [key] in the data layer.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @param T The type of the value.
     * @param key The key to add.
     * @param value The value to add.
     *
     * @exception NotConfiguredException If [Sst.configure] has not been called.
     *
     * @sample samples.core.DataLayer.Usage.add
     */
    @JvmSynthetic
    suspend fun <T : Any> add(key: String, value: T) {
        return dataStoreCollection.add(key, value)
    }

    /**
     * Returns a map of all key-value pairs in the data layer.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @param coroutineScope The [CoroutineScope] to use for the operation.
     * @param coroutineContext The [CoroutineContext] to use for the operation.
     * @return A [CompletableFuture] that completes when the operation is done.  If successful, the result is a map of all key-value pairs in the data layer.
     *
     * @exception NotConfiguredException If [Sst.configure] has not been called.
     *
     * @suppress
     */
    @Throws(NotConfiguredException::class)
    fun all(
        coroutineScope: CoroutineScope, coroutineContext: CoroutineContext = EmptyCoroutineContext
    ): CompletableFuture<Map<String, Any?>> {
        return dataStoreCollection.all(coroutineScope, coroutineContext)
    }

    /**
     * Returns a map of all key-value pairs in the data layer.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @return A map of all key-value pairs in the data layer.
     *
     * @exception NotConfiguredException If [Sst.configure] has not been called.
     *
     * @sample samples.core.DataLayer.Usage.all
     */
    @JvmSynthetic
    suspend fun all(): Map<String, Any?> {
        return dataStoreCollection.all()
    }

    /**
     * Clears all key-value pairs from the data layer.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @param coroutineScope The [CoroutineScope] to use for the operation.
     * @param coroutineContext The [CoroutineContext] to use for the operation.
     * @return A [CompletableFuture] that completes when the operation is done.
     *
     * @exception NotConfiguredException If [Sst.configure] has not been called.
     *
     * @suppress
     */
    @JvmOverloads
    @Throws(NotConfiguredException::class)
    fun clear(
        coroutineScope: CoroutineScope, coroutineContext: CoroutineContext = EmptyCoroutineContext
    ): CompletableFuture<Void?> {
        return dataStoreCollection.clear(coroutineScope, coroutineContext)
    }

    /**
     * Clears all key-value pairs from the data layer.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @exception NotConfiguredException If [Sst.configure] has not been called.
     *
     * @sample samples.core.DataLayer.Usage.clear
     */
    @JvmSynthetic
    suspend fun clear() {
        dataStoreCollection.clear()
    }

    /**
     * Determines whether the data layer contains the specified key.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @param coroutineScope The [CoroutineScope] to use for the operation.
     * @param coroutineContext The [CoroutineContext] to use for the operation.
     * @param key The key to check.
     * @return A [CompletableFuture] that completes when the operation is done.  If successful, the result is `true` if the data layer contains the key, `false` otherwise.
     *
     * @exception NotConfiguredException If [Sst.configure] has not been called.
     *
     * @suppress
     */
    @JvmOverloads
    @Throws(NotConfiguredException::class)
    fun contains(
        coroutineScope: CoroutineScope,
        coroutineContext: CoroutineContext = EmptyCoroutineContext,
        key: String
    ): CompletableFuture<Boolean> {
        return dataStoreCollection.contains(coroutineScope, coroutineContext, key)
    }

    /**
     * Determines whether the data layer contains the specified key.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @param key The key to check.
     * @return `true` if the data layer contains the key, `false` otherwise.
     *
     * @exception NotConfiguredException If [Sst.configure] has not been called.
     *
     * @sample samples.core.DataLayer.Usage.contains
     */
    @JvmSynthetic
    suspend fun contains(key: String): Boolean {
        return dataStoreCollection.contains(key)
    }

    /**
     * Returns the value associated with the specified [key], or `null` if the key is not in the data layer.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @param T The type of the value.
     * @param key The key to get.
     * @return The value associated with the specified key, or `null` if the key is not in the data layer
     *
     * @exception NotConfiguredException If [Sst.configure] has not been called.
     *
     * @sample samples.core.DataLayer.Usage.get
     */
    @JvmSynthetic
    suspend inline fun <reified T : Any> get(key: String): T? {
        return get(key, T::class.javaObjectType)
    }

    @JvmSynthetic
    @PublishedApi
    internal suspend fun <T : Any> get(key: String, valueClass: Class<T>): T? {
        return dataStoreCollection.get(key, valueClass)
    }

    /**
     * Returns the value associated with the specified [key], or `null` if the key is not in the data layer.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @param coroutineScope The [CoroutineScope] to use for the operation.
     * @param coroutineContext The [CoroutineContext] to use for the operation.
     * @param T The type of the value.
     * @param key The key to get.
     * @param valueClass The class of the value.
     * @return A [CompletableFuture] that completes when the operation is done.  If successful, the result is the value associated with the specified key, or `null` if the key is not in the data layer
     *
     * @exception NotConfiguredException If [Sst.configure] has not been called.
     *
     * @suppress
     */
    @JvmOverloads
    @Throws(NotConfiguredException::class)
    fun <T : Any> get(
        coroutineScope: CoroutineScope,
        coroutineContext: CoroutineContext = EmptyCoroutineContext,
        key: String,
        valueClass: Class<T>
    ): CompletableFuture<T?> {
        return dataStoreCollection.get(coroutineScope, coroutineContext, key, valueClass)
    }

    /**
     * Removes the specified key and its corresponding value from the data layer.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @param coroutineScope The [CoroutineScope] to use for the operation.
     * @param coroutineContext The [CoroutineContext] to use for the operation.
     * @param key The key to remove.
     * @return `true` if the key was removed, `false` if the key was not in the data layer.
     *
     * @exception NotConfiguredException If [Sst.configure] has not been called.
     *
     * @suppress
     */
    @JvmOverloads
    @Throws(NotConfiguredException::class)
    fun remove(
        coroutineScope: CoroutineScope,
        coroutineContext: CoroutineContext = EmptyCoroutineContext,
        key: String
    ): CompletableFuture<Boolean> {
        return dataStoreCollection.remove(coroutineScope, coroutineContext, key)
    }

    /**
     * Removes the specified key and its corresponding value from the data layer.
     *
     * **[Sst.configure] must be called once before this method is called.**
     *
     * @param key The key to remove.
     * @return `true` if the key was removed, `false` if the key was not in the data layer.
     *
     * @exception NotConfiguredException If [Sst.configure] has not been called.
     *
     * @sample samples.core.DataLayer.Usage.remove
     */
    @JvmSynthetic
    suspend fun remove(key: String): Boolean {
        return dataStoreCollection.remove(key)
    }

    @JvmSynthetic
    internal suspend fun raw(): Map<String, String?> {
        return dataStoreCollection.raw()
    }
}
