package ai.cheq.sst.android.core.storage

import ai.cheq.sst.android.core.ContextProvider
import ai.cheq.sst.android.core.Log
import ai.cheq.sst.android.core.exceptions.NotConfiguredException
import ai.cheq.sst.android.core.internal.DataStoreCollection
import ai.cheq.sst.android.core.internal.EventBus
import ai.cheq.sst.android.core.internal.lateVar
import ai.cheq.sst.android.core.serializers.storageDataStore
import ai.cheq.sst.android.protobuf.storage.Storage
import android.content.Context
import com.fasterxml.jackson.core.JsonGenerator
import com.fasterxml.jackson.core.JsonParser
import com.fasterxml.jackson.core.JsonToken
import com.fasterxml.jackson.databind.DeserializationContext
import com.fasterxml.jackson.databind.JsonNode
import com.fasterxml.jackson.databind.SerializerProvider
import com.fasterxml.jackson.databind.annotation.JsonDeserialize
import com.fasterxml.jackson.databind.annotation.JsonSerialize
import com.fasterxml.jackson.databind.deser.std.StdDeserializer
import com.fasterxml.jackson.databind.ser.std.StdSerializer

/**
 * Provides access to [Cookies], [LocalStorage], and [SessionStorage].
 *
 * The storage is implemented as a singleton, and is accessed using the [ai.cheq.sst.android.core.Sst.storage] property.
 */
class Storage internal constructor() {
    private var contextProvider: ContextProvider by lateVar {
        NotConfiguredException("Sst.configure must be called before using storage")
    }
    private var _cookies: Cookies by lateVar {
        NotConfiguredException("Sst.configure must be called before using storage")
    }
    private var _local: LocalStorage by lateVar {
        NotConfiguredException("Sst.configure must be called before using storage")
    }
    private var _session: SessionStorage by lateVar {
        NotConfiguredException("Sst.configure must be called before using storage")
    }
    private lateinit var log: Log
    private lateinit var eventBus: EventBus

    /**
     * [Cookies] for automatic inclusion in all [ai.cheq.sst.android.core.Sst.trackEvent] calls.
     *
     * **[ai.cheq.sst.android.core.Sst.configure] must be called once before methods on this object are called.**
     *
     * @sample samples.core.Storage.Usage.cookies
     */
    val cookies get() = _cookies

    /**
     * [LocalStorage] for automatic inclusion in all [ai.cheq.sst.android.core.Sst.trackEvent] calls.
     *
     * **[ai.cheq.sst.android.core.Sst.configure] must be called once before methods on this object are called.**
     *
     * @sample samples.core.Storage.Usage.local
     */
    val local get() = _local

    /**
     * [SessionStorage] for automatic inclusion in all [ai.cheq.sst.android.core.Sst.trackEvent] calls.
     *
     * **[ai.cheq.sst.android.core.Sst.configure] must be called once before methods on this object are called.**
     *
     * @sample samples.core.Storage.Usage.session
     */
    val session get() = _session

    internal fun configure(contextProvider: ContextProvider, log: Log, eventBus: EventBus) {
        this.contextProvider = contextProvider
        this.log = log
        this.eventBus = eventBus
        _cookies = Cookies(
            DataStoreStorageCollection.create<Cookie>(
                DataStoreCollection.Config<Storage, Storage, Storage.Builder>(
                    "storage.cookies",
                    Context::storageDataStore,
                    Storage.Builder::putCookies,
                    Storage.Builder::clearCookies,
                    Storage.Builder::removeCookies,
                    Storage::getCookiesMap,
                    Storage::getDefaultInstance
                ), contextProvider, log, eventBus
            )
        )
        _local = LocalStorage(
            DataStoreStorageCollection.create<Item>(
                DataStoreCollection.Config<Storage, Storage, Storage.Builder>(
                    "storage.local",
                    Context::storageDataStore,
                    Storage.Builder::putLocal,
                    Storage.Builder::clearLocal,
                    Storage.Builder::removeLocal,
                    Storage::getLocalMap,
                    Storage::getDefaultInstance
                ), contextProvider, log, eventBus
            )
        )
        _session = SessionStorage(
            DataStoreStorageCollection.create<Item>(
                DataStoreCollection.Config<Storage, Storage, Storage.Builder>(
                    "storage.session",
                    Context::storageDataStore,
                    Storage.Builder::putSession,
                    Storage.Builder::clearSession,
                    Storage.Builder::removeSession,
                    Storage::getSessionMap,
                    Storage::getDefaultInstance
                ), contextProvider, log, eventBus
            )
        )
    }

    internal suspend fun raw(): Data {
        return Data(
            _cookies.raw(), _local.raw(), _session.raw()
        )
    }

    @JsonSerialize(using = Data.Serializer::class)
    @JsonDeserialize(using = Data.Deserializer::class)
    internal data class Data(
        val cookies: List<String>, val localStorage: List<String>, val sessionStorage: List<String>
    ) {
        internal class Serializer : StdSerializer<Data>(Data::class.java, false) {
            override fun serialize(
                value: Data?, gen: JsonGenerator?, provider: SerializerProvider?
            ) {
                gen!!.writeStartObject()
                writeList(gen, "cookies", value!!.cookies)
                writeList(gen, "localStorage", value.localStorage)
                writeList(gen, "sessionStorage", value.sessionStorage)
                gen.writeEndObject()
            }

            override fun isEmpty(provider: SerializerProvider?, value: Data?) =
                value == null || (value.cookies.isEmpty() && value.localStorage.isEmpty() && value.sessionStorage.isEmpty())

            private fun writeList(gen: JsonGenerator, key: String, values: List<String>) {
                if (values.isNotEmpty()) {
                    gen.writeFieldName(key)
                    gen.writeStartArray()
                    values.forEach {
                        gen.writeRawValue(it)
                    }
                    gen.writeEndArray()
                }
            }
        }

        internal class Deserializer : StdDeserializer<Data>(Data::class.java) {
            override fun deserialize(
                p: JsonParser?, ctxt: DeserializationContext?
            ): Data {
                val cookies = mutableListOf<String>()
                val localStorage = mutableListOf<String>()
                val sessionStorage = mutableListOf<String>()
                if (p != null && p.codec != null && ctxt != null) {
                    if (p.currentToken == JsonToken.FIELD_NAME) {
                        p.nextToken()
                    }
                    val node: JsonNode? = p.codec.readTree(p)
                    node?.fields()?.forEach {
                        readList(node, it, "cookies", cookies) || readList(
                            node, it, "localStorage", localStorage
                        ) || readList(node, it, "sessionStorage", sessionStorage)
                    }
                }
                return Data(cookies, localStorage, sessionStorage)
            }

            private fun readList(
                parent: JsonNode,
                field: MutableMap.MutableEntry<String, JsonNode>,
                key: String,
                destination: MutableList<String>
            ): Boolean {
                if (field.key != key) {
                    return false
                }
                val node = parent.get(key)
                node.forEach { f ->
                    destination.add(f.toString())
                }
                return true
            }
        }
    }
}
