package ai.cheq.sst.android.core.monitoring

import ai.cheq.sst.android.core.Log
import ai.cheq.sst.android.core.Utils
import ai.cheq.sst.android.core.internal.EventBus
import ai.cheq.sst.android.core.internal.MapTypeReference
import io.ktor.client.HttpClient
import io.ktor.client.call.HttpClientCall
import io.ktor.client.call.body
import io.ktor.client.plugins.api.ClientHook
import io.ktor.client.plugins.api.createClientPlugin
import io.ktor.client.request.HttpRequestBuilder
import io.ktor.client.request.HttpSendPipeline
import io.ktor.client.statement.HttpReceivePipeline
import io.ktor.client.statement.HttpResponse
import io.ktor.client.statement.HttpResponseContainer
import io.ktor.client.statement.HttpResponsePipeline
import io.ktor.client.statement.request
import io.ktor.http.charset
import io.ktor.http.content.OutgoingContent
import io.ktor.util.pipeline.PipelineContext
import io.ktor.util.toMap
import io.ktor.utils.io.ByteReadChannel
import io.ktor.utils.io.core.readText
import io.ktor.utils.io.readRemaining
import io.ktor.utils.io.writer
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineScope
import java.nio.charset.Charset

private typealias HttpResponseEvent = ai.cheq.sst.android.core.monitoring.HttpResponse

fun interface EventHandler<T : Event> {
    suspend fun handle(event: T)
}

private object SendHook :
    ClientHook<suspend SendHook.Context.(response: HttpRequestBuilder) -> Unit> {
    class Context(private val context: PipelineContext<Any, HttpRequestBuilder>) {
        suspend fun proceedWith(content: Any) = context.proceedWith(content)
    }

    override fun install(
        client: HttpClient, handler: suspend Context.(request: HttpRequestBuilder) -> Unit
    ) {
        client.sendPipeline.intercept(HttpSendPipeline.Monitoring) {
            handler(Context(this), context)
        }
    }
}

private object ResponseHook :
    ClientHook<suspend ResponseHook.Context.(response: HttpResponse) -> Unit> {
    class Context(private val context: PipelineContext<HttpResponse, Unit>) {
        suspend fun proceed() = context.proceed()
    }

    override fun install(
        client: HttpClient, handler: suspend Context.(response: HttpResponse) -> Unit
    ) {
        client.receivePipeline.intercept(HttpReceivePipeline.State) {
            handler(Context(this), subject)
        }
    }
}

private object ReceiveHook :
    ClientHook<suspend ReceiveHook.Context.(call: HttpClientCall) -> Unit> {
    class Context(private val context: PipelineContext<HttpResponseContainer, HttpClientCall>) {
        suspend fun proceed() = context.proceed()
    }

    override fun install(
        client: HttpClient, handler: suspend Context.(call: HttpClientCall) -> Unit
    ) {
        client.responsePipeline.intercept(HttpResponsePipeline.Receive) {
            handler(Context(this), context)
        }
    }
}

private fun OutgoingContent.WriteChannelContent.toReadChannel(
    monitoringScope: CoroutineScope, monitoringDispatcher: CoroutineDispatcher
): ByteReadChannel = monitoringScope.writer(monitoringDispatcher) {
    writeTo(channel)
}.channel

private suspend inline fun ByteReadChannel.tryReadText(charset: Charset): String? = try {
    readRemaining().readText(charset = charset)
} catch (cause: Throwable) {
    null
}

internal class MonitoringPluginConfig {
    lateinit var eventBus: EventBus
    lateinit var monitoringScope: CoroutineScope
    lateinit var monitoringDispatcher: CoroutineDispatcher
}

internal fun monitoringPlugin(log: Log) =
    createClientPlugin("MonitoringPlugin", ::MonitoringPluginConfig) {
        val eventBus = pluginConfig.eventBus
        val monitoringScope = pluginConfig.monitoringScope
        val monitoringDispatcher = pluginConfig.monitoringDispatcher

        on(SendHook) { request ->
            var content = request.body
            if (eventBus.enabled<HttpRequest>()) {
                var bodyObject: Map<String, Any>? = null
                if (request.body is OutgoingContent.WriteChannelContent) {
                    val bodyContent = request.body as OutgoingContent.WriteChannelContent
                    val charset = bodyContent.contentType?.charset() ?: Charsets.UTF_8
                    val body = bodyContent.toReadChannel(monitoringScope, monitoringDispatcher)
                        .tryReadText(charset)
                    if (body != null) {
                        bodyObject = Utils.jsonMapper.readValue(body, MapTypeReference.INSTANCE)
                    }
                    content = bodyContent
                }
                val requestData = HttpRequest(
                    request.url.toString(),
                    request.method.value,
                    request.headers.build().toMap(),
                    bodyObject
                )

                if (log.isDebugLoggable()) {
                    log.d("CHEQ SST Http request: ${Utils.jsonMapper.writeValueAsString(requestData)}")
                }
                eventBus.publish(requestData)
            }
            proceedWith(content)
        }

        on(ResponseHook) { response ->
            if (eventBus.enabled<HttpResponseEvent>()) {
                val responseStatus = HttpStatusCode(response.status.value, response.status.description)
                val responseData = HttpResponseEvent(
                    response.request.url.toString(),
                    response.headers.toMap(),
                    responseStatus,
                    response.body()
                )
                if (log.isDebugLoggable()) {
                    log.d(
                        "CHEQ SST Http response: ${
                            Utils.jsonMapper.writeValueAsString(
                                responseData
                            )
                        }"
                    )
                }
                eventBus.publish(responseData)
            }

            proceed()
        }

        on(ReceiveHook) { _ ->
            proceed()
        }
    }