package ai.cheq.sst.android.core

import ai.cheq.sst.android.core.internal.ConfigurableContextProvider
import ai.cheq.sst.android.core.internal.EventBus
import ai.cheq.sst.android.core.internal.store.DataStoreCollection
import ai.cheq.sst.android.core.internal.store.NoopDataStoreCollection
import ai.cheq.sst.android.core.internal.store.ProtobufDataStoreCollection
import ai.cheq.sst.android.core.serializers.dataLayerDataStore
import ai.cheq.sst.android.protobuf.dataLayer.DataLayer
import android.content.Context
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineScope
import java.util.concurrent.CompletableFuture

/**
 * A data layer that stores key-value pairs to persistent storage.
 *
 * The data layer is a simple key-value store that can be used to store data that needs to
 * be included in every [Sst.trackEvent] call. The data layer is stored in the application's
 * data store, and can be accessed using the [add], [all], [clear], [contains], [get],
 * and [remove] methods.
 *
 * The data layer is implemented as a singleton, and is accessed using the [Sst.dataLayer] property.
 */
class DataLayer internal constructor() {
    private var contextProvider = ConfigurableContextProvider()
    private var dataStoreCollection: DataStoreCollection = NoopDataStoreCollection()

    internal fun configure(
        contextProvider: ConfigurableContextProvider,
        log: Log,
        eventBus: EventBus,
        coroutineScope: CoroutineScope,
        coroutineDispatcher: CoroutineDispatcher
    ) {
        this.contextProvider.configure(contextProvider)
        this.dataStoreCollection = ProtobufDataStoreCollection(
            ProtobufDataStoreCollection.Config<DataLayer, DataLayer, DataLayer.Builder>(
                "dataLayer",
                Context::dataLayerDataStore,
                DataLayer.Builder::putDataLayer,
                DataLayer.Builder::clearDataLayer,
                DataLayer.Builder::removeDataLayer,
                DataLayer::getDataLayerMap,
                DataLayer::getDefaultInstance,
            ), contextProvider, log, eventBus, coroutineScope, coroutineDispatcher
        )
    }

    /**
     * Associates the specified [value] with the specified [key] in the data layer.
     *
     * **Important:** [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op.
     *
     * @param T The type of the value.
     * @param key The key to add.
     * @param value The value to add.
     * @return A [CompletableFuture] that completes when the operation is done.
     *
     * @suppress
     */
    fun <T : Any> addAsync(key: String, value: T): CompletableFuture<Void?> {
        return dataStoreCollection.addAsync(key, value)
    }

    /**
     * Associates the specified [value] with the specified [key] in the data layer.
     *
     * **Important:**
     *   * [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op
     *   * This method cannot be called in the main thread as it may block leading to ANRs
     *
     * @param T The type of the value.
     * @param key The key to add.
     * @param value The value to add.
     *
     * @sample samples.core.DataLayer.Usage.add
     */
    @JvmSynthetic
    suspend fun <T : Any> add(key: String, value: T) {
        return dataStoreCollection.add(key, value)
    }

    /**
     * Returns a map of all key-value pairs in the data layer.
     *
     * **Important:** [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op.
     *
     * @return A [CompletableFuture] that completes when the operation is done.  If successful, the result is a map of all key-value pairs in the data layer.
     *
     * @suppress
     */
    fun allAsync(): CompletableFuture<Map<String, Any?>> {
        return dataStoreCollection.allAsync()
    }

    /**
     * Returns a map of all key-value pairs in the data layer.
     *
     * **Important:**
     *   * [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op
     *   * This method cannot be called in the main thread as it may block leading to ANRs
     *
     * @return A map of all key-value pairs in the data layer.
     *
     * @sample samples.core.DataLayer.Usage.all
     */
    @JvmSynthetic
    suspend fun all(): Map<String, Any?> {
        return dataStoreCollection.all()
    }

    /**
     * Clears all key-value pairs from the data layer.
     *
     * **Important:** [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op.
     *
     * @return A [CompletableFuture] that completes when the operation is done.
     *
     * @suppress
     */
    fun clearAsync(): CompletableFuture<Void?> {
        return dataStoreCollection.clearAsync()
    }

    /**
     * Clears all key-value pairs from the data layer.
     *
     * **Important:**
     *   * [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op
     *   * This method cannot be called in the main thread as it may block leading to ANRs
     *
     * @sample samples.core.DataLayer.Usage.clear
     */
    @JvmSynthetic
    suspend fun clear() {
        dataStoreCollection.clear()
    }

    /**
     * Determines whether the data layer contains the specified key.
     *
     * **Important:** [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op.
     *
     * @param key The key to check.
     * @return A [CompletableFuture] that completes when the operation is done.  If successful, the result is `true` if the data layer contains the key, `false` otherwise.
     *
     * @suppress
     */
    fun containsAsync(key: String): CompletableFuture<Boolean> {
        return dataStoreCollection.containsAsync(key)
    }

    /**
     * Determines whether the data layer contains the specified key.
     *
     * **Important:**
     *   * [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op
     *   * This method cannot be called in the main thread as it may block leading to ANRs
     *
     * @param key The key to check.
     * @return `true` if the data layer contains the key, `false` otherwise.
     *
     * @sample samples.core.DataLayer.Usage.contains
     */
    @JvmSynthetic
    suspend fun contains(key: String): Boolean {
        return dataStoreCollection.contains(key)
    }

    /**
     * Returns the value associated with the specified [key], or `null` if the key is not in the data layer.
     *
     * **Important:**
     *   * [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op
     *   * This method cannot be called in the main thread as it may block leading to ANRs
     *
     * @param T The type of the value.
     * @param key The key to get.
     * @return The value associated with the specified key, or `null` if the key is not in the data layer
     *
     * @sample samples.core.DataLayer.Usage.get
     */
    @JvmSynthetic
    suspend inline fun <reified T : Any> get(key: String): T? {
        return get(key, T::class.javaObjectType)
    }

    @JvmSynthetic
    @PublishedApi
    internal suspend fun <T : Any> get(key: String, valueClass: Class<T>): T? {
        return dataStoreCollection.get(key, valueClass)
    }

    /**
     * Returns the value associated with the specified [key], or `null` if the key is not in the data layer.
     *
     * **Important:** [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op.
     *
     * @param T The type of the value.
     * @param key The key to get.
     * @param valueClass The class of the value.
     * @return A [CompletableFuture] that completes when the operation is done.  If successful, the result is the value associated with the specified key, or `null` if the key is not in the data layer
     *
     * @suppress
     */
    fun <T : Any> getAsync(key: String, valueClass: Class<T>): CompletableFuture<T?> {
        return dataStoreCollection.getAsync(key, valueClass)
    }

    /**
     * Removes the specified key and its corresponding value from the data layer.
     *
     * **Important:** [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op.
     *
     * @param key The key to remove.
     * @return `true` if the key was removed, `false` if the key was not in the data layer.
     *
     * @suppress
     */
    fun removeAsync(key: String): CompletableFuture<Boolean> {
        return dataStoreCollection.removeAsync(key)
    }

    /**
     * Removes the specified key and its corresponding value from the data layer.
     *
     * **Important:**
     *   * [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op
     *   * This method cannot be called in the main thread as it may block leading to ANRs
     *
     * @param key The key to remove.
     * @return `true` if the key was removed, `false` if the key was not in the data layer.
     *
     * @sample samples.core.DataLayer.Usage.remove
     */
    @JvmSynthetic
    suspend fun remove(key: String): Boolean {
        return dataStoreCollection.remove(key)
    }

    @JvmSynthetic
    internal suspend fun raw(): Map<String, String?> {
        return dataStoreCollection.raw()
    }

    @JvmSynthetic
    internal suspend fun reset(data: Boolean = false) {
        if (data) {
            dataStoreCollection.clear()
        }
        contextProvider = ConfigurableContextProvider()
        dataStoreCollection = NoopDataStoreCollection()
    }
}
