package ai.cheq.sst.android.core

import ai.cheq.sst.android.core.internal.AndroidLog

/**
 * The policy to enforce when the SST SDK is not configured.
 */
enum class NotConfiguredPolicy {
    /**
     * Ignore the fact that the SST SDK is not configured.
     */
    IGNORE,

    /**
     * Log a message if the SST SDK is not configured.
     */
    NOTIFY,

    /**
     * Throw an exception if the SST SDK is not configured.
     */
    ERROR;

    internal companion object {
        private const val PROPERTY_KEY = "cheq.sst.configurePolicy"
        private const val MESSAGE =
            "SST SDK is not configured. Please call Sst.configure() before using the SDK."
        private val log = AndroidLog()

        fun set(policy: NotConfiguredPolicy) {
            System.setProperty(PROPERTY_KEY, policy.name)
        }

        fun get(): NotConfiguredPolicy = System.getProperty(PROPERTY_KEY).let {
            if (it.isNullOrBlank()) {
                IGNORE
            } else {
                NotConfiguredPolicy.valueOf(it)
            }
        }

        fun enforce() {
            val policy = get()
            if (policy == IGNORE) {
                return
            }
            val exception = IllegalStateException(MESSAGE)
            if (policy == NOTIFY || policy == ERROR) {
                log.e(MESSAGE, exception)
            }
            if (policy == ERROR) {
                throw exception
            }
        }

        fun error(): Nothing {
            val exception = IllegalStateException(MESSAGE)
            log.e(MESSAGE, exception)
            throw exception
        }
    }
}