package ai.cheq.sst.android.core

import ai.cheq.sst.android.core.Utils.Companion.truncate
import ai.cheq.sst.android.core.domain.Data
import ai.cheq.sst.android.core.internal.ConfigurableContextProvider
import ai.cheq.sst.android.core.internal.EventBus
import ai.cheq.sst.android.core.internal.HttpClient
import ai.cheq.sst.android.core.internal.HttpResult
import ai.cheq.sst.android.core.models.AppModel
import ai.cheq.sst.android.core.models.DeviceModel
import ai.cheq.sst.android.core.monitoring.Configured
import ai.cheq.sst.android.core.monitoring.Error
import ai.cheq.sst.android.core.monitoring.Event
import ai.cheq.sst.android.core.monitoring.EventHandler
import ai.cheq.sst.android.core.monitoring.HttpRequest
import ai.cheq.sst.android.core.monitoring.HttpResponse
import ai.cheq.sst.android.core.settings.SettingsRepository
import ai.cheq.sst.android.core.storage.Storage
import io.ktor.client.request.cookie
import io.ktor.client.request.header
import io.ktor.http.HttpHeaders
import io.ktor.http.HttpMethod
import io.ktor.http.URLBuilder
import io.ktor.http.URLProtocol
import io.ktor.http.setCookie
import io.reactivex.rxjava3.core.Observable
import kotlinx.coroutines.CoroutineName
import kotlinx.coroutines.future.future
import kotlinx.coroutines.launch
import java.util.concurrent.CompletableFuture
import kotlin.coroutines.cancellation.CancellationException
import kotlin.reflect.KClass

private typealias SstEvent = ai.cheq.sst.android.core.Event

private const val CHEQ_UUID_COOKIE_NAME = "uuid"

/**
 * The main entry point for the CHEQ SST SDK.
 *
 * The `Sst` object is the main entry point for the CHEQ SST SDK. It provides methods for configuring
 * the SDK, tracking events, and accessing the data layer.
 *
 * The Event and included model information is encoded according to
 * the [SST Protocol](https://help.ensighten.com/hc/en-us/articles/22990062690577-SST-Protocol-Reference)
 * before being sent to SST.
 *
 * __Links__
 *
 * More information about parameters that can be passed to the CHEQ SST service, see [Getting Started with Data Collection Pixels](https://help.ensighten.com/hc/en-us/articles/22962356688273-Getting-Started-with-Data-Collection-Pixels)
 *
 * @sample samples.core.Sst.Usage.trackEvent
 * @sample samples.core.Sst.Usage.trackEventWithParameters
 */
object Sst {
    private val contextProvider = ConfigurableContextProvider()
    private val client: HttpClient = HttpClient()
    private var config: Config = Config.EMPTY
    private val eventBus = EventBus()

    /**
     * The [DataLayer] that stores key-value pairs to persistent storage for automatic inclusion in all [Sst.trackEvent] calls.
     *
     * @sample samples.core.Sst.Usage.dataLayer
     */
    @JvmStatic
    val dataLayer = DataLayer()

    /**
     * [Storage] provides read/write access to cookies, local data, and session data for automatic inclusion in all [Sst.trackEvent] calls.
     *
     * @sample samples.core.Sst.Usage.storageCookies
     * @sample samples.core.Sst.Usage.storageLocal
     * @sample samples.core.Sst.Usage.storageSession
     */
    @JvmStatic
    val storage = Storage()

    /**
     * Configures the CHEQ SST SDK with the specified [contextProvider] and [config].
     *
     * **Important:**
     *   * This method should be called before any other methods in the SDK are used
     *   * This method cannot be called in the main thread as it may block leading to ANRs
     *
     * @param config The [Config] object to use for the SDK.
     * @param contextProvider The Android [ContextProvider] to use for the SDK.
     *
     * @sample samples.core.Sst.Usage.configure
     * @sample samples.core.Sst.Usage.configureOverrideDefaults
     * @sample samples.core.Sst.Usage.configureWithCustomModel
     * @sample samples.core.Sst.Usage.configureWithoutDeviceModelAndOverriddenUserAgent
     */
    @JvmStatic
    @JvmSynthetic
    suspend fun configure(config: Config, contextProvider: ContextProvider): Boolean {
        config.log.i("CHEQ SST configure: start")
        try {
            reset()
            this.contextProvider.configure(contextProvider)
            this.config = config
            configureEventBus(config)
            dataLayer.configure(
                this.contextProvider, config.log, eventBus, config.sstScope, config.sstDispatcher
            )
            storage.configure(
                this.contextProvider, config.log, eventBus, config.sstScope, config.sstDispatcher
            )
            client.configure(config, eventBus)
            val result = config.models.initialize(config, contextProvider)
            val failureCount = result.failures.count()
            if (failureCount > 0) {
                publishConfigureError(message = result.toString())
                if (failureCount == result.size) {
                    return false
                }
            }
            eventBus.publish(Configured())
            config.log.i("CHEQ SST configure: complete")
            return true
        } catch (e: CancellationException) {
            throw e
        } catch (e: Exception) {
            publishConfigureError(e::class.simpleName, e.message)
            return false
        }
    }

    /**
     * Configures the CHEQ SST SDK with the specified [contextProvider] and [config].
     *
     * **Important:** This should be called before any other methods in the SDK are used.
     *
     * @param config The [Config] object to use for the SDK.
     * @param contextProvider The Android [ContextProvider] to use for the SDK.
     *
     * @return A [CompletableFuture] that completes when the operation is done.
     * @suppress
     */
    @JvmStatic
    fun configureAsync(
        config: Config,
        contextProvider: ContextProvider
    ): CompletableFuture<Boolean> {
        return config.sstScope.future(config.sstDispatcher) {
            return@future configure(config, contextProvider)
        }
    }

    @JvmSynthetic
    internal suspend fun reset(data: Boolean = false) {
        if (config != Config.EMPTY && contextProvider.isConfigured()) {
            config.models.stop(config, contextProvider.requireProvider())
        }
        storage.reset(data)
        dataLayer.reset(data)
        eventBus.reset()
        client.reset()
        contextProvider.reset()
        config = Config.EMPTY
    }

    /**
     * Returns the current configuration of the SDK.
     *
     * @return The [Config] object or an empty config if the [configure] has not been called.
     */
    @JvmStatic
    fun config() = config

    /**
     * Registers a callback to be invoked when [Sst] is configured.
     *
     * **Important:** This method cannot be called in the main thread as it may block leading to ANRs
     *
     * @param action The callback to be invoked when [Sst] is configured.
     */
    @JvmStatic
    @JvmSynthetic
    suspend fun onConfigured(action: EventHandler<Configured>) {
        eventBus.subscribe<Configured>(action::handle)
    }

    /**
     * Returns an [Observable] that emits [Configured] objects.
     *
     * @return An [Observable] that emits [Configured] objects.
     *
     * @suppress
     */
    @JvmStatic
    fun observeConfigured(): Observable<Configured> {
        return eventBus.observe<Configured>(config.sstDispatcher)
    }

    /**
     * Registers a callback to be invoked when a [HttpRequest] is emitted.
     *
     * **Important:** This method cannot be called in the main thread as it may block leading to ANRs
     *
     * @param action The callback to be invoked when a [HttpRequest] is emitted.
     */
    @JvmStatic
    @JvmSynthetic
    suspend fun onRequest(action: EventHandler<HttpRequest>) {
        eventBus.subscribe<HttpRequest>(action::handle)
    }

    /**
     * Returns an [Observable] that emits [HttpRequest] objects.
     *
     * @return An [Observable] that emits [HttpRequest] objects.
     *
     * @suppress
     */
    @JvmStatic
    fun observeRequest(): Observable<HttpRequest> {
        return eventBus.observe<HttpRequest>(config.sstDispatcher)
    }

    /**
     * Registers a callback to be invoked when a [HttpResponse] is emitted.
     *
     * **Important:** This method cannot be called in the main thread as it may block leading to ANRs
     *
     * @param action The callback to be invoked when a [HttpResponse] is emitted.
     */
    @JvmStatic
    @JvmSynthetic
    suspend fun onResponse(action: EventHandler<HttpResponse>) {
        eventBus.subscribe<HttpResponse>(action::handle)
    }

    /**
     * Returns an [Observable] that emits [HttpResponse] objects.
     *
     * @return An [Observable] that emits [HttpResponse] objects.
     *
     * @suppress
     */
    @JvmStatic
    fun observeResponse(): Observable<HttpResponse> {
        return eventBus.observe<HttpResponse>(config.sstDispatcher)
    }

    /**
     * Registers a callback to be invoked when an [Error] is emitted.
     *
     * **Important:** This method cannot be called in the main thread as it may block leading to ANRs
     *
     * @param action The callback to be invoked when an [Error] is emitted.
     */
    @JvmStatic
    suspend fun onError(action: EventHandler<Error>) {
        eventBus.subscribe<Error>(action::handle)
    }

    /**
     * Returns an [Observable] that emits [Error] objects.
     *
     * @return An [Observable] that emits [Error] objects.
     *
     * @suppress
     */
    @JvmStatic
    fun observerError(): Observable<Error> {
        return eventBus.observe<Error>(config.sstDispatcher)
    }

    /**
     * Sends the specified [event] to the CHEQ SST server, including data from all configured models
     * and the key-value pairs stored in the [dataLayer].
     *
     * **Important:** [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op.
     *
     * @param event The [Event] to track.
     *
     * @return A [CompletableFuture] that completes when the operation is done.
     *
     * @suppress
     */
    @JvmStatic
    fun trackEventAsync(event: ai.cheq.sst.android.core.Event): CompletableFuture<Void?> {
        return config.sstScope.future(config.sstDispatcher) {
            trackEvent(event)
            null
        }
    }

    /**
     * Sends the specified [event] to the CHEQ SST server, including data from all configured models
     * and the key-value pairs stored in the [dataLayer].
     *
     * **Important:**
     *   * [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op
     *   * This method cannot be called in the main thread as it may block leading to ANRs.
     *
     * @param event The [Event] to track.
     *
     * __Links__
     *
     * More information about parameters that can be passed to the CHEQ SST service, see [Getting Started with Data Collection Pixels](https://help.ensighten.com/hc/en-us/articles/22962356688273-Getting-Started-with-Data-Collection-Pixels)
     *
     * @sample samples.core.Sst.Usage.trackEvent
     * @sample samples.core.Sst.Usage.trackEventWithParameters
     */
    @JvmSynthetic
    suspend fun trackEvent(event: SstEvent) {
        try {
            val modelData = config.models.collect(config, contextProvider)
            val data = Data(config, event, modelData, dataLayer.raw(), storage.raw())
            val url = trackEventUrl()
            val cheqUuid = SettingsRepository.readCheqUuid(contextProvider)
            val userAgent =
                config.virtualBrowser.userAgent ?: data.dataLayer.mobileData?.get<DeviceModel.Data>(
                    includeAll = true
                )?.userAgent
            val result = client.sendHttp(
                url,
                HttpMethod.Post,
                data = data,
                parameters = event.parameters,
                userAgent = userAgent
            ) {
                if (cheqUuid != null) {
                    cookie(
                        CHEQ_UUID_COOKIE_NAME,
                        cheqUuid.toString(),
                        domain = config.domain,
                        secure = true,
                        httpOnly = true
                    )
                }
            }
            when (result) {
                is HttpResult.Skipped -> {}
                is HttpResult.Success -> {
                    SettingsRepository.storeCheqUuid(
                        contextProvider,
                        result.response.setCookie()
                            .firstOrNull { it.name == CHEQ_UUID_COOKIE_NAME }?.value
                    )
                }

                is HttpResult.Error.HttpError -> handleHttpResultError(result)
                is HttpResult.Error.NetworkError -> handleHttpResultError(result)
                is HttpResult.Error.SerializationError -> handleHttpResultError(result)
                is HttpResult.Error.UnknownError -> handleHttpResultError(result)
            }
        } catch (e: CancellationException) {
            throw e
        } catch (e: Exception) {
            publishError(
                "TrackEvent",
                e::class.simpleName ?: "Error",
                "Sst.trackEvent",
                e.message ?: "Unknown"
            )
        }
    }

    /**
     * Gets the CHEQ UUID (universally unique identifier) from persistent storage.
     * The CHEQ UUID is automatically set by the SDK when an [Sst.trackEvent] call is made.
     *
     * The CHEQ UUID is used to group together events from the same device and application across multiple sessions.
     *
     * **Important:** [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op.
     *
     * @return A [CompletableFuture] that completes when the operation is done. If successful, the result is the CHEQ UUID or `null` if it does not exist.
     *
     * @suppress
     */
    @JvmStatic
    fun getCheqUuidAsync(): CompletableFuture<String?> {
        val ctxProvider = contextProvider
        return config.sstScope.future(config.sstDispatcher) {
            getCheqUuid(ctxProvider)
        }
    }

    /**
     * Gets the CHEQ UUID (universally unique identifier) from persistent storage.
     * The CHEQ UUID is automatically set by the SDK when an [Sst.trackEvent] call is made.
     *
     * The CHEQ UUID is used to group together events from the same device and application across multiple sessions.
     *
     * **Important:**
     *   * [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op
     *   * This method cannot be called in the main thread as it may block leading to ANRs
     *
     * @return The CHEQ UUID or `null` if it does not exist.
     *
     * @sample samples.core.Sst.Usage.getCheqUuid
     */
    @JvmSynthetic
    suspend fun getCheqUuid(): String? {
        return getCheqUuid(contextProvider)
    }

    /**
     * Clears the CHEQ UUID (universally unique identifier) from persistent storage.
     * The CHEQ UUID is automatically set by the SDK when an [Sst.trackEvent] call is made.
     *
     * The CHEQ UUID is used to group together events from the same device and application across multiple sessions.
     *
     * **Important:** [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op.
     *
     * @return A [CompletableFuture] that completes when the operation is done.
     *
     * @suppress
     */
    @JvmStatic
    fun clearCheqUuidAsync(): CompletableFuture<Void?> {
        val ctxProvider = contextProvider
        return config.sstScope.future(config.sstDispatcher) {
            clearCheqUuid(ctxProvider)
            null
        }
    }

    /**
     * Clears the CHEQ UUID (universally unique identifier) from persistent storage.
     * The CHEQ UUID is automatically set by the SDK when an [Sst.trackEvent] call is made.
     *
     * The CHEQ UUID is used to group together events from the same device and application across multiple sessions.
     *
     * **Important:**
     *   * [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op
     *   * This method cannot be called in the main thread as it may block leading to ANRs
     *
     * @sample samples.core.Sst.Usage.clearCheqUuid
     */
    @JvmSynthetic
    suspend fun clearCheqUuid() {
        clearCheqUuid(contextProvider)
    }

    /**
     * Gets the mobile data collected by the SDK.
     *
     * **Important:**
     *   * [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op
     *   * This method cannot be called in the main thread as it may block leading to ANRs
     *
     * @return A map of the mobile data collected by the SDK.
     *
     * @sample samples.core.Sst.Usage.getMobileData
     */
    @JvmSynthetic
    suspend fun getMobileData(): Map<String, Any> {
        return getMobileData(contextProvider)
    }

    /**
     * Gets the mobile data collected by the SDK.
     *
     * **Important:** [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op.
     *
     * @return A [CompletableFuture] that completes when the operation is done.  If successful, the result is a map of the mobile data collected by the SDK.
     *
     * @suppress
     */
    @JvmStatic
    fun getMobileDataAsync(): CompletableFuture<Map<String, Any>> {
        val ctxProvider = contextProvider
        return config.sstScope.future(config.sstDispatcher) {
            getMobileData(ctxProvider)
        }
    }

    @JvmSynthetic
    internal suspend fun storeCheqUuid(cheqUuid: String) {
        SettingsRepository.storeCheqUuid(contextProvider, cheqUuid)
    }

    private suspend fun getMobileData(contextProvider: ConfigurableContextProvider): Map<String, Any> {
        return config.models.collect(config, contextProvider).toMap()
    }

    private fun trackEventUrl(): URLBuilder {
        val url = URLBuilder(
            URLProtocol.HTTPS,
            config.domain,
            pathSegments = listOf("pc", config.clientName, "sst"),
        )
        url.parameters.append("sstVersion", "1.0.0")
        url.parameters.append("sstOrigin", "mobile")
        return url
    }

    private suspend fun handleError(name: String, function: String, message: String) {
        try {
            var sourceFile = "$function ${BuildConfig.LIBRARY_NAME}:${BuildConfig.LIBRARY_VERSION}"
            try {
                val appModel =
                    config.models.collect<AppModel, AppModel.Data>(config, contextProvider)
                if (appModel != null) {
                    sourceFile += " ${appModel.name}:${appModel.version}"
                }
            } catch (e: CancellationException) {
                throw e
            } catch (_: Exception) {
            }

            val url = URLBuilder(
                URLProtocol.HTTPS, config.nexusHost, pathSegments = listOf("error", "e.gif")
            )
            url.parameters.append("msg", message.truncate(1024))
            url.parameters.append("fn", sourceFile.truncate(256))
            url.parameters.append("client", config.clientName.truncate(256))
            url.parameters.append("publishPath", config.publishPath.truncate(256))
            url.parameters.append("errorName", name.truncate(256))
            url.parameters
            var userAgent: String? = null
            try {
                userAgent = config.virtualBrowser.userAgent
                    ?: config.models.collect<DeviceModel, DeviceModel.Data>(
                        config, contextProvider
                    )?.userAgent
            } catch (e: CancellationException) {
                throw e
            } catch (_: Exception) {
            }
            client.sendHttp<Any>(url, HttpMethod.Get, userAgent = userAgent) {
                header(HttpHeaders.Referrer, trackEventUrl().buildString())
            }
        } catch (e: CancellationException) {
            throw e
        } catch (_: Exception) {
        }
    }

    private suspend fun handleHttpResultError(result: HttpResult.Error.HttpError) {
        handleHttpResultError(
            result::class, "${result.response.status.description}(${result.response.status.value}"
        )
    }

    private suspend fun handleHttpResultError(result: HttpResult.Error.NetworkError) {
        handleHttpResultError(result::class, result.throwable.message ?: "Unknown")
    }

    private suspend fun handleHttpResultError(result: HttpResult.Error.SerializationError) {
        handleHttpResultError(result::class, result.throwable.message ?: "Unknown")
    }

    private suspend fun handleHttpResultError(result: HttpResult.Error.UnknownError) {
        handleHttpResultError(result::class, result.throwable.message ?: "Unknown")
    }

    private suspend fun <T : HttpResult.Error> handleHttpResultError(
        errorClass: KClass<T>, message: String
    ) {
        val errorName = errorClass.simpleName ?: "Error"
        publishError("TrackEvent", errorName, "Sst.trackEvent", message)
    }

    private suspend fun publishConfigureError(function: String? = null, message: String?) {
        publishError(
            "Configure",
            "Sst.configure",
            function ?: "Error",
            message ?: "Unknown"
        )
    }

    private suspend fun publishError(
        title: String, name: String, function: String, message: String
    ) {
        if (config.log.isErrorLoggable()) {
            config.log.e("CHEQ SST $title $name: $message")
        }
        eventBus.publish(Error(name, function, message))
    }

    private fun configureEventBus(config: Config) {
        ErrorMonitor.start()
        if (config.debug) {
            eventBus.enable<HttpRequest>()
            eventBus.enable<HttpResponse>()
        } else {
            eventBus.disable<HttpRequest>()
            eventBus.disable<HttpResponse>()
        }
    }

    private suspend fun clearCheqUuid(contextProvider: ConfigurableContextProvider) {
        return try {
            SettingsRepository.clearCheqUuid(contextProvider)
        } catch (e: CancellationException) {
            throw e
        } catch (_: Exception) {
        }
    }

    private suspend fun getCheqUuid(contextProvider: ConfigurableContextProvider): String? {
        return try {
            SettingsRepository.readCheqUuid(contextProvider)
        } catch (e: CancellationException) {
            throw e
        } catch (_: Exception) {
            null
        }
    }

    private object ErrorMonitor {
        @Volatile private var initialized = false
        private val lock = this

        fun start() {
            if (!initialized) {
                synchronized(lock) {
                    if (!initialized) {
                        config.monitoringScope.launch(CoroutineName("Sst.ErrorMonitor") + config.monitoringDispatcher) {
                            eventBus.subscribe<Error> {
                                handleError(it.name, it.function, it.message)
                            }
                        }
                        initialized = true
                    }
                }
            }
        }
    }
}
