package ai.cheq.sst.android.core.android

import ai.cheq.sst.android.core.ContextProvider
import ai.cheq.sst.android.core.Log
import com.google.android.gms.appset.AppSet
import com.google.android.gms.appset.AppSetIdClient
import kotlinx.coroutines.CancellationException
import kotlinx.coroutines.tasks.await
import kotlinx.coroutines.withTimeoutOrNull
import kotlin.time.Duration.Companion.seconds

internal class AppSetIdProvider {
    private var appSetIdLoader: AppSetIdLoader = NullAppSetIdLoader()

    fun initialize(log: Log, contextProvider: ContextProvider) {
        try {
            appSetIdLoader = AndroidAppSetIdLoader(contextProvider)
        } catch (e: CancellationException) {
            throw e
        } catch (e: Exception) {
            log.e("Failed to initialize AppSetIdProvider", e)
        }
    }

    suspend fun id(): String? {
        return try {
            appSetIdLoader.id()
        } catch (e: CancellationException) {
            throw e
        } catch (e: Exception) {
            println(e.message)
            null
        }
    }

    private interface AppSetIdLoader {
        suspend fun id(): String?
    }

    private class NullAppSetIdLoader : AppSetIdLoader {
        override suspend fun id(): String? {
            return null
        }
    }

    private class AndroidAppSetIdLoader(contextProvider: ContextProvider) : AppSetIdLoader {
        private var appSetIdClient: AppSetIdClient =
            AppSet.getClient(contextProvider.context())

        override suspend fun id(): String? {
            return withTimeoutOrNull(30.seconds) {
                appSetIdClient.appSetIdInfo.continueWith {
                    if (it.isSuccessful) {
                        it.result?.id
                    } else {
                        null
                    }
                }.await()
            }
        }
    }
}