package ai.cheq.sst.android.core.android

import ai.cheq.sst.android.core.ContextProvider
import ai.cheq.sst.android.core.Log
import ai.cheq.sst.android.core.models.Constants
import android.content.Context
import android.content.res.Configuration
import android.hardware.display.DisplayManager
import android.os.Build
import android.view.Display
import android.view.Surface
import kotlinx.coroutines.CancellationException

internal class OrientationProvider {
    private var orientationLoader: OrientationLoader = NullOrientationLoader()

    fun initialize(log: Log, contextProvider: ContextProvider) {
        try {
            orientationLoader = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
                AndroidOrientationLoaderR(contextProvider)
            } else {
                AndroidOrientationLoader(contextProvider)
            }
        } catch (e: CancellationException) {
            throw e
        } catch (e: Exception) {
            log.e("Failed to initialize AppSetIdProvider", e)
        }
    }

    fun orientation(): String? {
        return try {
            orientationLoader.orientation()
        } catch (e: Exception) {
            null
        }
    }

    private interface OrientationLoader {
        fun orientation(): String?
    }

    private class NullOrientationLoader : OrientationLoader {
        override fun orientation(): String? {
            return null
        }
    }

    private class AndroidOrientationLoader(contextProvider: ContextProvider) :
        OrientationLoader {
        private val resources = contextProvider.context().resources

        override fun orientation(): String {
            return when (resources.configuration.orientation) {
                Configuration.ORIENTATION_PORTRAIT -> Constants.SCREEN_ORIENTATION_PORTRAIT
                Configuration.ORIENTATION_LANDSCAPE -> Constants.SCREEN_ORIENTATION_LANDSCAPE
                else -> Constants.UNKNOWN
            }
        }
    }

    private class AndroidOrientationLoaderR(contextProvider: ContextProvider) :
        OrientationLoader {
        private val displayManager =
            contextProvider.context().getSystemService(Context.DISPLAY_SERVICE) as DisplayManager

        override fun orientation(): String {
            return when (displayManager.getDisplay(Display.DEFAULT_DISPLAY)?.rotation ?: 0) {
                Surface.ROTATION_0 -> Constants.SCREEN_ORIENTATION_PORTRAIT
                Surface.ROTATION_90 -> Constants.SCREEN_ORIENTATION_LANDSCAPE_RIGHT
                Surface.ROTATION_180 -> Constants.SCREEN_ORIENTATION_PORTRAIT_UPSIDE_DOWN
                Surface.ROTATION_270 -> Constants.SCREEN_ORIENTATION_LANDSCAPE_LEFT
                else -> Constants.UNKNOWN
            }
        }
    }
}