package ai.cheq.sst.android.core.android

import ai.cheq.sst.android.core.ContextProvider
import ai.cheq.sst.android.core.Log
import android.content.Context
import android.graphics.Point
import android.os.Build
import android.view.WindowManager
import kotlinx.coroutines.CancellationException

internal class ResolutionProvider {
    private var resolutionLoader: ResolutionLoader = NullResolutionLoader()

    fun initialize(log: Log, contextProvider: ContextProvider) {
        try {
            resolutionLoader = AndroidResolutionLoader(contextProvider)
        } catch (e: CancellationException) {
            throw e
        } catch (e: Exception) {
            log.e("Failed to initialize AppSetIdProvider", e)
        }
    }

    fun resolution(): Pair<Int, Int>? {
        return try {
            resolutionLoader.resolution()
        } catch (e: Exception) {
            null
        }
    }

    private interface ResolutionLoader {
        fun resolution(): Pair<Int, Int>?
    }

    private class NullResolutionLoader : ResolutionLoader {
        override fun resolution(): Pair<Int, Int>? {
            return null
        }
    }

    private class AndroidResolutionLoader(contextProvider: ContextProvider) : ResolutionLoader {
        private var windowManager: WindowManager = contextProvider.context()
            .getSystemService(Context.WINDOW_SERVICE) as WindowManager

        override fun resolution(): Pair<Int, Int> = when (Build.VERSION.SDK_INT) {
            in Build.VERSION_CODES.R..Int.MAX_VALUE -> Pair(
                windowManager.currentWindowMetrics.bounds.width(),
                windowManager.currentWindowMetrics.bounds.height()
            )

            else -> {
                val size = Point()
                @Suppress("DEPRECATION") windowManager.defaultDisplay.getSize(size)
                Pair(size.x, size.y)
            }
        }
    }
}
