package ai.cheq.sst.android.core.internal

import ai.cheq.sst.android.core.BuildConfig
import ai.cheq.sst.android.core.Config
import ai.cheq.sst.android.core.Utils
import ai.cheq.sst.android.core.monitoring.monitoringPlugin
import com.fasterxml.jackson.core.JacksonException
import io.ktor.client.HttpClient
import io.ktor.client.plugins.ClientRequestException
import io.ktor.client.plugins.ServerResponseException
import io.ktor.client.plugins.contentnegotiation.ContentNegotiation
import io.ktor.client.request.HttpRequestBuilder
import io.ktor.client.request.header
import io.ktor.client.request.request
import io.ktor.client.request.setBody
import io.ktor.client.statement.HttpResponse
import io.ktor.http.ContentType
import io.ktor.http.HttpHeaders
import io.ktor.http.HttpMethod
import io.ktor.http.URLBuilder
import io.ktor.http.Url
import io.ktor.http.contentType
import io.ktor.serialization.jackson.JacksonConverter
import kotlinx.coroutines.CancellationException
import kotlinx.io.IOException
import kotlinx.serialization.SerializationException

internal class HttpClient {
    private val client = ConfigurableClient()

    fun configure(config: Config, eventBus: EventBus) {
        client.configure(config, eventBus)
    }

    fun reset() {
        client.reset()
    }

    suspend inline fun <reified T> sendHttp(
        url: URLBuilder,
        method: HttpMethod,
        userAgent: String? = null,
        parameters: Map<String, String> = emptyMap(),
        data: T? = null,
        noinline requestBuilder: (HttpRequestBuilder.() -> Unit)? = null
    ): HttpResult {
        return try {
            if (parameters.isNotEmpty()) {
                parameters.forEach { url.parameters.append(it.key, it.value) }
            }
            val requestUrl = url.build()

            val response = client.request(requestUrl) {
                this.method = method
                header(HttpHeaders.UserAgent, userAgent.let {
                    if (it.isNullOrEmpty()) "${BuildConfig.LIBRARY_NAME}/${BuildConfig.LIBRARY_VERSION}" else it
                })
                if (data != null) {
                    contentType(ContentType.Application.Json)
                    setBody(data)
                }
                if (requestBuilder != null) {
                    requestBuilder()
                }
            }
            if (response == null) {
                return HttpResult.Skipped
            }
            HttpResult.Success(response)
        } catch (e: CancellationException) {
            throw e
        } catch (e: ClientRequestException) {
            HttpResult.Error.HttpError(e.response)
        } catch (e: ServerResponseException) {
            HttpResult.Error.HttpError(e.response)
        } catch (e: Throwable) {
            when (e) {
                is SerializationException -> HttpResult.Error.SerializationError(e)
                is JacksonException -> HttpResult.Error.SerializationError(e)
                is IOException -> HttpResult.Error.NetworkError(e)
                else -> HttpResult.Error.UnknownError(e)
            }
        }
    }

    interface Client {
        suspend fun request(url: Url, block: HttpRequestBuilder.() -> Unit = {}): HttpResponse?
    }

    class ConfigurableClient : Client {
        private var client: Client = NoopClient()

        fun configure(config: Config, eventBus: EventBus) {
            client = KtorClient(config, eventBus)
        }

        fun reset() {
            client = NoopClient()
        }

        override suspend fun request(
            url: Url, block: HttpRequestBuilder.() -> Unit
        ): HttpResponse? {
            return client.request(url, block)
        }

        class NoopClient : Client {
            override suspend fun request(
                url: Url, block: HttpRequestBuilder.() -> Unit
            ): HttpResponse? {
                return null
            }
        }

        class KtorClient(config: Config, eventBus: EventBus) : Client {
            private val client = HttpClient(config.httpClientEngine) {
                install(ContentNegotiation) {
                    register(ContentType.Application.Json, JacksonConverter(Utils.jsonMapper))
                }
                install(monitoringPlugin(config.log)) {
                    this.eventBus = eventBus
                    this.monitoringScope = config.monitoringScope
                    this.monitoringDispatcher = config.monitoringDispatcher
                }
            }

            override suspend fun request(
                url: Url, block: HttpRequestBuilder.() -> Unit
            ): HttpResponse {
                return client.request(url, block)
            }
        }
    }
}