package ai.cheq.sst.android.core.internal.store

import ai.cheq.sst.android.core.Log
import ai.cheq.sst.android.core.Utils
import ai.cheq.sst.android.core.internal.ConfigurableContextProvider
import ai.cheq.sst.android.core.internal.EventBus
import ai.cheq.sst.android.core.monitoring.Error
import android.content.Context
import androidx.datastore.core.DataStore
import com.google.protobuf.GeneratedMessageLite
import kotlinx.coroutines.CancellationException
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.coroutines.future.future
import java.util.concurrent.CompletableFuture

internal class ProtobufDataStoreCollection<T : GeneratedMessageLite<S, B>, S : T, B : GeneratedMessageLite.Builder<S, B>> internal constructor(
    private val config: Config<T, S, B>,
    private val contextProvider: ConfigurableContextProvider,
    private val log: Log,
    private val eventBus: EventBus,
    private val coroutineScope: CoroutineScope,
    private val coroutineDispatcher: CoroutineDispatcher
) : DataStoreCollection {
    internal class Config<T : GeneratedMessageLite<S, B>, S : T, B : GeneratedMessageLite.Builder<S, B>>(
        val type: String,
        val dataStoreFunction: (Context) -> DataStore<S>,
        val putFunction: (B, String, String) -> B,
        val clearFunction: (B) -> B,
        val removeFunction: (B, String) -> B,
        val mapFunction: (S) -> MutableMap<String, String>?,
        val defaultFunction: () -> S
    )

    override fun <T : Any> addAsync(key: String, value: T): CompletableFuture<Void?> {
        return contextProvider.context()?.let {
            return coroutineScope.future(coroutineDispatcher) {
                add(it, key, value)
                null
            }
        } ?: CompletableFuture.completedFuture(null)
    }

    @JvmSynthetic
    override suspend fun <T : Any> add(key: String, value: T) {
        return contextProvider.context()?.let {
            add(it, key, value)
        } ?: Unit
    }

    override fun allAsync(): CompletableFuture<Map<String, Any?>> {
        return contextProvider.context()?.let {
            return coroutineScope.future(coroutineDispatcher) {
                all(it)
            }
        } ?: CompletableFuture.completedFuture(emptyMap())
    }

    @JvmSynthetic
    override suspend fun all(): Map<String, Any?> {
        return contextProvider.context()?.let {
            all(it)
        } ?: emptyMap()
    }

    override fun clearAsync(): CompletableFuture<Void?> {
        return contextProvider.context()?.let {
            return coroutineScope.future(coroutineDispatcher) {
                clear(it)
                null
            }
        } ?: CompletableFuture.completedFuture(null)
    }

    @JvmSynthetic
    override suspend fun clear() {
        return contextProvider.context()?.let {
            clear(it)
        } ?: Unit
    }

    override fun containsAsync(key: String): CompletableFuture<Boolean> {
        return contextProvider.context()?.let {
            return coroutineScope.future(coroutineDispatcher) {
                contains(it, key)
            }
        } ?: CompletableFuture.completedFuture(false)
    }

    @JvmSynthetic
    override suspend fun contains(key: String): Boolean {
        return contextProvider.context()?.let {
            contains(it, key)
        } ?: false
    }

    override fun <T : Any> getAsync(key: String, valueClass: Class<T>): CompletableFuture<T?> {
        return contextProvider.context()?.let { context ->
            return dataMapAsync(context).thenCompose<T?> {
                CompletableFuture.completedFuture(deserialize(key, it?.get(key), valueClass))
            }
        } ?: CompletableFuture.completedFuture(null)
    }

    @JvmSynthetic
    override suspend fun <T : Any> get(key: String, valueClass: Class<T>): T? {
        return contextProvider.context()?.let {
            dataMap(it)?.get(key)?.let { valueSerialized ->
                deserialize(key, valueSerialized, valueClass)
            }
        }
    }

    override fun removeAsync(key: String): CompletableFuture<Boolean> {
        return contextProvider.context()?.let { context ->
            return containsAsync(key).thenComposeAsync {
                if (!it) {
                    CompletableFuture.completedFuture(false)
                } else {
                    removeAsync(context, key)
                }
            }
        } ?: CompletableFuture.completedFuture(false)
    }

    @JvmSynthetic
    override suspend fun remove(key: String): Boolean {
        return contextProvider.context()?.let {
            remove(it, key)
        } ?: false
    }

    @JvmSynthetic
    override suspend fun raw(): Map<String, String?> {
        return contextProvider.context()?.let {
            raw(it)
        } ?: emptyMap()
    }

    private suspend fun <T : Any> add(context: Context, key: String, value: T) {
        try {
            config.dataStoreFunction(context).updateData {
                config.putFunction(it.toBuilder(), key, serialize(value)).build()
            }
        } catch (e: CancellationException) {
            throw e
        } catch (e: Exception) {
            eventBus.publish(
                Error(
                    "SerializationError", "Sst.${config.type}.add", "Key $key, details ${e.message}"
                )
            )
        }
    }

    private suspend fun all(context: Context): Map<String, Any?> {
        return raw(context).mapValues { entry ->
            try {
                deserialize(entry.key, entry.value, Any::class.java)
            } catch (e: CancellationException) {
                throw e
            } catch (e: Exception) {
                if (log.isErrorLoggable()) {
                    log.e("CHEQ SST failed to get ${entry.key} from ${config.type}: ${e.message}")
                }
                null
            }
        }
    }

    private suspend fun clear(context: Context) {
        config.dataStoreFunction(context).updateData {
            config.clearFunction(it.toBuilder()).build()
        }
    }

    private suspend fun contains(context: Context, key: String): Boolean {
        return dataMap(context)?.containsKey(key) ?: false
    }

    private fun removeAsync(context: Context, key: String): CompletableFuture<Boolean> {
        return coroutineScope.future(coroutineDispatcher) {
            remove(context, key)
        }
    }

    private suspend fun remove(context: Context, key: String): Boolean {
        if (!contains(context, key)) {
            return false
        }
        config.dataStoreFunction(context).updateData {
            config.removeFunction(it.toBuilder(), key).build()
        }
        return true
    }

    private suspend fun raw(context: Context): Map<String, String?> {
        return dataMap(context) ?: emptyMap()
    }

    private fun dataMapAsync(context: Context): CompletableFuture<Map<String, String>?> {
        return coroutineScope.future(coroutineDispatcher) {
            dataMap(context)
        }
    }

    private suspend fun dataMap(context: Context): Map<String, String>? {
        val dataLayer =
            config.dataStoreFunction(context).data.firstOrNull() ?: config.defaultFunction()
        return config.mapFunction(dataLayer)?.toMap()
    }

    private fun serialize(value: Any): String = Utils.jsonMapper.writeValueAsString(value)

    private fun <T : Any> deserialize(
        key: String, valueSerialized: String?, valueClass: Class<T>
    ): T? {
        return valueSerialized?.let {
            try {
                Utils.jsonMapper.readValue(it, valueClass) ?: null
            } catch (e: Exception) {
                if (log.isErrorLoggable()) {
                    log.e("CHEQ SST failed to get $key from ${config.type}: ${e.message}")
                }
                null
            }
        }
    }
}