package ai.cheq.sst.android.core.models

import ai.cheq.sst.android.core.BuildConfig
import ai.cheq.sst.android.core.ContextProvider
import ai.cheq.sst.android.core.android.AppSetIdProvider
import ai.cheq.sst.android.core.android.OrientationProvider
import ai.cheq.sst.android.core.android.ResolutionProvider
import ai.cheq.sst.android.core.android.UserAgentProvider
import ai.cheq.sst.android.core.models.DeviceModel.Data.Os
import ai.cheq.sst.android.core.models.DeviceModel.Data.Screen
import ai.cheq.sst.android.core.settings.SettingsRepository
import android.os.Build
import com.fasterxml.jackson.annotation.JsonAutoDetect
import com.fasterxml.jackson.annotation.JsonCreator
import com.fasterxml.jackson.annotation.JsonInclude
import com.fasterxml.jackson.annotation.JsonProperty
import kotlinx.coroutines.CancellationException
import kotlinx.coroutines.Job
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.cancellable
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach

/**
 * Represents the device model that collects and exposes device-related data.
 *
 * @sample samples.core.DeviceModel.Usage.defaultConfiguration
 * @sample samples.core.DeviceModel.Usage.customConfiguration
 */
class DeviceModel private constructor(val settings: Settings) :
    Model<DeviceModel.Data>(Data::class, Constants.MODEL_KEY_DEVICE, BuildConfig.LIBRARY_VERSION) {
    private lateinit var builder: Builder
    private val resolutionProvider = ResolutionProvider()
    private val orientationProvider = OrientationProvider()
    private val appSetIdProvider = AppSetIdProvider()
    private val userAgentProvider = UserAgentProvider()

    companion object {
        /**
         * Creates an instance of the device model with the default configuration.
         */
        @JvmStatic
        @JvmName("defaultModel")
        fun default() = DeviceModel(Config())

        /**
         * Creates an instance of the device model with a custom configuration.
         */
        @JvmStatic
        @JvmName("customModel")
        fun custom(): Config = Config()
    }

    /**
     * The settings of the device model.
     *
     * @property screenEnabled Indicates if screen data collection is enabled.
     * @property osEnabled Indicates if OS data collection is enabled.
     * @property idEnabled Indicates if ID data collection is enabled.
     */
    interface Settings {
        val screenEnabled: Boolean
        val osEnabled: Boolean
        val idEnabled: Boolean
    }

    /**
     * Configuration class for customizing the device model. All properties are enabled by default.
     *
     * @property screenEnabled Indicates if screen data collection is enabled.
     * @property osEnabled Indicates if OS data collection is enabled.
     * @property idEnabled Indicates if ID data collection is enabled.
     */
    class Config : Settings {
        override var screenEnabled: Boolean = true
            private set
        override var osEnabled: Boolean = true
            private set
        override var idEnabled: Boolean = true
            private set

        /**
         * Disables screen data collection.
         */
        fun disableScreen(): Config {
            screenEnabled = false
            return this
        }

        /**
         * Disables OS data collection.
         */
        fun disableOs(): Config {
            osEnabled = false
            return this
        }

        /**
         * Disables ID data collection.
         */
        fun disableId(): Config {
            idEnabled = false
            return this
        }

        /**
         * Creates a new instance of the device model with the custom configuration.
         */
        fun create(): DeviceModel {
            return DeviceModel(this)
        }
    }

    override fun modelType(): ModelType = ModelType.DEFAULT

    override suspend fun initialize(modelContext: ModelContext) {
        super.initialize(modelContext)
        if (!this::builder.isInitialized) {
            builder = Builder(
                modelContext,
                settings,
                resolutionProvider,
                orientationProvider,
                appSetIdProvider,
                userAgentProvider
            )
        }
        builder.initialize(modelContext)
    }

    /**
     * Retrieves the data associated with this model.
     *
     * @param modelContext The context in which the data is being retrieved.
     * @return The data associated with the model.
     */
    override suspend fun get(modelContext: ModelContext): Data {
        return builder.build(modelContext.contextProvider)
    }

    private class Builder(
        private val modelContext: ModelContext,
        private val settings: Settings,
        private val resolutionProvider: ResolutionProvider,
        private val orientationProvider: OrientationProvider,
        private val appSetIdProvider: AppSetIdProvider,
        private val userAgentProvider: UserAgentProvider
    ) {
        private lateinit var appSetIdFlow: Flow<String?>
        private lateinit var appSetIdJob: Job

        private val screen: Screen?
            get() {
                val resolution = resolutionProvider.resolution()
                val orientation = orientationProvider.orientation()
                if (resolution == null && orientation == null) {
                    return null
                }
                return Screen(resolution?.first, resolution?.second, orientation)
            }

        private val os: Os
            get() = Os(Constants.ANDROID, Build.VERSION.RELEASE)

        private suspend fun id(contextProvider: ContextProvider): String? {
            try {
                val deviceId = SettingsRepository.readDeviceUuid(contextProvider)
                if (deviceId != null) {
                    return deviceId
                }
                if (!this::appSetIdFlow.isInitialized) {
                    appSetIdFlow = flow {
                        emit(appSetIdProvider.id())
                    }.cancellable()
                }
                appSetIdJob = appSetIdFlow.onEach {
                    if (it != null) {
                        SettingsRepository.storeDeviceUuid(contextProvider, it)
                    }
                }.launchIn(modelContext.modelScope)
                return null
            } catch (e: CancellationException) {
                throw e
            } catch (e: Exception) {
                modelContext.log.e("Failed to retrieve device id", e)
                return Constants.UNKNOWN
            }
        }

        fun initialize(modelContext: ModelContext) {
            resolutionProvider.initialize(modelContext.log, modelContext.contextProvider)
            orientationProvider.initialize(modelContext.log, modelContext.contextProvider)
            appSetIdProvider.initialize(modelContext.log, modelContext.contextProvider)
            userAgentProvider.initialize(modelContext.log, modelContext.contextProvider)
        }

        suspend fun build(contextProvider: ContextProvider): Data {
            val screen = if (settings.screenEnabled) screen else null
            val os = if (settings.osEnabled) os else null
            val id = if (settings.idEnabled) id(contextProvider) else null
            return Data(
                id,
                Build.MANUFACTURER,
                Build.MODEL,
                System.getProperty("os.arch") ?: Constants.UNKNOWN,
                screen,
                os,
                userAgentProvider.userAgent()
            )
        }
    }

    /**
     * Data class representing the device model data.
     *
     * This class holds various information about a device and its environment,
     * including device ID, screen details, and OS information.
     *
     * @property id The unique identifier of the device.
     * @property screen Information about the device's screen (see [Screen]).
     * @property manufacturer The device manufacturer (e.g., "Samsung").
     * @property model The device model (e.g., "Galaxy S21").
     * @property architecture The device's CPU architecture (e.g., "arm64-v8a").
     * @property os Information about the operating system (see [Os]).
     */
    class Data internal constructor(
        @get:JsonInclude(JsonInclude.Include.NON_NULL) @param:JsonProperty("id") @get:JsonProperty("id") val id: String?,
        @get:JsonInclude(JsonInclude.Include.NON_NULL) @param:JsonProperty("manufacturer") @get:JsonProperty(
            "manufacturer"
        ) val manufacturer: String,
        @get:JsonInclude(JsonInclude.Include.NON_NULL) @param:JsonProperty("model") @get:JsonProperty(
            "model"
        ) val model: String,
        @get:JsonInclude(JsonInclude.Include.NON_NULL) @param:JsonProperty("architecture") @get:JsonProperty(
            "architecture"
        ) val architecture: String,
        @get:JsonInclude(JsonInclude.Include.NON_NULL) @param:JsonProperty("screen") @get:JsonProperty(
            "screen"
        ) val screen: Screen?,
        @get:JsonInclude(JsonInclude.Include.NON_NULL) @param:JsonProperty("os") @get:JsonProperty("os") val os: Os?
    ) : Model.Data() {
        internal constructor(
            id: String?,
            manufacturer: String,
            model: String,
            architecture: String,
            screen: Screen?,
            os: Os?,
            userAgent: String
        ) : this(id, manufacturer, model, architecture, screen, os) {
            this.userAgent = userAgent
        }

        internal var userAgent: String? = null; private set

        /**
         * Represents information about the operating system.
         *
         * @property name The name of the operating system (always "Android").
         * @property version The version of the operating system (e.g., "12").
         */
        @JsonAutoDetect(
            fieldVisibility = JsonAutoDetect.Visibility.NONE,
            setterVisibility = JsonAutoDetect.Visibility.NONE,
            getterVisibility = JsonAutoDetect.Visibility.NONE,
            isGetterVisibility = JsonAutoDetect.Visibility.NONE,
            creatorVisibility = JsonAutoDetect.Visibility.NONE
        )
        class Os @JsonCreator internal constructor(
            @param:JsonProperty("name") @get:JsonProperty("name") val name: String,
            @param:JsonProperty("version") @get:JsonProperty("version") val version: String
        )

        /**
         * Represents information about the device's screen.
         *
         * @property width The width of the screen in pixels.
         * @property height The height of the screen in pixels.
         * @property orientation The screen orientation (e.g., "portrait" or "landscape").
         * @property depth The color depth of the screen (always 24).
         */
        @JsonAutoDetect(
            fieldVisibility = JsonAutoDetect.Visibility.NONE,
            setterVisibility = JsonAutoDetect.Visibility.NONE,
            getterVisibility = JsonAutoDetect.Visibility.NONE,
            isGetterVisibility = JsonAutoDetect.Visibility.NONE,
            creatorVisibility = JsonAutoDetect.Visibility.NONE
        )
        @ConsistentCopyVisibility
        data class Screen @JsonCreator internal constructor(
            @get:JsonInclude(JsonInclude.Include.NON_NULL) @param:JsonProperty("width") @get:JsonProperty(
                "width"
            ) val width: Int?,
            @get:JsonInclude(JsonInclude.Include.NON_NULL) @param:JsonProperty("height") @get:JsonProperty(
                "height"
            ) val height: Int?,
            @get:JsonInclude(JsonInclude.Include.NON_NULL) @param:JsonProperty("orientation") @get:JsonProperty(
                "orientation"
            ) val orientation: String?,
        ) {
            val depth: Int = 24
        }
    }
}
