package ai.cheq.sst.android.core.models

import ai.cheq.sst.android.core.Utils
import ai.cheq.sst.android.core.internal.MapTypeReference
import com.fasterxml.jackson.annotation.JsonAutoDetect
import kotlinx.coroutines.CancellationException
import kotlin.reflect.KClass

/**
 * The base class which provides a common structure for models that collect and expose data structures.
 *
 * @param T The type of data associated with this model. Must inherit from [Model.Data].
 * @property dataClass The class representing the data structure for this model.
 */
abstract class Model<out T : Model.Data> {
    internal val dataClass: Class<out T>
    internal val identifier: ModelIdentifier
    internal var enabled = true

    /**
     * Creates a new instance of the model with the specified key, version, and data class.
     *
     * @param dataClass The class representing the data structure for this model.
     * @param key A unique key identifying the model.
     * @param version The version of the model.
     * @throws IllegalArgumentException If the key or version is blank or only whitespace.
     */
    @Throws(IllegalArgumentException::class)
    protected constructor(
        dataClass: Class<out T>, key: String, version: String = ModelIdentifier.DEFAULT_VERSION
    ) {
        this.dataClass = dataClass
        this.identifier = ModelIdentifier(key, version)
    }

    /**
     * Creates a new instance of the model with the specified key, version, and data class.
     *
     * @param key A unique key identifying the model.
     * @param version The version of the model.
     * @param dataClass The class representing the data structure for this model.
     * @throws IllegalArgumentException If the key or version is blank or only whitespace.
     */
    @Throws(IllegalArgumentException::class)
    protected constructor(
        dataClass: KClass<out T>, key: String, version: String = ModelIdentifier.DEFAULT_VERSION
    ) : this(
        dataClass.java, key, version
    )

    internal open fun modelType() = ModelType.STANDARD

    internal suspend fun tryInitialize(modelContext: ModelContext): ModelResult<Unit> {
        try {
            initialize(modelContext)
            return ModelResult(this.identifier)
        } catch (e: CancellationException) {
            throw e
        } catch (e: Exception) {
            return ModelResult(this.identifier, exception = e)
        }
    }

    internal suspend fun tryGet(modelContext: ModelContext): ModelResult<T> {
        try {
            val data = get(modelContext)
            return ModelResult(this.identifier, data)
        } catch (e: CancellationException) {
            throw e
        } catch (e: Exception) {
            return ModelResult(this.identifier, exception = e)
        }
    }

    internal suspend fun tryStop(modelContext: ModelContext): ModelResult<Unit> {
        try {
            stop(modelContext)
            return ModelResult(this.identifier)
        } catch (e: CancellationException) {
            throw e
        } catch (e: Exception) {
            return ModelResult(this.identifier, exception = e)
        }
    }

    /**
     * Sets up the model with the specified [ModelContext].
     *
     * This method can be overridden by subclasses to perform initialization or setup tasks
     * specific to the model.
     *
     * @param modelContext The context in which the model is being set up.
     */
    open suspend fun initialize(modelContext: ModelContext) {
    }

    /**
     * Retrieves the data associated with this model.
     *
     * This method must be implemented by subclasses to provide the logic for fetching
     * the model data.
     *
     * @param modelContext The context in which the data is being retrieved.
     * @return The data associated with the model, or null if no data is found.
     */
    abstract suspend fun get(modelContext: ModelContext): T?

    /**
     * Stops the model and releases any resources it holds.
     *
     * This method can be overridden by subclasses to perform cleanup tasks specific to the model.
     *
     * @param modelContext The context in which the data is being retrieved.
     */
    open suspend fun stop(modelContext: ModelContext) {
    }

    /**
     * An abstract base class for representing model data.
     *
     * This class provides a common structure for data associated with models.
     * It includes a utility function for converting the data to a map representation.
     */
    @JsonAutoDetect(
        fieldVisibility = JsonAutoDetect.Visibility.NONE,
        setterVisibility = JsonAutoDetect.Visibility.NONE,
        getterVisibility = JsonAutoDetect.Visibility.NONE,
        isGetterVisibility = JsonAutoDetect.Visibility.NONE,
        creatorVisibility = JsonAutoDetect.Visibility.NONE
    )
    abstract class Data {
        /**
         * Converts the data object to a map representation.
         *
         * @return A map containing the data properties and their values.
         */
        internal fun toMap(): Map<String, Any> {
            return Utils.jsonMapper.convertValue(this, MapTypeReference.INSTANCE)
        }
    }
}