package ai.cheq.sst.android.core.storage

import ai.cheq.sst.android.core.Log
import ai.cheq.sst.android.core.Sst
import ai.cheq.sst.android.core.internal.ConfigurableContextProvider
import ai.cheq.sst.android.core.internal.EventBus
import ai.cheq.sst.android.core.internal.store.ProtobufDataStoreCollection
import ai.cheq.sst.android.core.serializers.storageDataStore
import ai.cheq.sst.android.protobuf.storage.Storage
import android.content.Context
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineScope
import java.util.concurrent.CompletableFuture

/**
 * A collection of [Cookie] objects that can be stored and retrieved from persistent storage.
 */
class Cookies private constructor(private val layer: DataStoreStorageCollection<Cookie>) :
    StorageCollection<Cookie> by layer {
    /**
     * Associates the specified [value] to the storage layer.
     *
     * **Important:** [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op.
     *
     * @param name The name of the cookie.
     * @param value The value of the cookie.
     * @return A [CompletableFuture] that completes when the operation is done.
     **
     * @suppress
     */
    fun addAsync(name: String, value: String): CompletableFuture<Void?> {
        return addAsync(Cookie(name, value))
    }

    /**
     * Associates the specified [value] to the storage layer.
     *
     * **Important:**
     *   * [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op
     *   * This method cannot be called in the main thread as it may block leading to ANRs.
     *
     * @param name The name of the cookie.
     * @param value The value of the cookie.
     *
     * @sample samples.core.StorageLayer.Usage.add
     */
    @JvmSynthetic
    suspend fun add(name: String, value: String) {
        return add(Cookie(name, value))
    }

    override fun allAsync(): CompletableFuture<Map<String, Any?>> {
        return layer.allAsync()
    }

    @JvmSynthetic
    internal suspend fun raw(): List<String> {
        return layer.raw()
    }

    internal fun configure(
        contextProvider: ConfigurableContextProvider,
        log: Log,
        eventBus: EventBus,
        coroutineScope: CoroutineScope,
        coroutineDispatcher: CoroutineDispatcher
    ) {
        layer.configure(contextProvider, log, eventBus, coroutineScope, coroutineDispatcher)
    }

    companion object {
        internal fun create(): Cookies {
            return Cookies(
                DataStoreStorageCollection.create<Cookie>(
                    ProtobufDataStoreCollection.Config<Storage, Storage, Storage.Builder>(
                        "storage.cookies",
                        Context::storageDataStore,
                        Storage.Builder::putCookies,
                        Storage.Builder::clearCookies,
                        Storage.Builder::removeCookies,
                        Storage::getCookiesMap,
                        Storage::getDefaultInstance
                    )
                )
            )
        }
    }
}