package ai.cheq.sst.android.core.storage

import ai.cheq.sst.android.core.Log
import ai.cheq.sst.android.core.Sst
import ai.cheq.sst.android.core.internal.ConfigurableContextProvider
import ai.cheq.sst.android.core.internal.EventBus
import ai.cheq.sst.android.core.internal.store.ProtobufDataStoreCollection
import ai.cheq.sst.android.core.serializers.storageDataStore
import ai.cheq.sst.android.protobuf.storage.Storage
import android.content.Context
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineScope
import java.util.concurrent.CompletableFuture

/**
 * A collection of session storage [Item] objects that can be stored and retrieved from persistent storage.
 */
class SessionStorage private constructor(private val layer: DataStoreStorageCollection<Item>) :
    StorageCollection<Item> by layer {
    /**
     * Associates the specified [value] to the storage layer.
     *
     * **Important:** [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op.
     *
     * @param value The storage item.
     * @return A [CompletableFuture] that completes when the operation is done.
     *
     * @suppress
     */
    override fun addAsync(value: Item): CompletableFuture<Void?> {
        return layer.addAsync(value)
    }

    /**
     * Associates the specified [value] to the storage layer.
     *
     * **Important:**
     *   * [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op
     *   * This method cannot be called in the main thread as it may block leading to ANRs.
     *
     * @param key The key of the session storage item.
     * @param value The value of session storage item.
     *
     * @sample samples.core.StorageLayer.Usage.add
     */
    @JvmSynthetic
    suspend fun add(key: String, value: String) {
        return add(Item(key, value))
    }

    /**
     * Associates the specified [value] to the storage layer.
     *
     * **Important:** [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op.
     *
     * @param key The key of the session storage item.
     * @param value The value of session storage item.
     * @return A [CompletableFuture] that completes when the operation is done.
     *
     * @suppress
     */
    fun addAsync(key: String, value: String): CompletableFuture<Void?> {
        return addAsync(Item(key, value))
    }

    @JvmSynthetic
    internal suspend fun raw(): List<String> {
        return layer.raw()
    }

    internal fun configure(
        contextProvider: ConfigurableContextProvider,
        log: Log,
        eventBus: EventBus,
        coroutineScope: CoroutineScope,
        coroutineDispatcher: CoroutineDispatcher
    ) {
        layer.configure(contextProvider, log, eventBus, coroutineScope, coroutineDispatcher)
    }

    companion object {
        internal fun create(): SessionStorage {
            return SessionStorage(
                DataStoreStorageCollection.create<Item>(
                    ProtobufDataStoreCollection.Config<Storage, Storage, Storage.Builder>(
                        "storage.session",
                        Context::storageDataStore,
                        Storage.Builder::putSession,
                        Storage.Builder::clearSession,
                        Storage.Builder::removeSession,
                        Storage::getSessionMap,
                        Storage::getDefaultInstance
                    )
                )
            )
        }
    }
}