package ai.cheq.sst.android.core.storage

import ai.cheq.sst.android.core.Log
import ai.cheq.sst.android.core.Sst
import ai.cheq.sst.android.core.internal.ConfigurableContextProvider
import ai.cheq.sst.android.core.internal.EventBus
import com.fasterxml.jackson.core.JsonGenerator
import com.fasterxml.jackson.core.JsonParser
import com.fasterxml.jackson.core.JsonToken
import com.fasterxml.jackson.databind.DeserializationContext
import com.fasterxml.jackson.databind.JsonNode
import com.fasterxml.jackson.databind.SerializerProvider
import com.fasterxml.jackson.databind.annotation.JsonDeserialize
import com.fasterxml.jackson.databind.annotation.JsonSerialize
import com.fasterxml.jackson.databind.deser.std.StdDeserializer
import com.fasterxml.jackson.databind.ser.std.StdSerializer
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineScope

/**
 * Provides access to [Cookies], [LocalStorage], and [SessionStorage].
 *
 * The storage is implemented as a singleton, and is accessed using the [ai.cheq.sst.android.core.Sst.storage] property.
 */
class Storage internal constructor() {
    private var contextProvider = ConfigurableContextProvider()
    private var _cookies = Cookies.create()
    private var _local = LocalStorage.create()
    private var _session = SessionStorage.create()

    /**
     * [Cookies] for automatic inclusion in all [ai.cheq.sst.android.core.Sst.trackEvent] calls.
     *
     * **[Sst.configure] should be called before this method is called, otherwise calling this method is a no-op.**
     *
     * @sample samples.core.Storage.Usage.cookies
     */
    val cookies get() = _cookies

    /**
     * [LocalStorage] for automatic inclusion in all [ai.cheq.sst.android.core.Sst.trackEvent] calls.
     *
     * **[Sst.configure] should be called before this method is called, otherwise calling this method is a no-op.**
     *
     * @sample samples.core.Storage.Usage.local
     */
    val local get() = _local

    /**
     * [SessionStorage] for automatic inclusion in all [ai.cheq.sst.android.core.Sst.trackEvent] calls.
     *
     * **[Sst.configure] should be called before this method is called, otherwise calling this method is a no-op.**
     *
     * @sample samples.core.Storage.Usage.session
     */
    val session get() = _session

    internal fun configure(
        contextProvider: ConfigurableContextProvider,
        log: Log,
        eventBus: EventBus,
        coroutineScope: CoroutineScope,
        coroutineDispatcher: CoroutineDispatcher
    ) {
        this.contextProvider.configure(contextProvider)
        _cookies.configure(contextProvider, log, eventBus, coroutineScope, coroutineDispatcher)
        _local.configure(contextProvider, log, eventBus, coroutineScope, coroutineDispatcher)
        _session.configure(contextProvider, log, eventBus, coroutineScope, coroutineDispatcher)
    }

    internal suspend fun raw(): Data {
        return Data(
            _cookies.raw(), _local.raw(), _session.raw()
        )
    }

    @JsonSerialize(using = Data.Serializer::class)
    @JsonDeserialize(using = Data.Deserializer::class)
    internal data class Data(
        val cookies: List<String>, val localStorage: List<String>, val sessionStorage: List<String>
    ) {
        internal class Serializer : StdSerializer<Data>(Data::class.java, false) {
            override fun serialize(
                value: Data?, gen: JsonGenerator?, provider: SerializerProvider?
            ) {
                gen!!.writeStartObject()
                writeList(gen, "cookies", value!!.cookies)
                writeList(gen, "localStorage", value.localStorage)
                writeList(gen, "sessionStorage", value.sessionStorage)
                gen.writeEndObject()
            }

            override fun isEmpty(provider: SerializerProvider?, value: Data?) =
                value == null || (value.cookies.isEmpty() && value.localStorage.isEmpty() && value.sessionStorage.isEmpty())

            private fun writeList(gen: JsonGenerator, key: String, values: List<String>) {
                if (values.isNotEmpty()) {
                    gen.writeFieldName(key)
                    gen.writeStartArray()
                    values.forEach {
                        gen.writeRawValue(it)
                    }
                    gen.writeEndArray()
                }
            }
        }

        internal class Deserializer : StdDeserializer<Data>(Data::class.java) {
            override fun deserialize(
                p: JsonParser?, ctxt: DeserializationContext?
            ): Data {
                val cookies = mutableListOf<String>()
                val localStorage = mutableListOf<String>()
                val sessionStorage = mutableListOf<String>()
                if (p != null && p.codec != null && ctxt != null) {
                    if (p.currentToken == JsonToken.FIELD_NAME) {
                        p.nextToken()
                    }
                    val node: JsonNode? = p.codec.readTree(p)
                    node?.fields()?.forEach {
                        readList(node, it, "cookies", cookies) || readList(
                            node, it, "localStorage", localStorage
                        ) || readList(node, it, "sessionStorage", sessionStorage)
                    }
                }
                return Data(cookies, localStorage, sessionStorage)
            }

            private fun readList(
                parent: JsonNode,
                field: MutableMap.MutableEntry<String, JsonNode>,
                key: String,
                destination: MutableList<String>
            ): Boolean {
                if (field.key != key) {
                    return false
                }
                val node = parent.get(key)
                node.forEach { f ->
                    destination.add(f.toString())
                }
                return true
            }
        }
    }

    internal suspend fun reset(data: Boolean = false) {
        if (data) {
            _cookies.clear()
            _local.clear()
            _session.clear()
        }
        contextProvider = ConfigurableContextProvider()
        _cookies = Cookies.create()
        _local = LocalStorage.create()
        _session = SessionStorage.create()
    }
}
