package ai.cheq.sst.android.core.storage

import ai.cheq.sst.android.core.Sst
import java.util.concurrent.CompletableFuture

/**
 * A generic collection of values that can be stored and retrieved from persistent storage.
 *
 * @param T the type of elements contained in the collection.
 */
interface StorageCollection<T : Identifiable> {
    /**
     * Associates the specified [value] to the storage layer.
     *
     * **Important:** [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op.
     *
     * @param value The value to add.
     * @return A [CompletableFuture] that completes when the operation is done.
     *
     * @suppress
     */
    fun addAsync(value: T): CompletableFuture<Void?>

    /**
     * Associates the specified [value] to the storage layer.
     *
     * **Important:**
     *   * [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op
     *   * This method cannot be called in the main thread as it may block leading to ANRs.
     *
     * @param value The value to add.
     *
     * @sample samples.core.StorageLayer.Usage.add
     */
    @JvmSynthetic
    suspend fun add(value: T)

    /**
     * Returns a map of all key-value pairs in the storage layer.
     *
     * **Important:** [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op.
     *
     * @return A [CompletableFuture] that completes when the operation is done.  If successful, the result is a map of all key-value pairs in the storage layer.
     *
     * @suppress
     */
    fun allAsync(): CompletableFuture<Map<String, Any?>>

    /**
     * Returns a map of all key-value pairs in the storage layer.
     *
     * **Important:**
     *   * [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op
     *   * This method cannot be called in the main thread as it may block leading to ANRs.
     *
     * @return A map of all key-value pairs in the storage layer.
     *
     * @sample samples.core.StorageLayer.Usage.all
     */
    @JvmSynthetic
    suspend fun all(): Map<String, Any?>

    /**
     * Clears all key-value pairs from the storage layer.
     *
     * **Important:** [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op.
     *
     * @return A [CompletableFuture] that completes when the operation is done.
     *
     * @suppress
     */
    fun clearAsync(): CompletableFuture<Void?>

    /**
     * Clears all key-value pairs from the storage layer.
     *
     * **Important:**
     *   * [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op
     *   * This method cannot be called in the main thread as it may block leading to ANRs.
     *
     * @sample samples.core.StorageLayer.Usage.clear
     */
    @JvmSynthetic
    suspend fun clear()

    /**
     * Determines whether the storage layer contains the specified key.
     *
     * **Important:** [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op.
     *
     * @param key The key to check.
     * @return A [CompletableFuture] that completes when the operation is done.  If successful, the result is `true` if the storage layer contains the key, `false` otherwise.
     *
     * @suppress
     */
    fun containsAsync(key: String): CompletableFuture<Boolean>

    /**
     * Determines whether the storage layer contains the specified key.
     *
     * **Important:**
     *   * [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op
     *   * This method cannot be called in the main thread as it may block leading to ANRs.
     *
     * @param key The key to check.
     * @return `true` if the storage layer contains the key, `false` otherwise.
     *
     * @sample samples.core.StorageLayer.Usage.contains
     */
    @JvmSynthetic
    suspend fun contains(key: String): Boolean

    /**
     * Returns the value associated with the specified [key], or `null` if the key is not in the storage layer.
     *
     * **Important:**
     *   * [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op
     *   * This method cannot be called in the main thread as it may block leading to ANRs.
     *
     * @param key The key to get.
     * @return The value associated with the specified key, or `null` if the key is not in the storage layer
     *
     * @sample samples.core.StorageLayer.Usage.get
     */
    @JvmSynthetic
    suspend fun get(key: String): T?

    /**
     * Returns the value associated with the specified [key], or `null` if the key is not in the storage layer.
     *
     * **Important:** [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op.
     *
     * @param key The key to get.
     * @return A [CompletableFuture] that completes when the operation is done.  If successful, the result is the value associated with the specified key, or `null` if the key is not in the storage layer
     *
     * @suppress
     */
    fun getAsync(key: String): CompletableFuture<T?>

    /**
     * Removes the specified key and its corresponding value from the storage layer.
     *
     * **Important:** [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op.
     *
     * @param key The key to remove.
     * @return `true` if the key was removed, `false` if the key was not in the storage layer.
     *
     * @suppress
     */
    fun removeAsync(key: String): CompletableFuture<Boolean>

    /**
     * Removes the specified key and its corresponding value from the storage layer.
     *
     * **Important:** [Sst.configure] should be called before this method is called, otherwise calling this method is a no-op.
     *
     * @param key The key to remove.
     * @return `true` if the key was removed, `false` if the key was not in the storage layer.
     *
     * @sample samples.core.StorageLayer.Usage.remove
     */
    @JvmSynthetic
    suspend fun remove(key: String): Boolean
}