/*
 * Copyright 2021 Rikai authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package ai.eto.rikai.sql.model

import io.circe.syntax._
import scala.collection.JavaConverters.mapAsJavaMap

import org.apache.spark.sql.catalyst.FunctionIdentifier
import org.apache.spark.sql.catalyst.analysis.UnresolvedFunction
import org.apache.spark.sql.catalyst.expressions.Expression

import ai.eto.rikai.sql.spark.SparkRunnable

/**
  * A Machine Learning Model in Rikai Catalog.
  */
trait Model {

  /** Model Name */
  val name: String

  /** Model URI in the registry */
  val uri: String

  /** Model Options. */
  var options: Map[String, String] = Map.empty

  /** Return options as java Map, so that it is easily accessible in Python via py4j. */
  final def javaOptions: java.util.Map[String, String] = mapAsJavaMap(options)

}

object Model {

  /** Model Name Pattern */
  val namePattern = """[a-zA-Z]\w{0,255}""".r

  @throws[ModelNameException]
  def verifyName(name: String): Unit = {
    if (!name.matches(namePattern.regex)) {
      throw new ModelNameException(s"Model name '${name}' is not valid")
    }
  }

  def serializeOptions(options: Map[String, String]): String = {
    options.asJson.asString.getOrElse("")
  }
}

class ModelNameException(message: String) extends Exception(message);

/**
  * A [[Model]] that can be turned into a Spark UDF.
  *
  * @param name model name.
  * @param uri the model uri.
  * @param funcName the name of a UDF which will be called when this model is invoked.
  */
class SparkUDFModel(val name: String, val uri: String, val funcName: String)
    extends Model
    with SparkRunnable {

  override def toString: String = s"SparkUDFModel(name=${name}, uri=${uri})"

  /** Convert a [[Model]] to a Spark Expression in Spark SQL's logical plan. */
  override def asSpark(args: Seq[Expression]): Expression = {
    new UnresolvedFunction(
      new FunctionIdentifier(funcName),
      arguments = args,
      isDistinct = false
    )
  }
}
