package com.ai.osmos.AdFetcherSDK

import android.util.Log
import com.ai.osmos.utils.ConfigManager
import com.ai.osmos.network.WebConstants
import com.ai.osmos.network.WebService
import kotlinx.coroutines.CompletableDeferred
import org.json.JSONException


/**
 * Created by adeshmukh on 07/02/25.
 * Project Name: OSMOS-Android-SDK
 * File Name: AdFetcherSDK
 *
 * This class is responsible for making API calls to fetch different types of ads using
 * defined function signature for display ads with au and display ads with pt.
 */
public class AdFetcherSDK {

    private val webService = WebService()
    private var baseUrl = ""
    private var apiUrl = ""

    /**
     * Fetches display ads using Ad Unit-based targeting.
     *
     * @param cliUbid Unique client identifier
     * @param pageType Type of page requesting ads
     * @param productCount Number of products to display ads for
     * @param adUnits List of ad unit identifiers
     * @param targetingParams Optional: Targeting parameters for ad filtering
     * @param extraParams Optional: Additional parameters if required
     * @return The API response as a Map<String,Any>.
     * ```
     * Example:
     * val res = adFetcherSDK.fetchDisplayAdsWithAu(
     *             cliUbid = "xyz",
     *             pageType = "demo_page",
     *             productCount = 10,
     *             adUnits = arrayListOf("image_ad_inv"),
     *             targetingParams = mapOf("f.keyword" to "potato chips"),
     *             extraParams = mapOf()
     *         )
     * ```
     */
    public suspend fun fetchDisplayAdsWithAu(
        cliUbid: String,
        pageType: String,
        productCount: Int,
        adUnits: List<String>,
        targetingParams: Map<String, Any>? = emptyMap(),
        extraParams: Map<String, Any>? = emptyMap()
    ) : Map<String,Any> {

        if(ConfigManager.isDebugEnabled()) {
            Log.d("OSMOS","OSMOS: cli_ubid = $cliUbid")
            Log.d("OSMOS","OSMOS: page_type = $pageType")
            Log.d("OSMOS","OSMOS: product_count = $productCount")
            Log.d("OSMOS","OSMOS: ad_units = $adUnits")
            Log.d("OSMOS","OSMOS: targeting_params = ${targetingParams.toString()}")
            Log.d("OSMOS","OSMOS: extra_params = ${extraParams.toString()}")
        }

        val deferredResponse = CompletableDeferred<Map<String,Any>>()

        // Construct query parameters
        val queryParams = getQueryParam(
            isDisplayAuAds = true,
            cliUbid = cliUbid,
            pageType = pageType,
            productCount = productCount,
            adUnit = adUnits,
            targetingParams = targetingParams,
            extraParams = extraParams
        )


        // Get API base URL and timeout settings
        try {
            baseUrl = ConfigManager.config.displayAds?.domainUri
                ?: throw JSONException("Display Ads")

            apiUrl = WebConstants.buildUrl("$baseUrl/v2/bsda", queryParams)
            // Make API call to fetch ads
            webService.callApi(apiUrl, "GET", null, null) { response ->
                deferredResponse.complete(response)
            }
        }
        catch(e: JSONException) {
            val error = """
            {
                "status": false,
                "response": null,
                "error": {
                    "code": "ERROR_CONFIGURATION",
                    "description": "Display Ads parameters not configured at time of SDK initialization."
                }
            }
            """
            deferredResponse.complete(emptyMap())
        }

        return deferredResponse.await() // Suspend until response is available
    }

    /**
     * Fetches display ads using Page Type-based targeting.
     *
     * @param cliUbid Unique client identifier
     * @param pageType Type of page requesting ads
     * @param productCount Number of products to display ads for
     * @param targetingParams Optional : Targeting parameters for ad filtering
     * @param extraParams Optional: Additional parameters if required
     * @return The API response as a Map<String,Any>.
     *
     * Example:
     * ```
     * val res = adFetcherSDK.fetchDisplayAdsWithPt(
     *             cliUbid = "92419830283kjzxf2sdfk",
     *             pageType = "demo_page",
     *             productCount = 10,
     *             targetingParams = mapOf("f.keyword" to "potato chips"),
     *             extraParams = mapOf()
     *         )
     * ```
     *
     */
    public suspend fun fetchDisplayAdsWithPt(
        cliUbid: String,
        pageType: String,
        productCount: Int,
        targetingParams: Map<String, Any>? = emptyMap(),
        extraParams: Map<String, Any>? = emptyMap()
    ) : Map<String,Any> {

        if(ConfigManager.isDebugEnabled()) {
            Log.d("OSMOS","OSMOS: cli_ubid = $cliUbid")
            Log.d("OSMOS","OSMOS: page_type = $pageType")
            Log.d("OSMOS","OSMOS: product_count = $productCount")
            Log.d("OSMOS","OSMOS: targeting_params = ${targetingParams.toString()}")
            Log.d("OSMOS","OSMOS: extra_params = ${extraParams.toString()}")
        }

        val deferredResponse = CompletableDeferred<Map<String,Any>>()

        // Construct query parameters (without ad units)
        val queryParams = getQueryParam(
            isDisplayAuAds = false,
            cliUbid = cliUbid,
            pageType = pageType,
            productCount = productCount,
            targetingParams = targetingParams,
            extraParams = extraParams
        )
        try {
            // Get API base URL and timeout settings
            val baseUrl = ConfigManager.config.displayAds?.domainUri
                ?: throw JSONException("Display Ads")

            val apiUrl = WebConstants.buildUrl("$baseUrl/v2/bsda/pt", queryParams)

            // Make API call to fetch ads
            webService.callApi(apiUrl, "GET", null, null) {  response ->
                deferredResponse.complete(response) // Complete with the response
            }

        }
        catch(e: JSONException) {
            val error = """
            {
                "status": false,
                "response": null,
                "error": {
                    "code": "ERROR_CONFIGURATION",
                    "description": "Display Ads parameters not configured at time of SDK initialization."
                }
            }
            """
            Log.e("OSMOS", "OSMOS:$error")
            deferredResponse.complete(emptyMap())
        }
        return deferredResponse.await() // Suspend until response is available
    }

    /**
     * Fetches Product Listing Ads (PLA) based on the provided parameters.
     * Generic function for PLA Ads
     *
     * @param cliUbid Unique identifier for the client.
     * @param pageName Optional name of the page requesting ads. Defaults to an empty string.
     * @param pageType Type of the page (e.g., "SEARCH", "HOME").
     * @param productCount Number of products to display ads for.
     * @param filters Optional: Map containing filter criteria; values can be of any type for flexibility.
     * @param extraParams Optional: Additional parameters for the request; defaults to an empty map.
     * @return The API response as a Map<String,Any>.
     */
    public suspend fun fetchPlaAds(
        cliUbid: String,
        pageName: String? = "",
        pageType: String,
        productCount: Int,
        filters: Map<String, Any>? = emptyMap(),
        extraParams: Map<String, Any>? = emptyMap()
    ): Map<String, Any> {

        if(ConfigManager.isDebugEnabled()) {
            Log.d("OSMOS","OSMOS: cli_ubid = $cliUbid")
            Log.d("OSMOS","OSMOS: page_name = $pageName")
            Log.d("OSMOS","OSMOS: page_type = $pageType")
            Log.d("OSMOS","OSMOS: product_count = $productCount")
            Log.d("OSMOS","OSMOS: filters = ${filters.toString()}")
            Log.d("OSMOS","OSMOS: extra_params = ${extraParams.toString()}")
        }

        val deferredResponse = CompletableDeferred<Map<String,Any>>()

        // Build query parameters from the provided arguments
        val queryParams = buildQueryParams(
            cliUbid = cliUbid,
            pageType = pageType,
            productCount = productCount,
            pageName = pageName,
            filters = filters,
            extraParams = extraParams,
        )

        try {
            // Retrieve base URL and connection timeout for PLA ads
            val baseUrl = ConfigManager.config.plaAds?.domainUri
                ?: throw JSONException("PLA Ads")

            // Build url parameters from the baseUrl and query parameters
            val apiUrl = WebConstants.buildUrl("$baseUrl/sda", queryParams)

            // Make API call to fetch ads
            webService.callApi(apiUrl, "GET", null, null) {  response ->
                deferredResponse.complete(response) // Complete with the response
            }
        }
        catch(e: JSONException) {
            val error = """
            {
                "status": false,
                "response": null,
                "error": {
                    "code": "ERROR_CONFIGURATION",
                    "description": "PLA Ads parameters not configured at time of SDK initialization."
                }
            }
            """
            Log.e("OSMOS", "OSMOS:$error")
            deferredResponse.complete(emptyMap())
        }
        return deferredResponse.await() // Suspend until response is available
    }

    /**
     * Fetches PLA ads specifically for search pages.
     *
     * @param cliUbid Unique identifier for the client.
     * @param pageName Optional: Name of the search results page.
     * @param productCount Number of products to display ads for.
     * @param keywords List of search keywords.
     * @param filters Optional: Map containing filter criteria as key-value pairs.
     * @param extraParams Optional: Additional parameters for the request.
     * @return The API response as a Map<String,Any>.
     *
     * Example:
     * ```
     * val res = adFetcherSDK.fetchPlaSearchPageAds(
     *             cliUbid = "92k31hsd0aish242ljk4sjdas",
     *             pageName = "TOP_OF_SEARCH",
     *             productCount = 10,
     *             keywords = arrayListOf("Sparkling"),
     *             filters = mutableMapOf("store_id" to "Astore"),
     *             extraParams = mutableMapOf(),
     *         )
     * ```
     */
    suspend fun fetchPlaSearchPageAds(
        cliUbid: String,
        pageName: String? = "",
        productCount: Int,
        keywords: List<Any>,
        filters: Map<String, Any>? = emptyMap(),
        extraParams: Map<String, Any>? = emptyMap()
    ): Map<String, Any> {

        val pageType = "SEARCH"

        // Combine provided filters with additional parameters specific to search
        val allFilters = filters?.toMutableMap().apply {
            this?.set("keywords", keywords)
        }

        // Delegate the API call to the general fetch_pla_ads function
        return  fetchPlaAds(cliUbid, pageName, pageType, productCount, allFilters, extraParams)
    }

    /**
     * Fetches Product Listing Ads (PLA) for the home page.
     *
     * @param cliUbid Unique identifier for the client.
     * @param pageName Name of the home page requesting ads.
     * @param productCount Number of products to display ads for.
     * @param filters Optional: Map containing filter criteria; values can be of any type for flexibility.
     * @param extraParams Optional: Additional parameters for the request; defaults to an empty map.
     * @return The API response as a Map<String,Any>.
     *
     * Example:
     * ```
     * val res = adFetcherSDK.fetchPlaHomePageAds(
     *             cliUbid = "92k31hsd0",
     *             pageName = "",
     *             productCount = 20,
     *             filters = mutableMapOf("store_id" to "Astore", "brands" to arrayListOf("Whitakers")),
     *             extraParams = mutableMapOf(),
     *         )
     * ```
     */
    suspend fun fetchPlaHomePageAds(
        cliUbid: String,
        pageName: String? = "",
        productCount: Int,
        filters: Map<String, Any>? = emptyMap(),
        extraParams: Map<String, Any>? = emptyMap()
    ): Map<String, Any> {
        val pageType = "HOME"

        // Delegate the API call to the general fetch_pla_ads function
        return fetchPlaAds(cliUbid, pageName, pageType, productCount, filters, extraParams)
    }

    /**
     * Fetches PLA ads for a category page.
     *
     * @param cliUbid Unique identifier for the client.
     * @param pageName Optional: Name of the category page requesting ads.
     * @param productCount Number of products to display ads for.
     * @param categories List of categories to filter ads.
     * @param filters Optional: Map containing filter criteria as key-value pairs.
     * @param extraParams Optional: Additional parameters for the request; defaults to an empty map.
     * @return The API response as a Map<String,Any>.
     *
     * Example:
     * ```
     * val res = adFetcherSDK.fetchPlaCategoryPageAds(
     *             cliUbid = "B8241C5",
     *             pageName = "category_listing",
     *             productCount = 3,
     *             filters = mutableMapOf("store_id" to "Astore"),
     *             extraParams = mutableMapOf(),
     *             categories = arrayListOf("Kitchen & Pantry"),
     *         )
     * ```
     */
    suspend fun fetchPlaCategoryPageAds(
        cliUbid: String,
        pageName: String? = "",
        productCount: Int,
        categories: List<Any>,
        filters: Map<String, Any>? = emptyMap(),
        extraParams: Map<String, Any>? = emptyMap()
    ): Map<String, Any> {

        val pageType = "CATEGORY"

        // Combine provided filters with category-specific parameters
        val allFilters = filters?.toMutableMap().apply {
            this?.set("categories", categories)
        }
        // Delegate the API call to the general fetch_pla_ads function
        return fetchPlaAds(cliUbid, pageName, pageType, productCount, allFilters, extraParams)
    }

    /**
     * Fetches PLA ads for a product detail page.
     *
     * @param cliUbid Unique identifier for the client.
     * @param pageName Optional: Name of the product page requesting ads.
     * @param productCount Number of products to display ads for.
     * @param skuIds List of SKU identifiers for the products.
     * @param filters Optional: Map containing filter criteria as key-value pairs.
     * @param extraParams Optional: Additional parameters for the request.
     * @return The API response as a Map<String,Any>.
     *
     * Example:
     * ```
     * val res = adFetcherSDK.fetchPlaProductPageAds(
     *             cliUbid = "92k31hsd0",
     *             pageName = "",
     *             productCount = 20,
     *             filters = mutableMapOf("store_id" to "Astore"),
     *             extraParams = mutableMapOf(),
     *             skuIds = arrayListOf("2777534___Astore___Anet"),
     *         )
     * ```
     */
    suspend fun fetchPlaProductPageAds(
        cliUbid: String,
        pageName: String? = "",
        productCount: Int,
        skuIds: List<Any>,
        filters: Map<String, Any>? = emptyMap(),
        extraParams: Map<String, Any>? = emptyMap()
    ): Map<String, Any> {
        val pageType = "PRODUCT"

        // Combine provided filters with product-specific parameters
        val allFilters = filters?.toMutableMap().apply {
            this?.set("sku_ids", skuIds)
        }

        // Delegate the API call to the general fetch_pla_ads function
        return fetchPlaAds(cliUbid, pageName, pageType, productCount, allFilters, extraParams)
    }


    /**
     * Fetches PLA ads for a purchase detail page.
     *
     * @param cliUbid Unique identifier for the client.
     * @param pageName Optional: Name of the product page requesting ads.
     * @param productCount Number of products to display ads for.
     * @param skuIds List of SKU identifiers for the products.
     * @param filters Optional: Map containing filter criteria as key-value pairs.
     * @param extraParams Optional: Additional parameters for the request.
     * @return The API response as a Map<String,Any>.
     *
     * Example:
     * ```
     * val res = adFetcherSDK.fetchPlaPurchasePageAds(
     *             cliUbid = "9bd111c7-6a57-4a90-8ad0-51a5eb98cfc1",
     *             pageName = "",
     *             productCount = 20,
     *             filters = mutableMapOf("store_id" to "Astore"),
     *             extraParams = mutableMapOf(),
     *             skuIds = arrayListOf("2777534___Astore___Anet", "2777515___Astore___Anet"),
     *         )
     * ```
     */
    suspend fun fetchPlaPurchasePageAds(
        cliUbid: String,
        pageName: String? = "",
        productCount: Int,
        skuIds: List<Any>,
        filters: Map<String, Any>? = emptyMap(),
        extraParams: Map<String, Any>? = emptyMap()
    ): Map<String, Any> {
        val pageType = "PURCHASE"

        // Combine provided filters with purchase-specific parameters
        val allFilters = filters?.toMutableMap().apply {
            this?.set("sku_ids", skuIds)
        }

        // Delegate the API call to the general fetch_pla_ads function
        return fetchPlaAds(cliUbid, pageName, pageType, productCount, allFilters, extraParams)
    }

    /**
     * POST API Call for Tagged Product Ads.
     *
     * This function constructs a JSON payload with the provided parameters and sends
     * a POST request to the configured Tagged product ads. The response is logged for
     * debugging purposes.
     *
     * @param cliUbid Unique identifier for the client session.
     * @param pageName Optional: Name of the page requesting ads.
     * @param productCount Number of products to display ads for.
     * @param skuIds List of SKU identifiers related to the products.
     * @param filters Optional: Map containing filter criteria; values can be of any type for flexibility.
     * @param extraParams Optional: Additional parameters for the request; values can be of any type.
     * @return The API response as a Map<String,Any>.
     *
    ]     * Example:
     * ```
     * val res  = adFetcherSDK.fetchTpaPageAds(
     *             cliUbid = "9bd111c7-6a57-4a90-8ad0-51a5eb98cfc1",
     *             pageName = "",
     *             productCount = 5,
     *             filters = mutableMapOf("store_id" to "Astore"),
     *             extraParams = mutableMapOf(),
     *             skuIds = arrayListOf("2777534___Astore___Anet", "2777515___Astore___Anet"),
     *         )
     * ```
     */
    suspend fun fetchTpaPageAds(
        cliUbid: String,
        pageName: String? = "",
        productCount: Int,
        skuIds: List<Any>,
        filters: Map<String, Any>? = emptyMap(),
        extraParams: Map<String, Any>? = emptyMap()
    ): Map<String, Any> {

        if(ConfigManager.isDebugEnabled()) {
            Log.d("OSMOS","OSMOS: cli_ubid = $cliUbid")
            Log.d("OSMOS","OSMOS: page_name = $pageName")
            Log.d("OSMOS","OSMOS: product_count = $productCount")
            Log.d("OSMOS","OSMOS: sku_ids = $skuIds")
            Log.d("OSMOS","OSMOS: filters = ${filters.toString()}")
            Log.d("OSMOS","OSMOS: extra_params = ${extraParams.toString()}")
        }


        val deferredResponse = CompletableDeferred<Map<String,Any>>()

        try {
            val baseUrl = ConfigManager.config.plaAds?.domainUri
                ?: throw JSONException("PLA Ads")

            val clientId: String = ConfigManager.getString("client_id").toString()

            val data: MutableMap<String, Any> = mutableMapOf(
                "cli_ubid" to cliUbid,       // Client unique identifier
                "pcnt" to productCount,      // Product count
                "page_type" to "TPA",         // Page type set to "TPA"
                "client_id" to clientId,     // Client ID
                "sku_ids" to skuIds,         // List of SKU IDs
            ).apply {
                pageName?.let { put("page_name", it) }
                filters?.let { putAll(it) } // Add filters to the data map
                extraParams?.let { putAll(it) } // Add extra_params to the data map
            }

            // Make the API call using the web service
            webService.callApi(
                apiUrl = "$baseUrl/sda/v1",             // The base URL for the API
                methodType = "POST",          // HTTP method type
                requestHeaders = null,    // No custom headers
                requestBody = data        // JSON payload for the request
            ) {  response ->
                deferredResponse.complete(response)
            }
        }
        catch(e: JSONException) {
            val error = """
            {
                "status": false,
                "response": null,
                "error": {
                    "code": "ERROR_CONFIGURATION",
                    "description": "PLA Ads parameters not configured at time of SDK initialization."
                }
            }
            """
            Log.e("OSMOS", "OSMOS:$error")
            deferredResponse.complete(emptyMap())
        }
        return deferredResponse.await()
    }

    /**
     * Fetches display ads using Ad Unit-based targeting.
     *
     * @param cli_ubid Unique client identifier
     * @param page_type Type of page requesting ads
     * @param product_count Number of products to display ads for
     * @param ad_units List of ad unit identifiers
     * @param targeting_params Optional: Targeting parameters for ad filtering
     * @param extra_params Optional: Additional parameters if required
     * @return The API response as a Map<String,Any>.
     * ```
     * Example:
     * val res = adFetcherSDK.fetch_display_ads_with_au(
     *             cli_ubid = "xyz",
     *             page_type = "demo_page",
     *             product_count = 10,
     *             ad_units = arrayListOf("image_ad_inv"),
     *             targeting_params = mapOf("f.keyword" to "potato chips"),
     *             extra_params = mapOf()
     *         )
     * ```
     */
    @Deprecated(
        message = "Use fetchDisplayAdsWithAu instead",
        replaceWith = ReplaceWith("fetchDisplayAdsWithAu(cliUbid, pageType, productCount, adUnits, targetingParams, extraParams)"),
        level = DeprecationLevel.WARNING
    )
    public suspend fun fetch_display_ads_with_au(
        cli_ubid: String,
        page_type: String,
        product_count: Int,
        ad_units: List<String>,
        targeting_params: Map<String, Any>? = emptyMap(),
        extra_params: Map<String, Any>? = emptyMap()
    ) : Map<String,Any> {
        return fetchDisplayAdsWithAu(
            cliUbid = cli_ubid,
            pageType = page_type,
            productCount = product_count,
            adUnits = ad_units,
            targetingParams = targeting_params,
            extraParams = extra_params
        )
    }

    /**
     * Fetches display ads using Page Type-based targeting.
     *
     * @param cli_ubid Unique client identifier
     * @param page_type Type of page requesting ads
     * @param product_count Number of products to display ads for
     * @param targeting_params Optional : Targeting parameters for ad filtering
     * @param extra_params Optional: Additional parameters if required
     * @return The API response as a Map<String,Any>.
     *
     * Example:
     * ```
     * val res = adFetcherSDK.fetch_display_ads_with_pt(
     *             cli_ubid = "92419830283kjzxf2sdfk",
     *             page_type = "demo_page",
     *             product_count = 10,
     *             targeting_params = mapOf("f.keyword" to "potato chips"),
     *             extra_params = mapOf()
     *         )
     * ```
     *
     */
    @Deprecated(
        message = "Use fetchDisplayAdsWithPt instead",
        replaceWith = ReplaceWith("fetchDisplayAdsWithPt(cliUbid, pageName, pageType, productCount, filters, extraParams)"),
        level = DeprecationLevel.WARNING
    )
    public suspend fun fetch_display_ads_with_pt(
        cli_ubid: String,
        page_type: String,
        product_count: Int,
        targeting_params: Map<String, Any>? = emptyMap(),
        extra_params: Map<String, Any>?= emptyMap()
    ) : Map<String,Any> {
        return fetchDisplayAdsWithPt(
            cliUbid = cli_ubid,
            pageType = page_type,
            productCount = product_count,
            targetingParams = targeting_params,
            extraParams = extra_params
        )
    }

    /**
     * Fetches Product Listing Ads (PLA) based on the provided parameters.
     * Generic function for PLA Ads
     *
     * @param cli_ubid Unique identifier for the client.
     * @param page_name Optional name of the page requesting ads. Defaults to an empty string.
     * @param page_type Type of the page (e.g., "SEARCH", "HOME").
     * @param product_count Number of products to display ads for.
     * @param filters Optional: Map containing filter criteria; values can be of any type for flexibility.
     * @param extra_params Optional: Additional parameters for the request; defaults to an empty map.
     * @return The API response as a Map<String,Any>.
     */
    @Deprecated(
        message = "Use fetchPlaAds instead",
        replaceWith = ReplaceWith("fetchPlaAds(cliUbid, pageName, pageType, productCount, filters, extraParams)"),
        level = DeprecationLevel.WARNING
    )
    private suspend fun fetch_pla_ads(
        cli_ubid: String,
        page_name: String? = "",
        page_type: String,
        product_count: Int,
        filters: Map<String, Any> ? = emptyMap(),  // Accept Any type for flexibility
        extra_params: Map<String, Any> ?= emptyMap(),
    ) : Map<String,Any> {
        return fetchPlaAds(
            cliUbid = cli_ubid,
            pageName = page_name,
            pageType = page_type,
            productCount = product_count,
            filters = filters,
            extraParams = extra_params
        )
    }

    /**
     * Fetches PLA ads specifically for search pages.
     *
     * @param cli_ubid Unique identifier for the client.
     * @param page_name Optional: Name of the search results page.
     * @param product_count Number of products to display ads for.
     * @param keywords List of search keywords.
     * @param filters Optional: Map containing filter criteria as key-value pairs.
     * @param extra_params Optional: Additional parameters for the request.
     * @return The API response as a Map<String,Any>.
     *
     * Example:
     * ```
     * val res = adFetcherSDK.fetch_pla_search_page_ads(
     *             cli_ubid = "92k31hsd0aish242ljk4sjdas",
     *             page_name = "TOP_OF_SEARCH",
     *             product_count = 10,
     *             keywords = arrayListOf("Sparkling"),
     *             filters = mutableMapOf("store_id" to "Astore"),
     *             extra_params = mutableMapOf(),
     *         )
     * ```
     */
    @Deprecated(
        message = "Use fetchPlaSearchPageAds instead",
        replaceWith = ReplaceWith("fetchPlaSearchPageAds(cliUbid, pageName, productCount, keywords, filters, extraParams)"),
        level = DeprecationLevel.WARNING
    )
    suspend fun fetch_pla_search_page_ads(
        cli_ubid: String,
        page_name: String? ="",
        product_count: Int,
        keywords: List<Any>,
        filters: Map<String, Any>?= emptyMap(),
        extra_params: Map<String, Any>?= emptyMap(),
    ) : Map<String,Any> {
        return fetchPlaSearchPageAds(
            cliUbid = cli_ubid,
            pageName = page_name,
            productCount = product_count,
            keywords = keywords,
            filters = filters,
            extraParams = extra_params
        )
    }

    /**
     * Fetches Product Listing Ads (PLA) for the home page.
     *
     * @param cli_ubid Unique identifier for the client.
     * @param page_name Name of the home page requesting ads.
     * @param product_count Number of products to display ads for.
     * @param filters Optional: Map containing filter criteria; values can be of any type for flexibility.
     * @param extra_params Optional: Additional parameters for the request; defaults to an empty map.
     * @return The API response as a Map<String,Any>.
     *
     * Example:
     * ```
     * val res = adFetcherSDK.fetch_pla_home_page_ads(
     *             cli_ubid = "92k31hsd0",
     *             page_name = "",
     *             product_count = 20,
     *             filters = mutableMapOf("store_id" to "Astore", "brands" to arrayListOf("Whitakers")),
     *             extra_params = mutableMapOf(),
     *         )
     * ```
     */
    @Deprecated(
        message = "Use fetchPlaHomePageAds instead",
        replaceWith = ReplaceWith("fetchPlaHomePageAds(cliUbid, pageName, productCount, filters, extraParams)"),
        level = DeprecationLevel.WARNING
    )
    suspend fun fetch_pla_home_page_ads(
        cli_ubid: String,
        page_name: String? = "",
        product_count: Int,
        filters: Map<String, Any>?= emptyMap() ,
        extra_params: Map<String, Any> ?= emptyMap(),
    ) : Map<String,Any> {
        return fetchPlaHomePageAds(
            cliUbid = cli_ubid,
            pageName = page_name,
            productCount = product_count,
            filters = filters,
            extraParams = extra_params
        )
    }

    /**
     * Fetches PLA ads for a category page.
     *
     * @param cli_ubid Unique identifier for the client.
     * @param page_name Optional: Name of the category page requesting ads.
     * @param product_count Number of products to display ads for.
     * @param categories List of categories to filter ads.
     * @param filters Optional: Map containing filter criteria as key-value pairs.
     * @param extra_params Optional: Additional parameters for the request; defaults to an empty map.
     * @return The API response as a Map<String,Any>.
     *
     * Example:
     * ```
     * val res = adFetcherSDK.fetch_pla_category_page_ads(
     *             cli_ubid = "B8241C5",
     *             page_name = "category_listing",
     *             product_count = 3,
     *             filters = mutableMapOf("store_id" to "Astore"),
     *             extra_params = mutableMapOf(),
     *             categories = arrayListOf("Kitchen & Pantry"),
     *         )
     * ```
     */
    @Deprecated(
        message = "Use fetchPlaCategoryPageAds instead",
        replaceWith = ReplaceWith("fetchPlaCategoryPageAds(cliUbid, pageName, productCount, categories, filters, extraParams)"),
        level = DeprecationLevel.WARNING
    )
    suspend fun fetch_pla_category_page_ads(
        cli_ubid: String,
        page_name: String ?= "",
        product_count: Int,
        categories: List<Any>,
        filters: Map<String, Any> ? = emptyMap(),
        extra_params: Map<String, Any>? = emptyMap(),
    ) : Map<String,Any> {
        return fetchPlaCategoryPageAds(
            cliUbid = cli_ubid,
            pageName = page_name,
            productCount = product_count,
            categories = categories,
            filters = filters,
            extraParams = extra_params
        )
    }

    /**
     * Fetches PLA ads for a product detail page.
     *
     * @param cli_ubid Unique identifier for the client.
     * @param page_name Optional: Name of the product page requesting ads.
     * @param product_count Number of products to display ads for.
     * @param sku_ids List of SKU identifiers for the products.
     * @param filters Optional: Map containing filter criteria as key-value pairs.
     * @param extra_params OPtional: Additional parameters for the request.
     * @return The API response as a Map<String,Any>.
     *
     * Example:
     * ```
     * val res = adFetcherSDK.fetch_pla_product_page_ads(
     *             cli_ubid = "92k31hsd0",
     *             page_name = "",
     *             product_count = 20,
     *             filters = mutableMapOf("store_id" to "Astore"),
     *             extra_params = mutableMapOf(),
     *             sku_ids = arrayListOf("2777534___Astore___Anet"),
     *         )
     * ```
     */
    @Deprecated(
        message = "Use fetchPlaProductPageAds instead",
        replaceWith = ReplaceWith("fetchPlaProductPageAds(cliUbid, pageName, productCount, skuIds, filters, extraParams)"),
        level = DeprecationLevel.WARNING
    )
    suspend fun fetch_pla_product_page_ads(
        cli_ubid: String,
        page_name: String?="",
        product_count: Int,
        sku_ids: List<Any>,
        filters: Map<String, Any>?= emptyMap(),
        extra_params: Map<String, Any>? = emptyMap(),
    ) : Map<String,Any> {
        return fetchPlaProductPageAds(
            cliUbid = cli_ubid,
            pageName = page_name,
            productCount = product_count,
            skuIds = sku_ids,
            filters = filters,
            extraParams = extra_params
        )
    }


    /**
     * Fetches PLA ads for a purchase detail page.
     *
     * @param cli_ubid Unique identifier for the client.
     * @param page_name Optional: Name of the product page requesting ads.
     * @param product_count Number of products to display ads for.
     * @param sku_ids List of SKU identifiers for the products.
     * @param filters Optional: Map containing filter criteria as key-value pairs.
     * @param extra_params Optional: Additional parameters for the request.
     * @return The API response as a Map<String,Any>.
     *
     * Example:
     * ```
     * val res = adFetcherSDK.fetch_pla_purchase_page_ads(
     *             cli_ubid = "9bd111c7-6a57-4a90-8ad0-51a5eb98cfc1",
     *             page_name = "",
     *             product_count = 20,
     *             filters = mutableMapOf("store_id" to "Astore"),
     *             extra_params = mutableMapOf(),
     *             sku_ids = arrayListOf("2777534___Astore___Anet", "2777515___Astore___Anet"),
     *         )
     * ```
     */
    @Deprecated(
        message = "Use fetchPlaPurchasePageAds instead",
        replaceWith = ReplaceWith("fetchPlaPurchasePageAds(cliUbid, pageName, productCount, skuIds, filters, extraParams)"),
        level = DeprecationLevel.WARNING
    )
    suspend fun fetch_pla_purchase_page_ads(
        cli_ubid: String,
        page_name: String? = "",
        product_count: Int,
        sku_ids: List<Any>,
        filters: Map<String, Any> ?= emptyMap(),
        extra_params: Map<String, Any> ?= emptyMap(),
    ) : Map<String,Any> {
        return fetchPlaPurchasePageAds(
            cliUbid = cli_ubid,
            pageName = page_name,
            productCount = product_count,
            skuIds = sku_ids,
            filters = filters,
            extraParams = extra_params
        )
    }

    /**
     * POST API Call for Tagged Product Ads.
     *
     * This function constructs a JSON payload with the provided parameters and sends
     * a POST request to the configured Tagged product ads. The response is logged for
     * debugging purposes.
     *
     * @param cli_ubid Unique identifier for the client session.
     * @param page_name Optional: Name of the page requesting ads.
     * @param product_count Number of products to display ads for.
     * @param sku_ids List of SKU identifiers related to the products.
     * @param filters Optional: Map containing filter criteria; values can be of any type for flexibility.
     * @param extra_params Optional: Additional parameters for the request; values can be of any type.
     * @return The API response as a Map<String,Any>.
     *
    ]     * Example:
     * ```
     * val res  = adFetcherSDK.fetch_tpa_page_ads(
     *             cli_ubid = "9bd111c7-6a57-4a90-8ad0-51a5eb98cfc1",
     *             page_name = "",
     *             product_count = 5,
     *             filters = mutableMapOf("store_id" to "Astore"),
     *             extra_params = mutableMapOf(),
     *             sku_ids = arrayListOf("2777534___Astore___Anet", "2777515___Astore___Anet"),
     *         )
     * ```
     */
    @Deprecated(
        message = "Use fetchTpaPageAds instead",
        replaceWith = ReplaceWith("fetchTpaPageAds(cliUbid, pageName, productCount, skuIds, filters, extraParams)"),
        level = DeprecationLevel.WARNING
    )
    suspend fun fetch_tpa_page_ads(
        cli_ubid: String,
        page_name: String? = "",
        product_count: Int,
        sku_ids: List<Any>,
        filters: Map<String, Any>? = emptyMap(),
        extra_params: Map<String, Any>? = emptyMap(),
    ) : Map<String,Any> {
        return fetchTpaPageAds(
            cliUbid = cli_ubid,
            pageName = page_name,
            productCount = product_count,
            skuIds = sku_ids,
            filters = filters,
            extraParams = extra_params
        )
    }

    /**
     * Constructs query parameters for ad fetching requests.
     *
     * @param isDisplayAuAds Flag to determine if ads are fetched using Ad Units
     * @param cliUbid Unique client identifier
     * @param pageType Type of page requesting ads
     * @param productCount Number of products to display ads for
     * @param adUnit List of ad unit identifiers
     * @param targetingParams Optional: Targeting parameters for ad filtering
     * @param extraParams Optional: Additional parameters if required
     * @return Mutable map containing query parameters
     */
    private fun getQueryParam(
        isDisplayAuAds: Boolean,
        cliUbid: String,
        pageType: String,
        productCount: Int,
        adUnit: List<String>? = null,
        targetingParams: Map<String, Any> ? = emptyMap(),
        extraParams: Map<String, Any>? = emptyMap(),
        functionName: String? = null
    ): MutableMap<String, Any> {

        // Retrieve client ID from the configuration
        val client_id: String = ConfigManager.getString("client_id").toString()

        // Initialize query parameters
        val queryParams = mutableMapOf<String, Any>(
            "pt" to pageType,
            "pcnt_au" to productCount,
            "cli_ubid" to cliUbid,
            "client_id" to client_id
        )

        if (functionName != null){
            queryParams["f_name"] = functionName
        }
        // Add extra parameters if present
        extraParams?.let { queryParams.putAll(it) }

        if (!adUnit.isNullOrEmpty() && isDisplayAuAds) {
            // Add ad unit list
            queryParams["au[]"] = adUnit
        }

        targetingParams?.let { queryParams.putAll(it) }
        return queryParams
    }


    /**
     * Builds query parameters for an API request.
     *
     * This function constructs a mutable map of query parameters required for
     * an API call, incorporating client information, page details, filters,
     * and any extra parameters provided.
     *
     * @param cliUbid Unique identifier for the client session.
     * @param pageType The type of the page requesting the ads.
     * @param productCount The number of products to display ads for.
     * @param pageName Optional name of the page.
     * @param filters Optional: Map containing filter criteria; values can be of any type.
     * @param extraParams Optional: Additional parameters for the request; values can be of any type.
     * @return A mutable map of query parameters for the API request.
     */
    private fun buildQueryParams(
        cliUbid: String,
        pageType: String,
        productCount: Int,
        pageName: String?,
        filters: Map<String, Any>? = emptyMap(),
        extraParams: Map<String, Any>? = emptyMap(),
        functionName: String? = null
    ): MutableMap<String, Any> {

        // Retrieve the client ID from the configuration manager
        val clientId: String = ConfigManager.getString("client_id").toString()

        // Initialize the query parameters map with mandatory parameters
        return mutableMapOf<String, Any>().apply {
            put("cli_ubid", cliUbid)
            put("page_type", pageType)
            put("pcnt", productCount.toString())
            put("client_id", clientId) // Static for now but can be dynamic

            if (!functionName.isNullOrEmpty()) put("f_name", functionName)

            // Add page_name to the parameters if it's not null or empty
            if (!pageName.isNullOrEmpty()) put("page_name", pageName)


            // Add filters to the parameters
            filters?.forEach { (key, value) -> addQueryParam(key, value) }

            // Add extra parameters to the parameters if they are provided
            extraParams?.forEach { (key, value) -> addQueryParam(key, value) }
        }
    }

    /**
     * Helper function to add a query parameter to the map.
     *
     * This function handles the addition of key-value pairs to the query
     * parameters map, ensuring that list values are properly formatted.
     *
     * @receiver The mutable map of query parameters.
     * @param key The key for the query parameter.
     * @param value The value associated with the key; can be a list or a single value.
     */
    private fun MutableMap<String, Any>.addQueryParam(key: String, value: Any) {
        when (value) {
            is List<*> -> put("$key[]", value)  // Append "[]" to the key for list values
            else -> put(key, value)  // Convert other values to string
        }
    }
}