package com.ai.osmos.AdRenderSDK
import android.util.Log
import com.ai.osmos.AdFetcherSDK.AdFetcherSDK

/**
 * Created by adeshmukh on 21/03/25.
 * Project Name: OSMOS-Android-SDK
 * File Name: AdRenderer
 */

/**
 * The [AdRenderer] class is responsible for fetching and preparing ad data for rendering.
 * It acts as a bridge between the ad fetcher module and UI-level ad views.
 */
public class AdRenderer {
    private val adFetcherSDK = AdFetcherSDK()

    /**
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param pageType: page type from which request made. It should be registered while configuring ad inventory
     * @param adUnits:  Ad unit tags defined for the different slots on the page. au parameters are case sensitive. You can fetch multiple ad units using by specifying list of au. You can use maximum 5 au per ad request
     * @param targetingParams(optional): Filter params that you want to apply
     * @param extraParams(optional): Map of any other params
     * @return A map containing a single `"ad"` key with the parsed [BaseAd] object, or an empty map if no ad was found.
     * Example:
     * ```
     * val adData = adRender.fetchBannerAdsWithAu(
     *                 cliUbid = "123",
     *                 pageType = "demo_page",
     *                 adUnits = "banner_ads",
     *                 extraParams = mapOf(
     *                 "f.12312312" to 1))
     * ```
     */
    public suspend fun fetchBannerAdsWithAu(
        cliUbid: String,
        pageType: String,
        adUnits: String,
        targetingParams: Map<String, Any>? = emptyMap(),
        extraParams: Map<String, Any>? = emptyMap()
    ): Map<String, Any> {
        val adUnit = arrayListOf(adUnits)
        return try {
            val adData = adFetcherSDK.fetchDisplayAdsWithAu(
                cliUbid = cliUbid,
                pageType = pageType,
                productCount = 1,
                adUnits = adUnit,
                targetingParams = targetingParams ?: emptyMap(),
                extraParams = extraParams ?: emptyMap()
            ).toMutableMap()

            getAdDataMap(adData, cliUbid, false)
        }
        catch (e: Exception) {
            val error = """
            {
                "status": false,
                "response": null,
                "error": {
                    "code": "ERROR_CONFIGURATION",
                    "description": ""Error parsing ad data"
                }
            }
            """
            Log.e("OSMOS", "OSMOS:$error")
            emptyMap()
        }
    }

    /**
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param pageType: page type from which request made. It should be registered while configuring ad inventory
     * @param adUnits:  Ad unit tags defined for the different slots on the page. au parameters are case sensitive. You can fetch multiple ad units using by specifying list of au. You can use maximum 5 au per ad request
     * @param targetingParams(optional): Filter params that you want to apply
     * @param extraParams(optional): Map of any other params
     * @return A map containing a single `"ad"` key with the parsed [BaseAd] object, or an empty map if no ad was found.
     *Example:
     * ```
     * val adData = adRender.fetchCarouselAdWithAu(
     *                 cliUbid = "123",
     *                 pageType = "demo_page",
     *                 adUnits = "carousel_ads",
     *                 extraParams = mapOf(
     *                 "f.12312312" to 1))
     * ```
     */
    public suspend fun fetchCarouselAdWithAu(
        cliUbid: String,
        pageType: String,
        adUnits: String,
        targetingParams: Map<String, Any>? = emptyMap(),
        extraParams: Map<String, Any>? = emptyMap()
    ): Map<String, Any> {
        val adUnit = arrayListOf(adUnits)
        return try {

            val adData = adFetcherSDK.fetchDisplayAdsWithAu(
                cliUbid = cliUbid,
                pageType = pageType,
                productCount = 1,
                adUnits = adUnit,
                targetingParams = targetingParams ?: emptyMap(),
                extraParams = extraParams ?: emptyMap()
            ).toMutableMap()
            try {
                getAdDataMap(adData, cliUbid, false)
            } catch (e: Exception) {
                val error = """
            {
                "status": false,
                "response": null,
                "error": {
                    "code": "ERROR_CONFIGURATION",
                    "description": ""Error parsing ad data"
                }
            }
            """
                Log.e("OSMOS", "OSMOS:$error")
                emptyMap()
            }
        }
        catch (e: Exception) {
            val error = """
            {
                "status": false,
                "response": null,
                "error": {
                    "code": "ERROR_CONFIGURATION",
                    "description": "Error parsing ad data"
                }
            }
            """
            Log.e("OSMOS", "OSMOS:$error")
            emptyMap()
        }
    }

    /**
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param pageType: page type from which request made. It should be registered while configuring ad inventory
     * @param adUnits:  Ad unit tags defined for the different slots on the page. au parameters are case sensitive. You can fetch multiple ad units using by specifying list of au. You can use maximum 5 au per ad request
     * @param productCount: No of ads responded. MAX 10.
     * @param targetingParams(optional): Filter params that you want to apply
     * @param extraParams(optional): Map of any other params
     * @return A map containing a single `"ad"` key with the parsed [BaseAd] object, or an empty map if no ad was found.
     * Example:
     * ```
     * val adData = adRender.fetchMultiAdCarouselWithAu(
     *                 cliUbid = "123",
     *                 pageType = "demo_page",
     *                 adUnits = "all_in_one",
     *                 productCount = 3,
     *                 extraParams = mapOf(
     *                 "f.12312312" to 1))
     * ```
     */
    public suspend fun fetchMultiAdCarouselWithAu(
        cliUbid: String,
        pageType: String,
        adUnits: String,
        productCount: Int,
        targetingParams: Map<String, Any>? = emptyMap(),
        extraParams: Map<String, Any>? = emptyMap()
    ): Map<String, Any> {
        val adUnit = arrayListOf(adUnits)
        return try {
            val adData = adFetcherSDK.fetchDisplayAdsWithAu(
                cliUbid = cliUbid,
                pageType = pageType,
                productCount = productCount,
                adUnits = adUnit,
                targetingParams= targetingParams ?: emptyMap(),
                extraParams = extraParams ?: emptyMap()
            ).toMutableMap()

            return getAdDataMap(adData,cliUbid, true)

        }catch (e: Exception) {
            val error = """
            {
                "status": false,
                "response": null,
                "error": {
                    "code": "ERROR_CONFIGURATION",
                    "description": ""Error parsing ad data"
                }
            }
            """
            Log.e("OSMOS", "OSMOS:$error")
            emptyMap()
        }
    }


    /**
     *
     * @param cli_ubid: Retailers Generated Id to identify unique shopper.
     * @param page_type: page type from which request made. It should be registered while configuring ad inventory
     * @param ad_unit: Ad unit tags defined for the different slots on the page. au parameters are case sensitive. You can fetch multiple ad units using by specifying list of au. You can use maximum 5 au per ad request
     * @param targeting_params(optional): Filter params that you want to apply
     * @param extra_params(optional): map of any other params
     * @return A map containing a single `"ad"` key with the parsed [BaseAd] object, or an empty map if no ad was found.
     * Example:
     * ```
     * val adData = adRender.fetch_banner_ads_with_au(
     *                 cli_ubid = "123",
     *                 page_type = "demo_page",
     *                 ad_units = "banner_ads",
     *                 extra_params = mapOf(
     *                 "f.12312312" to 1))
     * ```
     */
    @Deprecated(
        message = "Use fetchBannerAdsWithAu instead",
        replaceWith = ReplaceWith("fetchBannerAdsWithAu(cliUbid, pageType, adUnits, targetingParams, extraParams)"),
        level = DeprecationLevel.WARNING
    )
    public suspend fun fetch_banner_ads_with_au(
        cli_ubid: String,
        page_type: String,
        ad_units: String,
        targeting_params: Map<String, Any>? = emptyMap(),
        extra_params: Map<String, Any>? = emptyMap()
    )  : Map<String,Any>{
        return fetchBannerAdsWithAu(
            cliUbid = cli_ubid,
            pageType = page_type,
            adUnits = ad_units,
            targetingParams = targeting_params,
            extraParams = extra_params
        )
    }

    /**
     *
     * @param cli_ubid: Retailers Generated Id to identify unique shopper
     * @param page_type: page type from which request made. It should be registered while configuring ad inventory
     * @param ad_units:  Ad unit tags defined for the different slots on the page. au parameters are case sensitive. You can fetch multiple ad units using by specifying list of au. You can use maximum 5 au per ad request
     * @param targeting_params(optional): Filter params that you want to apply
     * @param extra_params(optional): Map of any other params
     * @return A map containing a single `"ad"` key with the parsed [BaseAd] object, or an empty map if no ad was found.
     *Example:
     * ```
     * val adData = adRender.fetch_carousel_ad_with_au(
     *                 cli_ubid = "123",
     *                 page_type = "demo_page",
     *                 ad_units = "carousel_ads",
     *                 extra_params = mapOf(
     *                 "f.12312312" to 1))
     * ```
     */
    @Deprecated(
        message = "Use fetchCarouselAdWithAu instead",
        replaceWith = ReplaceWith("fetchCarouselAdWithAu(cliUbid, pageType, adUnits, targetingParams, extraParams)"),
        level = DeprecationLevel.WARNING
    )
    public suspend fun fetch_carousel_ad_with_au(
        cli_ubid: String,
        page_type: String,
        ad_units: String,
        targeting_params: Map<String, Any>? = emptyMap(),
        extra_params: Map<String, Any>? = emptyMap()
    )  : Map<String,Any>{
        return fetchCarouselAdWithAu(
            cliUbid = cli_ubid,
            pageType = page_type,
            adUnits = ad_units,
            targetingParams = targeting_params,
            extraParams = extra_params
        )
    }

    /**
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param pageType: page type from which request made. It should be registered while configuring ad inventory
     * @param adUnits:  Ad unit tags defined for the different slots on the page. au parameters are case sensitive. You can fetch multiple ad units using by specifying list of au. You can use maximum 5 au per ad request
     * @param productCount: No of ads responded. MAX 10.
     * @param targetingParams(optional): Filter params that you want to apply
     * @param extraParams(optional): Map of any other params
     * @return A map containing a single `"ad"` key with the parsed [BaseAd] object, or an empty map if no ad was found.
     * Example:
     * ```
     * val adData = adRender.fetch_multi_ad_carousel_with_au(
     *                 cliUbid = "123",
     *                 pageType = "demo_page",
     *                 adUnits = "all_in_one",
     *                 productCount = 3,
     *                 extraParams = mapOf(
     *                 "f.12312312" to 1))
     * ```
     */
    @Deprecated(
        message = "Use fetchMultiAdCarouselWithAu instead",
        replaceWith = ReplaceWith("fetchMultiAdCarouselWithAu(cliUbid, pageType, adUnits, productCount, targetingParams, extraParams)"),
        level = DeprecationLevel.WARNING
    )
    public suspend fun fetch_multi_ad_carousel_with_au(
        cli_ubid: String,
        page_type: String,
        ad_units: String,
        product_count: Int,
        targeting_params: Map<String, Any>? = emptyMap(),
        extra_params: Map<String, Any>? = emptyMap()
    )  : Map<String,Any>{
        return fetchMultiAdCarouselWithAu(
            cliUbid = cli_ubid,
            pageType = page_type,
            adUnits = ad_units,
            productCount = product_count,
            targetingParams = targeting_params,
            extraParams = extra_params
        )
    }


    private fun getAdDataMap(adData: MutableMap<String, Any>, cli_ubid: String, isMultiAd: Boolean): Map<String, Any> {
        if (adData.isEmpty()) return emptyMap()
        return try {
            val adMap = JSONParsing.parseAdData(adData, cli_ubid)

            if (!isMultiAd) {
                val firstAd = adMap.values.firstOrNull()?.firstOrNull()
                if (firstAd != null) {
                    mapOf("ad" to firstAd)
                } else {
                    emptyMap()
                }
            } else {
                val allAds = adMap.values.flatten()
                if (allAds.isNotEmpty()) {
                    mapOf("ad" to allAds)
                } else {
                    emptyMap()
                }
            }
        }catch (e: Exception) {
            val error = """
            {
                "status": false,
                "response": null,
                "error": {
                    "code": "ERROR_CONFIGURATION",
                    "description": ""Error parsing ad data"
                }
            }
            """
            Log.e("OSMOS", "OSMOS:$error")
            emptyMap()
        }
    }
}