package com.ai.osmos.AdRenderSDK

import android.util.Log
import com.ai.osmos.EventSDK.RegisterEvent
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

/**
 * Created by adeshmukh on 28/03/25.
 * Project Name: OSMOS-Android-SDK
 * File Name: AdTracker
 */

/**
 * Handles event tracking for ad impressions, clicks, and video-related interactions.
 *
 * @param coroutineScope The coroutine scope used to launch background tracking jobs.
 */
class AdTracker(private val coroutineScope: CoroutineScope) {
    private val registerEvent = RegisterEvent()

    /**
     * Tracks when an ad impression occurs (i.e., when an ad becomes visible).
     *
     * @param uclid
     * @param cliUbid Retailers Generated Id to identify unique shopper.
     */
    internal fun trackImpression(uclid: String, cliUbid: String) {
        coroutineScope.launch(Dispatchers.IO) {
            try {
                registerEvent.registerAdImpressionEvent(
                    cliUbid = cliUbid,
                    uclid = uclid,
                    position = 1,
                    extraParams = mapOf()
                )
            } catch (e: Exception) {
                Log.e("OsmosAdTracker", "Impression tracking failed: ${e.message}")
            }
        }
    }

    /**
     * Tracks when the user clicks on an ad.
     *
     * @param uclid Unique creative ID.
     * @param cliUbid Retailers Generated Id to identify unique shopper.
     */
    internal fun trackAdClick(uclid: String, cliUbid: String) {
        coroutineScope.launch(Dispatchers.IO) {
            try {
                registerEvent.registerAdClickEvent(
                    cliUbid = cliUbid,
                    uclid = uclid,
                    extraParams = mapOf()
                )
            } catch (e: Exception) {
                Log.e("OsmosAdTracker", "Click tracking failed: ${e.message}")
            }
        }
    }

    /**
     * Tracks a specific video action (e.g., "mute", "unmute").
     *
     * @param uclid Unique creative ID.
     * @param cliUbid Retailers Generated Id to identify unique shopper.
     * @param actionType Type of action performed (e.g., "mute", "unmute").
     * @param videoViewSec Number of seconds the video was viewed when the action click.
     */
    internal fun videoActionClick(uclid: String, cliUbid: String, actionType: String, videoViewSec: Int) {
        coroutineScope.launch(Dispatchers.IO) {
            try {
                registerEvent.registerVideoActionEvent(
                    cliUbid = cliUbid,
                    uclid = uclid,
                    videoViewSec = videoViewSec,
                    actionType = actionType,
                    extraParams = mapOf()
                )
            } catch (e: Exception) {
                Log.e("videoActionClick", "Action tracking failed: ${e.message}")
            }
        }
    }

    /**
     * Tracks periodic progress updates during video playback.
     *
     * @param uclid Unique creative ID.
     * @param cliUbid Retailers Generated Id to identify unique shopper.
     * @param videoViewSec Number of seconds the video has been watched.
     * @param videoDurationSec Total duration of the video in seconds.
     */
    internal fun videoProgressEvent(uclid: String, cliUbid: String, videoViewSec: Int, videoDurationSec: Int) {
        coroutineScope.launch(Dispatchers.IO) {
            try {
                registerEvent.registerVideoProgressEvent(
                    cliUbid = cliUbid,
                    uclid = uclid,
                    videoViewSec = videoViewSec,
                    videoDurationSec = videoDurationSec,
                    extraParams = mapOf()
                )
            } catch (e: Exception) {
                Log.e("videoActionClick", "Action tracking failed: ${e.message}")
            }
        }
    }
}