package com.ai.osmos.AdRenderSDK

import android.annotation.SuppressLint
import android.app.Activity
import android.content.Context
import android.util.AttributeSet
import android.view.Gravity
import com.ai.osmos.utils.ViewUtils
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

/**
 * Created by adeshmukh on 18/03/25
 * Project Name: OSMOS-Android-SDK
 * File Name: AdView
 */

/**
 * BannerAdView.
 *
 * @param context Application context.
 * @param adData BannerAd Data (defined data structure of banner ad)
 * @param width(optional): Width of banner ad view
 * @param height(optional): Height of banner ad view
 * @param attrs(optional): Attrs for ad view
 * @return BannerView or EmptyView
 * Example:
 * ```
 * val adView = BannerAdView(this@MainActivity,
 *                 adData,
 *                 width = 300,
 *                 height = 200)
 * ```
 */
@SuppressLint("ViewConstructor")
class BannerAdView @JvmOverloads constructor(
    context: Context,
    adData: Map<String, Any>,
    private var width: Int? = null,
    private var height: Int? = null,
    attrs: AttributeSet? = null
) : BaseAdView(context, attrs) {

    private val adTracker = AdTracker(coroutineScope)
    private val imageLoader = ImageLoader(coroutineScope, adTracker)
    private val videoLoader = VideoLoader(coroutineScope, adTracker)

    // Initialize the view by loading ad or showing an empty placeholder
    init {
        if (adData.isNotEmpty()) loadAd(adData,context) else showEmptyView(context)
    }
    /**
     * Loads and displays an ad as a banner based on the given ad data.
     *
     * @param adData A map containing the ad object and its associated metadata.
     *               Expected to include a key `"ad"` with value of type `BaseAd`.
     */
    override fun loadAd(adData: Map<String, Any>, context: Context) {
        coroutineScope.launch(Dispatchers.IO) {
            try {
                val ad = adData["ad"] as? BaseAd

                if (ad != null) {
                    val cliUbid = getCliUbid(ad)

                    if (isSupportedCrtImage(ad.crt) || isSupportedCrtVideo(ad.crt)) {
                        withContext(Dispatchers.Main) {
                            removeAllViews()

                            when (ad.elementsType) {
                                //
                                AdType.IMAGE -> {
                                    val imageAd = ad as? ImageAd
                                    val imageUrl = imageAd?.elements?.value
                                    if (!imageUrl.isNullOrEmpty()) {
                                        if (isSupportedImageFormat(ad)){

                                            val (screenWidth, screenHeight) = getScreenHeightWidth()

                                            val h = height ?: (imageAd.elements.height ?: screenHeight)
                                            val w = width ?: (imageAd.elements.width ?: screenWidth)

                                            val result = setMaxHeightWidth(h,w,null)

                                            val (width, height) = getAdDimensions(
                                                result.finalWidth,
                                                result.finalHeight,
                                                imageAd.elements.width ?: screenWidth,
                                                imageAd.elements.height ?: screenHeight
                                            )
                                            val imageView = imageLoader.createImageView(context,
                                                width,
                                                height,
                                                result.isClamped
                                            )

                                            imageLoader.setViewLoadListener { loadedAd, cli_ubid ->
                                                onViewLoadListener?.invoke(
                                                    loadedAd,
                                                    cli_ubid
                                                ) // Notify the listener
                                            }

                                            imageLoader.loadImage(
                                                imageAd,
                                                imageView,
                                                cliUbid,
                                                false,
                                            ) {
                                                adClickedListener?.invoke(imageAd.toMap()) // Pass adMetadata when clicked
                                            }
                                            addView(imageView)
                                        } else showEmptyView(context)
                                    } else showEmptyView(context)
                                }

                                AdType.VIDEO -> {
                                    val videoAd = ad as? VideoAd
                                    if (videoAd != null && videoAd.elements.value.isNotEmpty()) {
                                        if (isSupportedVideoFormat(ad)) {

                                            val (screenWidth, screenHeight) = getScreenHeightWidth()

                                            val h = height ?: (videoAd.elements.height ?: screenHeight)
                                            val w = width ?: (videoAd.elements.width ?: screenWidth)

                                            val result = setMaxHeightWidth(h,w,null)

                                            val (width, height) = getAdDimensions(
                                                result.finalWidth,
                                                result.finalHeight,
                                                videoAd.elements.width ?: screenWidth,
                                                videoAd.elements.height ?: screenHeight
                                            )
                                            val videoView = videoAd.let {

                                                videoLoader.createVideoView(context,it,
                                                    result.isClamped,
                                                    width,
                                                    height,
                                                    cliUbid
                                                ) {
                                                    adClickedListener?.invoke(videoAd.toMap()) // Pass adMetadata when clicked
                                                }
                                            }

                                            //Set the video view in center of framelayout
                                            videoView.let {
                                                val finalWidth = if (result.isClamped) width else ViewUtils.dpToPx(context, width)
                                                val finalHeight = if (result.isClamped) height else ViewUtils.dpToPx(context, height)

                                                val layoutParams = LayoutParams(
                                                    finalWidth,
                                                    finalHeight
                                                ).apply {
                                                    gravity = Gravity.CENTER
                                                }
                                                it.layoutParams = layoutParams
                                                addView(it)

                                            } ?: showEmptyView(context)
                                        } else showEmptyView(context)
                                    } else showEmptyView(context)
                                }

                                AdType.CAROUSEL -> {}
                                AdType.MULTI_AD -> {}
                                AdType.UNKNOWN -> {}
                            }
                        }
                    } else {
                        withContext(Dispatchers.Main) {
                            showEmptyView(context)
                        }
                    }
                }
                else{
                    withContext(Dispatchers.Main) {
                        showEmptyView(context)
                    }
                }
            } catch (e: Exception) {
                withContext(Dispatchers.Main) {
                    showEmptyView(context)
                }
            }
        }
    }


    override fun onDetachedFromWindow() {
        videoLoader.safeReleaseAllPlayers()
        imageLoader.setViewLoadListener { _, _ -> }
        imageLoader.clearViewLoadListener()
        super.onDetachedFromWindow()
    }

    override fun loadAd(adData: Map<String, Any>, activity: Activity?) {
    }

}