package com.ai.osmos.AdRenderSDK

import android.app.Activity
import android.content.Context
import android.content.res.Resources
import android.util.AttributeSet
import android.util.Log
import android.widget.FrameLayout
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.lifecycleScope
import com.ai.osmos.utils.Constants
import com.ai.osmos.utils.ViewUtils
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.cancelChildren

/**
 * Created by adeshmukh on 09/04/25.
 * Project Name: OSMOS-Android-SDK
 * File Name: BaseAdView
 */

/**
 * Abstract base class for rendering ads inside a custom view.
 * Provides shared functionality for all ad types (Banner, Popup, etc.)
 *
 * @param context The context of the view, usually the Activity.
 * @param attrs Optional XML attributes if the view is used in layout XML.
 */
abstract class BaseAdView @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null
) : FrameLayout(context, attrs) {

    private enum class MediaType {
        IMAGE, VIDEO
    }

    private val supervisorJob = SupervisorJob()
    protected val coroutineScope: CoroutineScope =
        (context as? LifecycleOwner)?.lifecycleScope ?: CoroutineScope(Dispatchers.Main + supervisorJob)
//        (context as? LifecycleOwner)?.lifecycleScope ?: CoroutineScope(Dispatchers.Main)

    @Volatile
    protected var adClickedListener: ((adMetadata: Map<String, Any>?) -> Unit)? = null
    @Volatile
    protected var onViewLoadListener: ((ImageAd, String) -> Unit)? = null
    @Volatile
    protected var itemClickedListener: ((adData: Map<String, Any>, clickedItem: Map<String,Any>, destinationUrl: String) -> Unit)? = null

    /**
     * Set a click listener for the ad.
     *
     * @param listener Callback that provides ad metadata when the ad is clicked.
     */
    fun setAdClickListener(listener: (adMetadata: Map<String, Any>?) -> Unit) {
        this.adClickedListener = listener
    }

    fun setAdItemClickListener(listener: (adData: Map<String, Any>, clickedItem: Map<String,Any>, destinationUrl: String) -> Unit) {
        this.itemClickedListener = listener
    }

    /**
     * Set a view load listener that gets triggered when an ad is fully loaded.
     *
     * @param listener Callback that returns the loaded ImageAd and tracking ID (cli_ubid).
     */
    fun setViewLoadListener(listener: (ImageAd, String) -> Unit) {
        this.onViewLoadListener = listener
    }

    /**
     * Abstract method to be implemented by subclasses for loading ads from given data.
     *
     * @param adData A map containing the ad object and related metadata.
     */
    abstract fun loadAd(adData: Map<String, Any>,context: Context)


    abstract fun loadAd(adData: Map<String, Any>,activity: Activity?)

    /**
     * Resolves the final width and height to use for the ad view.
     * Priority: requested size → fallback from ad → hardcoded fallback.
     *
     * @param requestedWidth Optional width provided by caller (in dp).
     * @param requestedHeight Optional height provided by caller (in dp).
     * @param fallbackWidth Width from ad metadata (if present).
     * @param fallbackHeight Height from ad metadata (if present).
     *
     * @return A pair of width and height in dp.
     */
    protected fun getAdDimensions(
        requestedWidth: Int?,
        requestedHeight: Int?,
        fallbackWidth: Int,
        fallbackHeight: Int
    ): Pair<Int, Int> {
        val finalWidth = requestedWidth ?: fallbackWidth
        val finalHeight = requestedHeight ?: fallbackHeight
        return Pair(finalWidth, finalHeight)
    }

    protected fun getScreenHeightWidth(): Pair<Int, Int> {

        val screenWidth = Resources.getSystem().displayMetrics.widthPixels
        val screenHeight = Resources.getSystem().displayMetrics.heightPixels

        val maxAdWidth = (screenWidth * 0.8f).toInt()
        val maxAdHeight = (screenHeight * 0.8f).toInt()

        return Pair(maxAdWidth, maxAdHeight)
    }
    data class LayoutResult(
        val finalWidth: Int,
        val finalHeight: Int,
        val coordinates: Coordinates?,
        val isClamped: Boolean
    )
    protected fun setMaxHeightWidth(
        height: Int,
        width: Int,
        inputCoordinates: Coordinates?
    ): LayoutResult {

        val screenWidth = Resources.getSystem().displayMetrics.widthPixels
        val screenHeight = Resources.getSystem().displayMetrics.heightPixels

        val maxAdWidth = (screenWidth * 0.8f).toInt()
        val maxAdHeight = (screenHeight * 0.8f).toInt()

        val finalWidth = minOf(width, maxAdWidth)
        val finalHeight = minOf(height, maxAdHeight)

        val maxX = screenWidth - finalWidth - 16
        val maxY = screenHeight - finalHeight - 34

        val finalX = inputCoordinates?.x?.coerceAtMost(maxX)
        val finalY = inputCoordinates?.y?.coerceAtMost(maxY)

        val clamped = (width > maxAdWidth || height > maxAdHeight)

        val coordinates = if (inputCoordinates != null) Coordinates(finalX!!, finalY!!) else null
        return LayoutResult(
            finalWidth = finalWidth,
            finalHeight = finalHeight,
            coordinates = coordinates,
            isClamped = clamped
        )
    }

    /**
     * Checks if the image format of the given ad is supported.
     *
     * This function verifies whether the ad is an instance of [ImageAd], and if so,
     * it checks the file extension of the URL to ensure it's a supported image format.
     * If the ad is not an [ImageAd], it returns `true` by default, assuming the format is supported.
     *
     * @param ad The ad to check. It must be of type [ImageAd] for this function to verify the image format.
     * @return `true` if the image format is supported, `false` if not, or `true` for non-ImageAds.
     */
    internal fun isSupportedImageFormat(ad: BaseAd?): Boolean {
        if (ad is ImageAd) {
            val url = ad.elements.value.trim().lowercase()
            val extension = url.substringAfterLast('.', "").substringBefore('?').lowercase()

            return isValidMediaUrl(url, extension, MediaType.IMAGE)
        }
        return true
    }

    /**
     * Checks if the video format of the given ad is supported.
     *
     * This function validates if the ad is a `VideoAd`, and if so, it performs checks to verify
     * whether the necessary dependencies for ExoPlayer and the required video formats (e.g., MP4, HLS, DASH) are available.
     *
     * @param ad The ad to check. It must be of type [VideoAd] for this function to verify the video format.
     * @return `true` if the video format is supported, `false` otherwise.
     */
    internal fun isSupportedVideoFormat(ad: BaseAd?): Boolean {
        if (ad is VideoAd) {
            // Check if core ExoPlayer + UI are present
            if (!isExoPlayerAvailable()) {
                errorLog("ExoPlayer core is missing. Please include 'media3-exoplayer' dependency in your app.")
                return false
            }

            if (!isExoPlayerUiAvailable()) {
                errorLog("ExoPlayer UI module is missing. Please include 'media3-ui' dependency in your app.")
                return false
            }

            //Check for unsupported video format
            val url = ad.elements.value.trim().lowercase()
            val extension = url.substringAfterLast('.', "").substringBefore('?')
            val isValid = isValidMediaUrl(url, extension, MediaType.VIDEO)

            if (isValid) {
                return when {
                    extension in listOf("mp4", "webm", "mov", "m4v", "mkv", "wmv") -> true
                    extension == "m3u8" -> {
                        if (isHlsAvailable()) {
                            true
                        } else {
                            errorLog("HLS support is missing. Please include 'media3-exoplayer-hls' dependency in your app.")
                            false
                        }
                    }

                    extension == "mpd" -> {
                        if (isDashAvailable()) {
                            true
                        } else {
                            errorLog("DASH support is missing. Please include 'media3-exoplayer-dash' dependency in your app.")
                            false
                        }
                    }

                    else -> {
                        errorLog("Unsupported video format")
                        false
                    }
                }
            } else {
                errorLog("Unsupported video format")
                return false
            }
        }
        return true
    }

    /**
     * Validates the given media URL asynchronously.
     *
     * This function uses a suspending coroutine to wrap the callback-based `isValidUrl` function
     * so that it can be used within Kotlin coroutines.
     *
     * It suspends the execution of the calling coroutine until the result of the URL validation is available,
     * and then resumes with a `true` or `false` value indicating whether the media URL is valid.
     *
     * @param url The URL to validate.
     * @param extension The file extension of the URL (e.g., ".mp4", ".jpg").
     * @param mediaType The type of media (IMAGE or VIDEO).
     * @return A `Boolean` indicating if the URL is valid or not. This is returned as a suspending function.
     */
    private fun isValidMediaUrl(
        url: String,
        extension: String,
        mediaType: MediaType
    ): Boolean {
        val supportedImageExtensions = listOf("png", "jpg", "jpeg", "gif")
        val supportedVideoExtensions = listOf("mp4", "webm", "mov", "m4v", "mkv", "wmv", "mpd", "m3u8")

        val supportedExtensions = when (mediaType) {
            MediaType.IMAGE -> supportedImageExtensions
            MediaType.VIDEO -> supportedVideoExtensions
        }
        if (extension !in supportedExtensions) {
            errorLog("Unsupported ${mediaType.name.lowercase()} format")
            return false
        }
        return true
    }
    /**
     * Logs an error message in a structured JSON format.
     *
     * This function formats an error message into a JSON string and logs it with an error log level using Log.e().
     * The JSON structure contains the status, a null response, and the error details including code and description.
     *
     * @param message The error message to be logged, describing the issue encountered.
     */
    private fun errorLog(message: String) {
        val error = """
            {
                "status": false,
                "response": null,
                "error": {
                    "code": "ERROR_CONFIGURATION",
                    "description": $message
                    }
            }
        """
        Log.e("TAG", error)
    }

    /**
     * Checks if the core ExoPlayer library is available in the classpath.
     *
     * @return true if 'androidx.media3.exoplayer.ExoPlayer' class is found; false otherwise.
     */
    private fun isExoPlayerAvailable(): Boolean {
        return try {
            Class.forName("androidx.media3.exoplayer.ExoPlayer")
            true
        } catch (e: ClassNotFoundException) {
            false
        }
    }

    /**
     * Checks if the ExoPlayer UI module is available in the classpath.
     *
     * @return true if 'androidx.media3.ui.PlayerView' class is found; false otherwise.
     */
    private fun isExoPlayerUiAvailable(): Boolean {
        return try {
            Class.forName("androidx.media3.ui.PlayerView")
            true
        } catch (e: ClassNotFoundException) {
            false
        }
    }
    /**
     * Checks if the HLS (HTTP Live Streaming) support is available.
     *
     * @return true if 'androidx.media3.exoplayer.hls.HlsMediaSource' is found; false otherwise.
     */
    private fun isHlsAvailable(): Boolean {
        return try {
            Class.forName("androidx.media3.exoplayer.hls.HlsMediaSource")
            true
        } catch (e: ClassNotFoundException) {
            false
        }
    }

    /**
     * Checks if the DASH (Dynamic Adaptive Streaming over HTTP) support is available.
     *
     * @return true if 'androidx.media3.exoplayer.dash.DashMediaSource' is found; false otherwise.
     */
    private fun isDashAvailable(): Boolean {
        return try {
            Class.forName("androidx.media3.exoplayer.dash.DashMediaSource")
            true
        } catch (e: ClassNotFoundException) {
            false
        }
    }

    /**
     * Checks if the RecyclerView library is available in the classpath.
     *
     * @return true if 'androidx.recyclerview.widget.RecyclerView' class is found; false otherwise.
     */
    internal fun isRecyclerViewAvailable(): Boolean {
        return try {
            Class.forName("androidx.recyclerview.widget.RecyclerView")
            true
        } catch (e: ClassNotFoundException) {
            false
        }
    }

    /**
     * Retrieves the client-side tracking ID from the ad.
     *
     * @param ad The ad from which to extract the tracking ID.
     * @return The cli_ubid string, or an empty string if null.
     */
    internal fun getCliUbid(ad: BaseAd?): String = ad?.cli_ubid ?: ""

    /**
     * Displays a fallback empty view if no ad is available or loading fails.
     */
    internal fun showEmptyView(context: Context) {
        removeAllViews()
        addView(ViewUtils.getEmptyView(context))
    }

    /**
     * Checks if the creative type (crt) is supported by the SDK.
     *
     * @param crt Creative type string from the ad metadata.
     * @return True if supported (i.e., starts with "osmos_sdk/image/v1"), false otherwise.
     */
    protected fun isSupportedCrtImage(crt: String): Boolean {
        return crt.startsWith(Constants.CRT_BANNER_IMAGE)
    }

    protected fun isSupportedCrtVideo(crt: String): Boolean {
        return crt.startsWith(Constants.CRT_BANNER_VIDEO)
    }

    protected fun isSupportedCrtCarousel(crt: String): Boolean {
        return crt.startsWith(Constants.CRT_CAROUSEL)
    }

    /**
     * Override onDetachedFromWindow to properly clean up resources and prevent memory leaks
     */
    override fun onDetachedFromWindow() {
        try {
            if (context !is LifecycleOwner) {
                supervisorJob.takeIf { it.isActive }?.let { job ->
                    job.cancelChildren() // Cancel children first
                    job.cancel()         // Then cancel the parent job
                }
            }
            clearAllListeners()
        } catch (e: Exception) {
            Log.e("BaseAdView", "Error during cleanup: ${e.message}")
        }
        super.onDetachedFromWindow()
    }
    
    /**
     * Clear all callback listeners to prevent memory leaks
     */
    private fun clearAllListeners() {
        adClickedListener = null
        onViewLoadListener = null
        itemClickedListener = null
    }

}

