package com.ai.osmos.AdRenderSDK

import android.content.Context
import android.content.res.Resources
import android.graphics.Color
import android.graphics.Rect
import android.util.Log
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import android.widget.FrameLayout
import android.widget.ImageView
import android.widget.LinearLayout
import android.widget.TextView
import androidx.media3.ui.PlayerView
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import com.ai.osmos.AdRenderSDK.CarouselLoader.CarouselAdapter.VideoViewHolder
import com.ai.osmos.utils.ConfigManager
import com.ai.osmos.utils.ViewUtils
import kotlinx.coroutines.CoroutineScope
import kotlin.math.abs
import kotlin.math.max

/**
 * Created by adeshmukh on 24/04/25.
 * Project Name: OSMOS-Android-SDK
 * File Name: CarouselLoader
 */
class CarouselLoader(
    private val coroutineScope: CoroutineScope,
    private val adTracker: AdTracker
) {
    private val imageLoader = ImageLoader(coroutineScope, adTracker)
    private val videoLoader = VideoLoader(coroutineScope, adTracker)
    private var currentIndex = -1
    private var dotsContainer: LinearLayout? = null
    private lateinit var recyclerView: RecyclerView

    private lateinit var multiRecyclerView: RecyclerView

    private var isCarouselAd : Boolean = false
    fun renderCarouselView(
        context: Context,
        width: Int?,
        height: Int?,
        cliUbid: String,
        adList: List<CarouselAdElement>,
        carouselAd: CarouselAd ,
        adClickedListener: ((adData: Map<String, Any>, clickedItem: Map<String, Any>, destinationUrl: String) -> Unit)?,
    ): View {

        val container = FrameLayout(context)
        val result = setMaxHeightWidth(context,width,height, true,carouselAd = carouselAd,null)

        recyclerView = RecyclerView(context).apply {
            layoutManager = LinearLayoutManager(context, LinearLayoutManager.HORIZONTAL, false).apply {
                layoutParams = ViewGroup.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT,
                    result.finalCarouselHeight
                )
            }
            isCarouselAd = true
            adapter = CarouselAdapter(
                context,
                adList,
                carouselAd,
                cliUbid,
                result.carouselWidth,
                result.carouselHeight,
                result.firstElementWidth,
                result.firstElementHeight,
                result.isClamped,
                adClickedListener
            )
        }

        setItemDecoration(context,recyclerView)

        container.addView(recyclerView)

        addDotIndicator(context,container,null,adList)

        addScrollListener(isCarouselAd,recyclerView)

        return container
    }

    fun renderMultiAdCarouselView(
        context: Context,
        width: Int?,
        height: Int?,
        multiAd: List<MultiAd>,
        adClickListener: ((Map<String, Any>?) -> Unit)?
    ): View {
        val container = FrameLayout(context)

        val result = setMaxHeightWidth(context,width,height, false,null,multiAd.firstOrNull())

        multiRecyclerView = RecyclerView(context).apply {
            layoutManager = LinearLayoutManager(context, LinearLayoutManager.HORIZONTAL, false).apply {
                layoutParams = ViewGroup.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT,
                    result.finalCarouselHeight
                )
            }
            isCarouselAd = false

            adapter = MultiAdAdapter(context,
                multiAd,
                result.carouselWidth,
                result.carouselHeight,
                result.firstElementWidth,
                result.firstElementHeight,
                result.isClamped,
                adClickListener)
        }

        setItemDecoration(context,multiRecyclerView)

        container.addView(multiRecyclerView)

        addDotIndicator(context,container,multiAd,null)

        addScrollListener(isCarouselAd,multiRecyclerView)

        return container
    }

    private fun addScrollListener(isCarouselAd: Boolean, recyclerView: RecyclerView) {

        recyclerView.addOnScrollListener(object : RecyclerView.OnScrollListener() {
            override fun onScrolled(rv: RecyclerView, dx: Int, dy: Int) {
                handleVisibleItem(isCarouselAd)
            }
        })

        recyclerView.post {
            handleVisibleItem(isCarouselAd)
        }
    }

    private fun addDotIndicator(
        context: Context,
        container: FrameLayout,
        multiAd: List<MultiAd>?,
        adList: List<CarouselAdElement>?
    ){
        dotsContainer = LinearLayout(context).apply {
            orientation = LinearLayout.HORIZONTAL
            gravity = Gravity.CENTER
            layoutParams = FrameLayout.LayoutParams(
                FrameLayout.LayoutParams.MATCH_PARENT,
                FrameLayout.LayoutParams.WRAP_CONTENT
            ).apply {
                gravity = Gravity.BOTTOM or Gravity.CENTER_HORIZONTAL
                bottomMargin = 16 // Optional: add some bottom margin
            }
        }
        (multiAd ?: adList)?.forEach { _ ->
            dotsContainer?.addView(TextView(context).apply {
                text = "•"
                textSize = 20f
                setTextColor(Color.GRAY)
            })
        }
        container.addView(dotsContainer)
    }

    private fun setItemDecoration(context: Context, multiRecyclerView: RecyclerView) {
        val density = context.resources.displayMetrics.density

        val margin = (10 * density).toInt()
        multiRecyclerView.addItemDecoration(object : RecyclerView.ItemDecoration() {
            override fun getItemOffsets(outRect: Rect, view: View, parent: RecyclerView, state: RecyclerView.State) {
                val position = parent.getChildAdapterPosition(view)
                outRect.left = if (position == 0) margin else margin / 2
                outRect.right = if (position == state.itemCount - 1) margin else margin / 2
            }
        })
    }

    private fun handleVisibleItem(isCarouselAd: Boolean) {
        val center = if (isCarouselAd)recyclerView.width / 2 else multiRecyclerView.width/2
        var bestIndex = -1
        var minDistance = Int.MAX_VALUE

        // Find the closest item to the center
        if (isCarouselAd)
        {
            for (i in 0 until recyclerView.childCount) {
                if (recyclerView.childCount == 0) return
                val view = recyclerView.getChildAt(i)
                val distance = abs(view.left + view.width / 2 - center)
                if (distance < minDistance) {
                    bestIndex = recyclerView.getChildAdapterPosition(view)
                    minDistance = distance
                }
            }
        }
        else{
            for (i in 0 until multiRecyclerView.childCount) {
                if (multiRecyclerView.childCount == 0) return
                val view = multiRecyclerView.getChildAt(i)
                val distance = abs(view.left + view.width / 2 - center)
                if (distance < minDistance) {
                    bestIndex = multiRecyclerView.getChildAdapterPosition(view)
                    minDistance = distance
                }
            }
        }

        // If the bestIndex has changed, update the visible item
        if (bestIndex != currentIndex) {
            // Pause the currently playing video (if any)

            pauseCurrentPlayingVideo(if (isCarouselAd) recyclerView else multiRecyclerView,currentIndex)

            // Update the currentIndex
            currentIndex = bestIndex
            updateDots()

            // Play the new video
            playCurrentVideo(if (isCarouselAd) recyclerView else multiRecyclerView,currentIndex)
        }
    }

    private fun playCurrentVideo(recyclerView: RecyclerView, currentIndex: Int) {
        val newHolder =
            recyclerView.findViewHolderForAdapterPosition(currentIndex) as? CarouselAdapter.VideoViewHolder
        newHolder?.frame?.post {
            newHolder.playerView.let {
                videoLoader.playExclusive(it)
            }
        }
    }

    private fun pauseCurrentPlayingVideo(recyclerView: RecyclerView, currentIndex: Int) {
        val currentHolder =
            recyclerView.findViewHolderForAdapterPosition(currentIndex) as? CarouselAdapter.VideoViewHolder
        currentHolder?.let {
            val currentPlayer = it.playerView.player
            currentPlayer?.playWhenReady = false  // Pause the current video
        }
    }

    private fun updateDots() {
        dotsContainer?.let {
            for (i in 0 until it.childCount) {
                (it.getChildAt(i) as? TextView)?.setTextColor(
                    if (i == currentIndex) Color.WHITE else Color.GRAY
                )
            }
        }
    }


    data class LayoutResult(
        val finalCarouselWidth: Int,
        val finalCarouselHeight: Int,
        val firstElementWidth: Int,
        val firstElementHeight: Int,
        val carouselWidth: Int,
        val carouselHeight: Int,
        val isClamped: Boolean
    )
    private fun setMaxHeightWidth(
        context: Context,
        width: Int?,
        height: Int?,
        isCarouselAd: Boolean,
        carouselAd: CarouselAd?,
        multiAd: MultiAd?
    ): LayoutResult {

        val screenWidth = Resources.getSystem().displayMetrics.widthPixels.toFloat()
        val screenHeight = Resources.getSystem().displayMetrics.heightPixels.toFloat()

        val maxAdWidth = (screenWidth * 0.8f).toInt()
        val maxAdHeight = (screenHeight * 0.8f).toInt()

        var firstElementWidth = maxAdWidth
        var firstElementHeight = maxAdHeight

        if (isCarouselAd) {
            val firstElement = carouselAd?.elements?.carousel_cards?.firstOrNull()
            firstElementWidth = firstElement?.media?.width ?: maxAdWidth
            firstElementHeight = firstElement?.media?.height ?: maxAdHeight
        } else {
            firstElementWidth = multiAd?.elements?.width ?: maxAdWidth
            firstElementHeight = multiAd?.elements?.height ?: maxAdHeight
        }

        val carouselWidth = minOf(width ?: maxAdWidth, maxAdWidth)
        val carouselHeight = minOf(height ?: firstElementHeight, maxAdHeight)

        var isClamped = false
        if (width != null  && height != null){
            isClamped = (width > maxAdWidth || height > maxAdHeight)
        }

        val finalCarouselWidth = if (isClamped) carouselWidth else ViewUtils.dpToPx(context,carouselWidth)
        val finalCarouselHeight = if (isClamped) carouselHeight else ViewUtils.dpToPx(context,carouselHeight)

        return LayoutResult(
            finalCarouselWidth = finalCarouselWidth,
            finalCarouselHeight = finalCarouselHeight,
            firstElementWidth = firstElementWidth,
            firstElementHeight = firstElementHeight,
            carouselHeight = carouselHeight,
            carouselWidth =  carouselWidth,
            isClamped = isClamped
        )
    }

    inner class CarouselAdapter(
        private val context: Context,
        private val items: List<CarouselAdElement>,
        private val carouselAd: CarouselAd,
        private val cliUbid: String,
        private val carouselWidth: Int?,
        private val carouselHeight: Int?,
        private val firstElementWidth: Int?,
        private val firstElementHeight: Int?,
        private val isClamped: Boolean,
        private val adClickedListener: ((adData: Map<String, Any>, clickedItem: Map<String, Any>, destinationUrl: String) -> Unit)?,

        ) : RecyclerView.Adapter<RecyclerView.ViewHolder>() {

        private val TYPE_VIDEO = 1
        private val TYPE_IMAGE = 0

        override fun getItemViewType(position: Int): Int {
            val type = items[position].media.type
            return if (type.equals("video", ignoreCase = true)) TYPE_VIDEO else TYPE_IMAGE
        }

        override fun getItemCount(): Int {
            return items.size
        }

        override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): RecyclerView.ViewHolder {
            val frame = FrameLayout(context)
            return if (viewType == TYPE_VIDEO) VideoViewHolder(frame) else ImageViewHolder(frame)
        }

        override fun onViewRecycled(holder: RecyclerView.ViewHolder) {
            if (holder is VideoViewHolder) {
                videoLoader.safeStopAndReleasePlayer(holder.playerView)
            }
            super.onViewRecycled(holder)
        }
        override fun onBindViewHolder(holder: RecyclerView.ViewHolder, position: Int) {
            val item = items[position]
            val url = item.media.value
            val cardWidth = minOf((item.media.width ?: firstElementWidth)!!, carouselWidth!!)
            val cardHeight = minOf((item.media.height ?: firstElementHeight)!!, carouselHeight!!)

            val density = context.resources.displayMetrics.density
            val pxW = (cardWidth * density).toInt()
            val pxH = (cardHeight * density).toInt()

            when (holder) {
                is VideoViewHolder -> {
                    holder.frame.removeAllViews()
                    // Generate a "per-element" video metadata object
                    val videoAd = getVideoAdObject(carouselAd,item,cardWidth,cardHeight)

                    val videoView = videoLoader.createCarouselVideoView(
                        context,
                        videoAd,
                        cardWidth,
                        cardHeight,
                        "$cliUbid-$position")
                    holder.playerView = videoView.getChildAt(0) as PlayerView
                    holder.frame.setOnClickListener {
                        adClickedListener?.invoke(carouselAd.toMap() ?: emptyMap(), createCarouselCardMap(item), videoAd.elements.destination_url.toString())
                        adTracker.trackAdClick(videoAd.uclid, videoAd.cli_ubid!!)
                    }


                    holder.frame.addView(videoView, FrameLayout.LayoutParams(pxW, pxH))
                }

                is ImageViewHolder -> {
                    holder.frame.removeAllViews()
                    val imageAd = getImageAdObject(carouselAd,item,pxW,pxH)

                    val imageView = imageLoader.createImageView(context, cardWidth, cardHeight, isClamped = isClamped )
                    imageLoader.loadImage(
                        imageAd,
                        imageView,
                        "$cliUbid-$position",
                        true
                    )
                    holder.frame.setOnClickListener {
                        adClickedListener?.invoke(carouselAd.toMap() ?: emptyMap(), createCarouselCardMap(item), imageAd.elements.destination_url.toString())
                        adTracker.trackAdClick(imageAd.uclid, imageAd.cli_ubid!!)
                    }
                    holder.frame.addView(imageView, FrameLayout.LayoutParams(pxW, pxH))
                }
            }
        }

        inner class VideoViewHolder(val frame: FrameLayout) : RecyclerView.ViewHolder(frame) {
            lateinit var playerView: PlayerView
        }

        inner class ImageViewHolder(val frame: FrameLayout) : RecyclerView.ViewHolder(frame)

        private fun getVideoAdObject(carouselAd: CarouselAd, item: CarouselAdElement,width:Int, height: Int): VideoAd {
            return VideoAd(
                client_id = carouselAd.client_id, // or extract if available
                au = carouselAd.au,        // optional ad unit or campaign string
                rank = carouselAd.rank,       // optional rank
                click_tracking_url = carouselAd.click_tracking_url,
                impression_tracking_url = carouselAd.impression_tracking_url, // add if available
                uclid = carouselAd.uclid,     // unique click ID if available
                crt = carouselAd.crt,       // creative type if relevant
                elements = AdElement(
                    type = item.media.type,
                    value = item.media.value,
                    height = height,
                    width = width,
                    destination_url = item.destination_url),
                adMetadata = carouselAd.adMetadata,
                cli_ubid = cliUbid
            )
        }

        private fun getImageAdObject(carouselAd: CarouselAd, item: CarouselAdElement,width:Int, height: Int): ImageAd {
            return ImageAd(
                client_id = carouselAd.client_id, // or extract if available
                au = carouselAd.au,        // optional ad unit or campaign string
                rank = carouselAd.rank,       // optional rank
                click_tracking_url = carouselAd.click_tracking_url,
                impression_tracking_url = carouselAd.impression_tracking_url, // add if available
                uclid = carouselAd.uclid,     // unique click ID if available
                crt = carouselAd.crt,       // creative type if relevant
                elements = AdElement(
                    type = item.media.type,
                    value = item.media.value,
                    height = height,
                    width = width,
                    destination_url = item.destination_url),
                adMetadata = carouselAd.adMetadata,
                cli_ubid = cliUbid
            )
        }

        private fun createCarouselCardMap(carouselAdElement: CarouselAdElement): Map<String, Any> {
            return mapOf(
                "value" to carouselAdElement.media.value,
                "height" to carouselAdElement.media.height,
                "width" to carouselAdElement.media.width,
                "type" to carouselAdElement.media.type,
                "destination_url" to carouselAdElement.destination_url
            )
        }
    }

    fun releaseAllPlayers() {
        videoLoader.safeReleaseAllPlayers()
    }

    inner class MultiAdAdapter(
        private val context: Context,
        private val items: List<MultiAd>,
        private val carouselWidth: Int?,
        private val carouselHeight: Int?,
        private val firstElementWidth: Int?,
        private val firstElementHeight: Int?,
        private val isClamped: Boolean,
        private val adClickListener: ((Map<String, Any>?) -> Unit)?
    ) : RecyclerView.Adapter<RecyclerView.ViewHolder>() {


        private val TYPE_VIDEO = 1
        private val TYPE_IMAGE = 0

        override fun getItemViewType(position: Int): Int {
            val multiAd = items[position]
            val type = multiAd.elements.type
            return if (type.equals("video", ignoreCase = true)) TYPE_VIDEO else TYPE_IMAGE
        }

        override fun getItemCount(): Int {
            return items.size
        }

        override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): RecyclerView.ViewHolder {
            val frame = FrameLayout(context)
            return if (viewType == TYPE_VIDEO) VideoViewHolder(frame) else ImageViewHolder(frame)
        }

        override fun onViewRecycled(holder: RecyclerView.ViewHolder) {
            if (holder is VideoViewHolder) {
                videoLoader.safeStopAndReleasePlayer(holder.playerView)
            }
            super.onViewRecycled(holder)
        }
        override fun onBindViewHolder(holder: RecyclerView.ViewHolder, position: Int) {
            val multiAd = items[position]
            val cardWidth = minOf((multiAd.elements.width ?: firstElementWidth)!!, carouselWidth!!)
            val cardHeight = minOf((multiAd.elements.height ?: firstElementHeight)!!, carouselHeight!!)

            val density = context.resources.displayMetrics.density
            val pxW = (cardWidth * density).toInt()
            val pxH = (cardHeight * density).toInt()

            when (holder) {
                is VideoViewHolder -> {
                    holder.frame.removeAllViews()
                    // Generate a "per-element" video metadata object
                    val videoAd = getVideoAdObject(multiAd,cardHeight,cardWidth)

                    val videoView = videoLoader.createCarouselVideoView(
                        context,
                        videoAd,
                        cardWidth,
                        cardHeight,
                        "${videoAd.cli_ubid}-$position",
                    )

                    val playerView = videoView.getChildAt(0) as PlayerView
                    holder.playerView = playerView
                    playerView.addOnAttachStateChangeListener(object : View.OnAttachStateChangeListener {
                        override fun onViewAttachedToWindow(p0: View) = Unit
                        override fun onViewDetachedFromWindow(p0: View) {
                            videoLoader.releasePlayer(playerView)                        }
                    })
                    holder.frame.setOnClickListener {
                        adClickListener?.invoke(videoAd.toMap())
                        adTracker.trackAdClick(videoAd.uclid, videoAd.cli_ubid!!)
                    }

                    holder.frame.addView(videoView, FrameLayout.LayoutParams(pxW, pxH))

                }

                is ImageViewHolder -> {
                    holder.frame.removeAllViews()
                    val imageAd = getImageAdObject(multiAd,cardHeight,cardWidth)

                    val imageView = imageLoader.createImageView(context, cardWidth, cardHeight,isClamped)
                    imageLoader.loadImage(
                        imageAd,
                        imageView,
                        "$imageAd.cli_ubid-$position",
                        true
                    )
                    holder.frame.setOnClickListener {
                        adClickListener?.invoke(imageAd.toMap())
                        adTracker.trackAdClick(imageAd.uclid, cliUbid = imageAd.cli_ubid!!)
                    }

                    holder.frame.addView(imageView, FrameLayout.LayoutParams(pxW, pxH))
                }
            }
        }

        inner class VideoViewHolder(val frame: FrameLayout) : RecyclerView.ViewHolder(frame) {
            lateinit var playerView: PlayerView
        }

        inner class ImageViewHolder(val frame: FrameLayout) : RecyclerView.ViewHolder(frame){

        }


        private fun getVideoAdObject(multiAd: MultiAd, cardHeight: Int, cardWidth: Int): VideoAd {
            return VideoAd(
                client_id = multiAd.client_id,
                au = multiAd.au,
                rank = multiAd.rank,
                click_tracking_url = multiAd.click_tracking_url,
                impression_tracking_url = multiAd.impression_tracking_url,
                uclid = multiAd.uclid,
                crt = multiAd.crt,
                elements = AdElement(
                    type = multiAd.elements.type,
                    value = multiAd.elements.value,
                    height = cardHeight,
                    width = cardWidth,
                    destination_url = multiAd.elements.destination_url),
                adMetadata = multiAd.adMetadata,
                cli_ubid = multiAd.cli_ubid
            )
        }

        private fun getImageAdObject(multiAd: MultiAd, cardHeight: Int, cardWidth: Int): ImageAd {
            return ImageAd(
                client_id = multiAd.client_id,
                au = multiAd.au,
                rank = multiAd.rank,
                click_tracking_url = multiAd.click_tracking_url,
                impression_tracking_url = multiAd.impression_tracking_url,
                uclid = multiAd.uclid,
                crt = multiAd.crt,
                elements = AdElement(
                    type = multiAd.elements.type,
                    value = multiAd.elements.value,
                    height = cardHeight,
                    width = cardWidth ,
                    destination_url = multiAd.elements.destination_url),
                adMetadata = multiAd.adMetadata,
                cli_ubid = multiAd.cli_ubid
            )
        }
    }

}
