package com.ai.osmos.AdRenderSDK

import android.content.Context
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.ImageDecoder
import android.graphics.drawable.AnimatedImageDrawable
import android.os.Build
import android.util.Log
import android.view.Gravity
import android.view.View
import android.view.ViewTreeObserver
import android.widget.FrameLayout.LayoutParams
import android.widget.ImageView
import com.ai.osmos.utils.ViewUtils
import com.ai.osmos.utils.isVisibleInScreen
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.net.URL
import java.nio.ByteBuffer

/**
 * Created by adeshmukh on 21/03/25.
 * Project Name: OSMOS-Android-SDK
 * File Name: ImageLoader
 */

/**
 * ImageLoader is responsible for asynchronously loading and displaying image ads,
 * including tracking impressions and handling click interactions.
 *
 * @param coroutineScope The CoroutineScope in which image loading and tracking operations are executed.
 * @param adTracker A utility class responsible for tracking ad impressions and clicks.
 */
class ImageLoader(private val coroutineScope: CoroutineScope, private val adTracker: AdTracker) {

    private var viewLoadListener: ((ImageAd, String) -> Unit)? = null

    /**
     * Creates and returns a configured [ImageView] with specified dimensions and density.
     *
     * The resulting ImageView:
     * - Uses FIT_XY scale type to scale the image to fit both width and height exactly.
     * - Has layout parameters set using the given width and height in DP, scaled by screen density.
     * - Is centered using [Gravity.CENTER].
     *
     * @param context The context used to create the ImageView.
     * @param density The screen density used to convert DP to pixels.
     * @param widthDp The desired width of the ImageView in DP.
     * @param heightDp The desired height of the ImageView in DP.
     * @return A new [ImageView] instance with the specified layout and scale settings.
     */
    internal fun createImageView(context: Context, widthDp: Int, heightDp: Int,isClamped: Boolean): ImageView {
        val finalWidth = if (isClamped) widthDp else ViewUtils.dpToPx(context, widthDp)
        val finalHeight = if (isClamped) heightDp else ViewUtils.dpToPx(context, heightDp)

        return ImageView(context).apply {
            layoutParams = LayoutParams(
                finalWidth,
                finalHeight
            ).apply {
                gravity = Gravity.CENTER
            }
            scaleType = ImageView.ScaleType.FIT_CENTER
        }
    }

    /**
     * Loads an image ad (supports static images and GIFs), renders it into the given [ImageView],
     * and handles ad tracking and click interaction.
     *
     * This function supports:
     * - Static image formats (e.g., PNG, JPG)
     * - Animated GIFs via [AnimatedImageDrawable]
     *
     * SVG format is currently not supported.
     *
     * @param imageAd The [ImageAd] containing the image URL.
     * @param imageView The [ImageView] where the ad image will be displayed.
     * @param cliUbid The unique identifier for tracking ad impressions and clicks.
     * @param adClickListener Optional listener triggered when the ad is clicked.
     */
    internal fun loadImage(
        imageAd: ImageAd?,
        imageView: ImageView,
        cliUbid: String,
        isCarouselAd: Boolean? = false,
        adClickListener: ((adMetadata: Map<String, Any>?) -> Unit)? = null
    ) {
        val url = imageAd?.elements?.value?.lowercase()

        coroutineScope.launch(Dispatchers.IO) {
            try {

                if (url != null) {
                    when {
                        url.endsWith(".svg") -> {
                            Log.e("OsmosAdView", "Image format not supported")
                        }

                        url.endsWith(".gif") -> {
                            URL(url).openStream().use { stream ->
                                val bytes = stream.readBytes()
                                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
                                    val source = ImageDecoder.createSource(ByteBuffer.wrap(bytes))
                                    val drawable = ImageDecoder.decodeDrawable(source)

                                    if (drawable is AnimatedImageDrawable) {
                                        drawable.start()
                                    }

                                    withContext(Dispatchers.Main) {
                                        imageView.setImageDrawable(drawable)
                                        handleAdLoaded(
                                            imageAd,
                                            imageView,
                                            cliUbid,
                                            isCarouselAd ?: false,
                                            adClickListener
                                        )
                                    }
                                }
                            }
                        }

                        else -> {
                            URL(url).openStream().use { stream ->
                                val bitmap: Bitmap = BitmapFactory.decodeStream(stream)

                                withContext(Dispatchers.Main) {
                                    imageView.setImageBitmap(bitmap)
                                    handleAdLoaded(
                                        imageAd,
                                        imageView,
                                        cliUbid,
                                        isCarouselAd?:false,
                                        adClickListener
                                    )
                                }
                            }

                        }
                    }
                }
            } catch (e: Exception) {
                Log.e("OsmosAdView", "Image load failed: ${e.message}")
            }
        }
    }


    /**
     * Handles the post-load behavior of an image ad by:
     * - Notifying that the ad has loaded
     * - Tracking ad visibility for impression logging
     * - Setting up click interaction
     *
     * @param imageAd The ad data object.
     * @param imageView The view in which the ad is rendered.
     * @param cliUbid The unique identifier used for tracking this specific ad instance.
     * @param adClickListener Optional callback triggered when the ad is clicked.
     */
    private fun handleAdLoaded(
        imageAd: ImageAd,
        imageView: ImageView,
        cliUbid: String,
        isCarouselAd: Boolean,
        adClickListener: ((adMetadata: Map<String, Any>?) -> Unit)?,
    )
    {
        viewLoadListener?.invoke(imageAd, cliUbid) // Notify listener that the image is loaded
        trackAdVisibility(imageView, imageAd, cliUbid) // Track impression on visibility
        if (!isCarouselAd){
            setupClickListener(imageView, imageAd, cliUbid, adClickListener) // Handle clicks
        }
    }

    /**
     * Sets up a click listener on the ad's ImageView.
     * When clicked, it:
     * - Triggers the adClickListener callback with ad metadata.
     * - Tracks the ad click via the adTracker.
     *
     * @param imageView The ad's visual component.
     * @param imageAd The ad data.
     * @param cliUbid Unique identifier for tracking.
     * @param adClickListener Optional listener invoked on click, providing metadata.
     */
    private fun setupClickListener(
        imageView: ImageView,
        imageAd: ImageAd,
        cliUbid: String,
        adClickListener: ((adMetadata: Map<String, Any>?) -> Unit)?
    ) {
        imageView.setOnClickListener {
            adClickListener?.invoke(imageAd.toMap())
            adTracker.trackAdClick(imageAd.uclid, cliUbid)
        }
    }

    /**
     * Sets a listener to be notified when the ad view is fully loaded.
     *
     * @param listener Callback receiving the loaded [ImageAd] and its tracking ID.
     */
    fun setViewLoadListener(listener: (ImageAd, String) -> Unit) {
        this.viewLoadListener = { imageAd, cliUbid ->
            listener(imageAd, cliUbid)
        }
    }

    /**
     * Tracks ad visibility and triggers impression tracking when the view becomes visible.
     * Uses [ViewTreeObserver.OnPreDrawListener] to detect visibility.
     *
     * @param imageView The view representing the ad.
     * @param imageAd The ad being tracked.
     * @param cliUbid The unique tracking ID for this ad.
     */
    private fun trackAdVisibility(imageView: ImageView, imageAd: ImageAd, cliUbid: String) {
        var preDrawListener: ViewTreeObserver.OnPreDrawListener? = null
        var attachStateListener: View.OnAttachStateChangeListener? = null

        preDrawListener = object : ViewTreeObserver.OnPreDrawListener {
            override fun onPreDraw(): Boolean {
                if (imageView.isVisibleInScreen()) {
                    adTracker.trackImpression(imageAd.uclid, cliUbid)
                    imageView.viewTreeObserver.removeOnPreDrawListener(this)
                    attachStateListener?.let { 
                        imageView.removeOnAttachStateChangeListener(it)
                    }
                }
                return true
            }
        }

        attachStateListener = object : View.OnAttachStateChangeListener {
            override fun onViewDetachedFromWindow(v: View) {
                preDrawListener?.let {
                    imageView.viewTreeObserver.removeOnPreDrawListener(it)
                }
                imageView.removeOnAttachStateChangeListener(this)
            }

            override fun onViewAttachedToWindow(v: View) {
                // No-op
            }
        }

        imageView.addOnAttachStateChangeListener(attachStateListener)
        imageView.viewTreeObserver.addOnPreDrawListener(preDrawListener)
    }
    
    /**
     * Clear the view load listener to prevent memory leaks
     */
    fun clearViewLoadListener() {
        viewLoadListener = null
        Log.d("ImageLoader", "View load listener cleared")
    }
}