package com.ai.osmos.AdRenderSDK

import android.annotation.SuppressLint
import android.content.Context
import android.graphics.Color
import android.graphics.drawable.GradientDrawable
import android.util.AttributeSet
import android.util.Log
import android.view.Gravity
import android.view.MotionEvent
import android.view.View
import android.widget.FrameLayout
import android.widget.ImageButton
import android.widget.ImageView
import androidx.media3.exoplayer.ExoPlayer
import com.ai.osmos.utils.ViewUtils
import com.ai.osmos.R
import kotlin.math.absoluteValue


/**
 * Created by adeshmukh on 17/04/25.
 * Project Name: OSMOS-Android-SDK
 * File Name: InAppPIPView
 */
@SuppressLint("ClickableViewAccessibility", "ViewConstructor")
class InAppPIPView @JvmOverloads constructor(
    context: Context,
    videoAd: VideoAd,
    cliUbid: String,
    private var exoPlayer: ExoPlayer,
    adTracker: AdTracker,
    height: Int,
    width: Int,
    coordinates: Coordinates?,
    adClickListener: ((Map<String, Any>?) -> Unit)? = null,
    attrs: AttributeSet? = null

) : FrameLayout(context, attrs) {

    private lateinit var videoView: View
    private val closeButton: ImageView
    private lateinit var muteButton: ImageButton
    private val fullscreenButton: ImageView
    private val buttonSize = ViewUtils.dpToPx(context,36) // or whatever size works for your design

    private var dX = 0f
    private var dY = 0f
    private var lastTouchDownX = 0f
    private var lastTouchDownY = 0f

    private var isMuted: Boolean = true
    private var onCloseClick: (() -> Unit)? = null
    private var onMuteToggleClick: (() -> Unit)? = null
    private var onFullscreenClick: (() -> Unit)? = null

    init {

        layoutParams = LayoutParams(width, height).apply {
            gravity = Gravity.BOTTOM or Gravity.END
            setMargins(ViewUtils.dpToPx(context,16), ViewUtils.dpToPx(context,16), ViewUtils.dpToPx(context,16), ViewUtils.dpToPx(context,16))
        }.apply {
            if (coordinates != null){
                this@InAppPIPView.x = coordinates.x.toFloat()
                this@InAppPIPView.y = coordinates.y.toFloat()
            }
        }
        background = GradientDrawable().apply {
            cornerRadius = ViewUtils.dpToPx(context,12).toFloat()
            setColor(Color.BLACK)
        }

        closeButton = ViewUtils.createStyledButton(
            context = context,
            iconRes = android.R.drawable.ic_menu_close_clear_cancel,
            size = buttonSize,
            gravity = Gravity.TOP or Gravity.END,
            marginTop = ViewUtils.dpToPx(context,4),
            marginEnd = ViewUtils.dpToPx(context,4),
            onClick = {
                // Don't release ExoPlayer here - let PIPLoader handle it in cleanUp()
                onCloseClick?.invoke()
            }
        )

        muteButton = ViewUtils.createStyledButton(
            context = context,
            iconRes = if (isMuted) R.drawable.mute else R.drawable.unmute,
            size = buttonSize,
            gravity = Gravity.BOTTOM or Gravity.END,
            marginEnd = ViewUtils.dpToPx(context,4),
            marginBottom = ViewUtils.dpToPx(context,4),
            onClick = {
                isMuted = !isMuted
                val actionType = if (isMuted) "mute" else "unmute"
                
                val volumeSet = try {
                    exoPlayer.volume = if (isMuted) 0f else 1f
                    true
                } catch (e: IllegalStateException) {
                    Log.w("InAppPIPView", "Cannot set volume on released player: ${e.message}")
                    false
                }

                if (volumeSet) {
                    muteButton.setImageResource(
                        if (isMuted) R.drawable.mute
                        else R.drawable.unmute
                    )

                    val currentPosition = exoPlayer.currentPosition ?: 0L

                    adTracker.videoActionClick(
                        videoAd.uclid,
                        cliUbid,
                        actionType,
                        (currentPosition / 1000).toInt()
                    )
                    onMuteToggleClick?.invoke()
                }
            }
        )

        fullscreenButton = ViewUtils.createStyledButton(
            context = context,
            iconRes = R.drawable.fullscreen,
            size = buttonSize,
            gravity = Gravity.TOP or Gravity.END,
            marginTop = ViewUtils.dpToPx(context,4),
            marginEnd = ViewUtils.dpToPx(context,50),
            onClick = {
                onFullscreenClick?.invoke()
            }
        )

        addView(closeButton)
        addView(muteButton)
        addView(fullscreenButton)

        setOnTouchListener { view, event ->
            val parent = view.parent as? View ?: return@setOnTouchListener false

            when (event.actionMasked) {
                MotionEvent.ACTION_DOWN -> {
                    dX = event.rawX - view.x
                    dY = event.rawY - view.y
                    lastTouchDownX = event.rawX
                    lastTouchDownY = event.rawY
                    true
                }

                MotionEvent.ACTION_MOVE -> {
                    val newX = event.rawX - dX
                    val newY = event.rawY - dY

                    val maxX = parent.width - view.width
                    val maxY = parent.height - view.height

                    view.x = newX.coerceIn(0f, maxX.toFloat())
                    view.y = newY.coerceIn(0f, maxY.toFloat())

                    true
                }

                MotionEvent.ACTION_UP -> {
                    val deltaX = (event.rawX - lastTouchDownX).absoluteValue
                    val deltaY = (event.rawY - lastTouchDownY).absoluteValue
                    val clickThreshold = 10

                    if (deltaX < clickThreshold && deltaY < clickThreshold) {
                        view.performClick() // triggers setOnClickListener
                    }
                    true
                }
                else -> false
            }
        }

        setOnClickListener {
            adClickListener?.invoke(videoAd.toMap())
            adTracker.trackAdClick(videoAd.uclid, cliUbid)
        }
    }
    fun updateMuteButtonState(isMuted: Boolean) {
        muteButton.setImageResource(if (isMuted) R.drawable.mute else R.drawable.unmute)
        try {
            exoPlayer.volume = if (isMuted) 0f else 1f
        } catch (e: IllegalStateException) {
            Log.w("InAppPIPView", "Cannot update mute state on released player: ${e.message}")
        }
    }

    fun setVideoView(view: View) {
        videoView = view
        videoView.isClickable = false
        videoView.isFocusable = false

        // Enable dragging
        addView(videoView, 0) // Ensure it's below the buttons
    }

    fun setOnCloseClick(listener: () -> Unit) {
        onCloseClick = listener
    }

    fun setOnMuteToggleClick(listener: () -> Unit) {

        onMuteToggleClick = listener
    }

    fun setOnFullscreenClick(listener: () -> Unit) {
        onFullscreenClick = listener
    }

    override fun onDetachedFromWindow() {
        release()
        super.onDetachedFromWindow()
    }

    fun release() {
        try {
            // Clear listener references to prevent memory leaks
            onCloseClick = null
            onMuteToggleClick = null
            onFullscreenClick = null
            
            // Remove video view if attached
            if (::videoView.isInitialized) {
                removeView(videoView)
            }
            
            Log.d("InAppPIPView", "PIP view resources released")
        } catch (e: Exception) {
            Log.e("InAppPIPView", "Error releasing PIP view: ${e.message}")
        }
    }

}