package com.ai.osmos.AdRenderSDK

import android.annotation.SuppressLint
import android.app.Activity
import android.content.Context
import android.util.AttributeSet
import android.view.Gravity
import com.ai.osmos.utils.ConfigManager
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

/**
 * Created by adeshmukh on 09/04/25.
 * Project Name: OSMOS-Android-SDK
 * File Name: InterstitialAdView
 */


data class Coordinates(val x: Int, val y: Int)
/**
 * InterstitialAdView.
 *
 * @param context Application context.
 * @param adData Popup ad Data (defined data structure of ad)
 * @param width(optional): Width of ad view
 * @param height(optional): Height of ad view
 * @param alignment(optional): alignment of ad view ("CENTER" or "BOTTOM")
 * @param coordinates(optional): object containing the X and Y coordinates for absolute positioning of the popup.
 * @param attrs(optional): Attrs for ad view
 * @return InterstitialAdView or EmptyView
 * Example:
 * ```
 * val adView = InterstitialAdView(this@MainActivity,
 *                 adData,
 *                 width = 300,                                     //optional
 *                 height = 200,                                    //optional
 *                 coordinates = Coordinates(x = 10, y = 10),       //optional
 *                 alignment = Gravity.CENTER                       //optional
 *                 )
 * ```
 */

@SuppressLint("ViewConstructor")
class InterstitialAdView @JvmOverloads constructor(
    context: Context,
    adData: Map<String, Any>,
    private val width: Int? = null,
    private val height: Int? = null,
    private val alignment: Int = Gravity.CENTER, // "CENTER", "BOTTOM"
    private var coordinates: Coordinates? = Coordinates(0, 0),
    attrs: AttributeSet? = null
) : BaseAdView(context, attrs) {

    private val adTracker = AdTracker(coroutineScope)
    private val interstitialLoader = InterstitialLoader(coroutineScope, adTracker)

    init {
        if (adData.isNotEmpty()) loadAd(adData,context) else showEmptyView(context)
    }

    override fun onDetachedFromWindow() {
        interstitialLoader.cleanup()
        super.onDetachedFromWindow()
    }

    /**
     * Loads and displays an ad as a popup based on the given ad data.
     *
     * @param adData A map containing the ad object and its associated metadata.
     *               Expected to include a key `"ad"` with value of type `BaseAd`.
     */
    override fun loadAd(adData: Map<String, Any>, context: Context) {
        coroutineScope.launch(Dispatchers.IO) {
            try {
                val ad = adData["ad"] as? BaseAd

                if (ad != null) {
                    val cliUbid = getCliUbid(ad)
                    if (isSupportedCrtImage(ad.crt) || isSupportedCrtVideo(ad.crt)) {
                        withContext(Dispatchers.Main) {
                            when (ad.elementsType) {

                                AdType.IMAGE -> {
                                    val imageAd = ad as? ImageAd

                                    if (!imageAd?.elements?.value.isNullOrEmpty()) {
                                        if (isSupportedImageFormat(ad)) {

                                            val (screenWidth, screenHeight) = getScreenHeightWidth()

                                            val h = height ?: (imageAd?.elements?.height ?: screenHeight)
                                            val w = width ?: (imageAd?.elements?.width ?: screenWidth)

                                            val result = setMaxHeightWidth(h,w,coordinates!!)

                                            coordinates = result.coordinates

                                            val (width, height) = getAdDimensions(
                                                result.finalWidth,
                                                result.finalHeight,
                                                imageAd?.elements?.width ?: screenWidth,
                                                imageAd?.elements?.height ?: screenHeight
                                            )

                                            if (imageAd != null) {
                                                interstitialLoader.showPopup(
                                                    context = context,
                                                    ad = imageAd,
                                                    cliUbid = cliUbid,
                                                    width = width,
                                                    height = height,
                                                    alignment = alignment,
                                                    coordinates = coordinates,
                                                    isClamped = result.isClamped,
                                                    adClickListener = adClickedListener,
                                                    closeButtonSec = imageAd.elements.close_button_sec
                                                            ?: ConfigManager.config.displayAds?.closeButtonSec ?:5
                                                )
                                            } else showEmptyView(context)
                                        } else showEmptyView(context)
                                    } else {
                                        showEmptyView(context)
                                    }
                                }

                                AdType.VIDEO -> {
                                    val videoAd = ad as? VideoAd
                                    if (!videoAd?.elements?.value.isNullOrEmpty()) {
                                        if (isSupportedVideoFormat(ad)) {
                                            val (screenWidth, screenHeight) = getScreenHeightWidth()

                                            val h = height ?: (videoAd?.elements?.height ?: screenHeight)
                                            val w = width ?: (videoAd?.elements?.width ?: screenWidth)

                                            val result = setMaxHeightWidth(h,w,coordinates!!)

                                            coordinates = result.coordinates
                                            val (width, height) = getAdDimensions(
                                                result.finalWidth,
                                                result.finalHeight,
                                                videoAd?.elements?.width ?: screenWidth,
                                                videoAd?.elements?.height ?: screenHeight
                                            )
                                            if (videoAd != null) {
                                                interstitialLoader.showPopup(
                                                    context = context,
                                                    ad = videoAd,
                                                    cliUbid = cliUbid,
                                                    width = width,
                                                    height = height,
                                                    alignment = alignment,
                                                    coordinates = coordinates,
                                                    isClamped = result.isClamped,
                                                    adClickListener = adClickedListener,
                                                    closeButtonSec = videoAd.elements.close_button_sec
                                                        ?: ConfigManager.config.displayAds?.closeButtonSec ?:5
                                                )
                                            } else showEmptyView(context)

                                        } else showEmptyView(context)
                                    } else showEmptyView(context)
                                }
                                else -> {
                                    showEmptyView(context)
                                }
                            }
                        }
                    } else {
                        withContext(Dispatchers.Main) {
                            showEmptyView(context)
                        }
                    }
                } else {
                    withContext(Dispatchers.Main) {
                        showEmptyView(context)
                    }
                }
            } catch (e: Exception) {
                withContext(Dispatchers.Main) {
                    showEmptyView(context)
                }
            }
        }
    }

    override fun loadAd(adData: Map<String, Any>, activity: Activity?) {

    }
}
