package com.ai.osmos.AdRenderSDK

import android.app.Activity
import android.content.Context
import android.graphics.Color
import android.util.Log
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import android.view.WindowManager
import android.widget.FrameLayout
import android.widget.ImageButton
import androidx.lifecycle.LifecycleOwner
import androidx.media3.exoplayer.ExoPlayer
import com.ai.osmos.OsmosSDK
import com.ai.osmos.utils.ViewUtils
import com.ai.osmos.R
import kotlinx.coroutines.CoroutineScope

/**
 * Created by adeshmukh on 17/04/25.
 * Project Name: OSMOS-Android-SDK
 * File Name: PIPLoader
 */

class PIPLoader(
    private val coroutineScope: CoroutineScope,
    private val adTracker: AdTracker
) {
    private var pipContainer: InAppPIPView? = null
    private lateinit var videoView: View
    private var isMuted: Boolean = true
    private var fullscreenContainer: FrameLayout? = null // Container for fullscreen view
    private lateinit var muteButton: ImageButton
    private val videoLoader = VideoLoader(coroutineScope, adTracker)
    private var currentExoPlayer: ExoPlayer? = null
    private var currentLifecycleObserver: PIPPlayerLifecycleObserver? = null
    private var currentActivity: Activity? = null
    fun showPIP(
        activity: Activity,
        ad: VideoAd,
        cliUbid: String,
        width: Int,
        height: Int,
        coordinates: Coordinates? = Coordinates(0, 0),
        isClamped: Boolean,
        adClickListener: ((Map<String, Any>?) -> Unit)?
    ) {


        // Release existing ExoPlayer before creating new one to prevent memory leaks
        currentExoPlayer?.let { existingPlayer ->
            try {
                existingPlayer.release()
            } catch (e: Exception) {
                Log.e("PIPLoader", "Error releasing existing ExoPlayer: ${e.message}")
            }
        }
        
        // Create individual ExoPlayer for this PIP ad to avoid memory leaks
        val exoPlayer = createPlayer(activity)
        currentExoPlayer = exoPlayer

        val finalWidth = if (isClamped) width else ViewUtils.dpToPx(activity,width)
        val finalHeight = if (isClamped) height else ViewUtils.dpToPx(activity,height)

        videoView = videoLoader.createPIPVideoView(
            context = activity,
            videoAd = ad,
            widthDp = finalWidth,
            heightDp = finalHeight,
            cliUbid = cliUbid,
            adClickListener = adClickListener,
            exoPlayer = exoPlayer)

        val pipView = InAppPIPView(
            activity,
            ad,
            cliUbid,
            exoPlayer,
            adTracker,
            finalHeight,
            finalWidth,
            coordinates,
            adClickListener).apply {
            layoutParams = FrameLayout.LayoutParams(
                finalWidth,
                finalHeight
            ).apply {
                if (coordinates != null) {
                    if (coordinates.x != 0 && coordinates.y != 0) {
                        gravity = Gravity.TOP or Gravity.START
                        x = coordinates.x.toFloat()
                        y = coordinates.y.toFloat()
                    } else {
                        gravity = Gravity.BOTTOM or Gravity.END or Gravity.RIGHT
                    }
                }
                setMargins(50, 50, 50, 50)
            }

            setVideoView(videoView)

            setOnCloseClick {
                // Properly clean up ExoPlayer and all resources
                cleanUp()
            }
            setOnMuteToggleClick {
                isMuted = !isMuted

            }
            setOnFullscreenClick {
                showFullscreen(activity, ad, cliUbid, exoPlayer)
            }

        }.apply {
            setBackgroundColor(Color.BLACK)

        }

        pipContainer = pipView

        //Added a observer to handled player play and pause event when app is in background
        if (activity is LifecycleOwner) {
            // Remove existing observer if any
            currentLifecycleObserver?.let { observer ->
                currentActivity?.let { oldActivity ->
                    if (oldActivity is LifecycleOwner) {
                        oldActivity.lifecycle.removeObserver(observer)
                    }
                }
            }
            
            // Create and add new observer
            val lifecycleObserver = PIPPlayerLifecycleObserver(exoPlayer)
            activity.lifecycle.addObserver(lifecycleObserver)
            currentLifecycleObserver = lifecycleObserver
            currentActivity = activity
        }

        // Add it to your current Activity’s root view (must be cast to Activity)
        val decor = activity.window.decorView as ViewGroup
        decor.addView(pipView)
    }

    private fun showFullscreen(activity: Activity, videoAd: VideoAd, cliUbid: String, exoPlayer: ExoPlayer) {
        // Create fullscreen container
        fullscreenContainer = FrameLayout(activity).apply {
            layoutParams = ViewGroup.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT)
            setBackgroundColor(Color.BLACK)
        }

        // Before adding the video view to fullscreen, ensure it's not attached to another parent
        val parent = videoView.parent
        if (parent != null) {
            (parent as? ViewGroup)?.removeView(videoView)
        }
        videoView.layoutParams = FrameLayout.LayoutParams(
            FrameLayout.LayoutParams.MATCH_PARENT,
            FrameLayout.LayoutParams.MATCH_PARENT
        )

        fullscreenContainer?.addView(videoView)

        // Hide PiP view and show fullscreen
        pipContainer?.visibility = View.GONE

        val buttonSize = ViewUtils.dpToPx(activity,36) // or whatever size works for your design

        muteButton = ViewUtils.createStyledButton(
            context = activity,
            iconRes = if (isMuted) R.drawable.mute else R.drawable.unmute,
            size = buttonSize,
            gravity = Gravity.BOTTOM or Gravity.RIGHT,
            marginBottom = ViewUtils.dpToPx(activity,40),
            marginEnd = ViewUtils.dpToPx(activity,10),
            onClick = {
                isMuted = !isMuted
                val actionType = if (isMuted) "mute" else "unmute"
                
                val volumeSet = try {
                    exoPlayer.volume = if (isMuted) 0f else 1f
                    true
                } catch (e: IllegalStateException) {
                    Log.w("PIPLoader", "Cannot set volume on released player: ${e.message}")
                    false
                }

                if (volumeSet) {
                    muteButton.setImageResource(
                        if (isMuted) R.drawable.mute
                        else R.drawable.unmute
                    )

                    val currentPosition = exoPlayer.currentPosition ?: 0L

                    adTracker.videoActionClick(
                        videoAd.uclid,
                        cliUbid,
                        actionType,
                        (currentPosition / 1000).toInt()
                    )
                }
            })

        val closeButton = ViewUtils.createStyledButton(
            context = activity,
            iconRes = android.R.drawable.ic_menu_close_clear_cancel,
            size = buttonSize,
            gravity = Gravity.TOP or Gravity.END,
            marginTop = ViewUtils.dpToPx(activity,30),
            marginEnd = ViewUtils.dpToPx(activity,10),
            onClick = {
                fullscreenContainer?.removeView(videoView)
                pipContainer?.setVideoView(videoView)

                fullscreenContainer?.visibility = View.GONE
                pipContainer?.visibility = View.VISIBLE
            }
        )
        val minimizeButton = ViewUtils.createStyledButton(
            context = activity,
            iconRes = R.drawable.exit_fullscreen,
            size = buttonSize,
            gravity = Gravity.TOP or Gravity.END,
            marginTop = ViewUtils.dpToPx(activity,30),
            marginEnd = ViewUtils.dpToPx(activity,50),
            onClick = {
                fullscreenContainer?.removeView(videoView)
                pipContainer?.setVideoView(videoView)

                pipContainer?.updateMuteButtonState(isMuted)

                fullscreenContainer?.visibility = View.GONE
                pipContainer?.visibility = View.VISIBLE
            }
        )

        updateMuteButtonState(exoPlayer,isMuted)

        fullscreenContainer?.addView(closeButton)
        fullscreenContainer?.addView(muteButton)
        fullscreenContainer?.addView(minimizeButton)

        // Add fullscreen container to the window
        val decor = activity.window.decorView as ViewGroup
        decor.addView(fullscreenContainer)
    }
    fun createPlayer(context: Context): ExoPlayer {
        return ExoPlayer.Builder(context.applicationContext).build()
    }
    fun updateMuteButtonState(exoPlayer: ExoPlayer?, isMuted: Boolean) {
        if (exoPlayer == null) return
        
        muteButton.setImageResource(if (isMuted) R.drawable.mute else R.drawable.unmute)
        try {
            exoPlayer.volume = if (isMuted) 0f else 1f
        } catch (e: IllegalStateException) {
            Log.w("PIPLoader", "Cannot update mute state on released player: ${e.message}")
        }
    }
    private fun removeViewFromWindow(context: Context) {
        pipContainer?.let {
            val windowManager = context.getSystemService(Context.WINDOW_SERVICE) as WindowManager
            windowManager.removeView(it)
        }
        fullscreenContainer?.let {
            val windowManager = context.getSystemService(Context.WINDOW_SERVICE) as WindowManager
            windowManager.removeView(it)
        }
        fullscreenContainer = null
        pipContainer = null
    }

    fun cleanUp() {
        try {
            // Remove lifecycle observer first
            currentLifecycleObserver?.let { observer ->
                currentActivity?.let { activity ->
                    if (activity is LifecycleOwner) {
                        try {
                            activity.lifecycle.removeObserver(observer)
                        } catch (e: Exception) {
                            Log.e("PIPLoader", "Error removing lifecycle observer: ${e.message}")
                        }
                    }
                }
            }
            currentLifecycleObserver = null
            currentActivity = null
            
            currentExoPlayer?.let { player ->
                try {
                    player.stop()
                    player.clearVideoSurface()
                    player.setVideoSurfaceView(null)
                    player.setVideoTextureView(null)
                    player.release()
                } catch (e: Exception) {
                    Log.e("PIPLoader", "Error releasing ExoPlayer: ${e.message}")
                }
            }
            currentExoPlayer = null
            
            // Skip VideoLoader.releaseAllPlayers() for PIP since we manage our own ExoPlayer instance
            // The currentExoPlayer is already properly released above
            
            pipContainer?.let { pip ->
                (pip.parent as? ViewGroup)?.removeView(pip)
                pip.release()
            }
            fullscreenContainer?.let { fullscreen ->
                (fullscreen.parent as? ViewGroup)?.removeView(fullscreen)
            }
            pipContainer = null
            fullscreenContainer = null
        } catch (e: Exception) {
            Log.e("PIPLoader", "Error during cleanup: ${e.message}")
        }
    }
}
