package com.ai.osmos.AdRenderSDK

import android.content.Context
import android.content.res.Resources
import android.graphics.Color
import android.graphics.PorterDuff
import android.graphics.drawable.GradientDrawable
import android.net.Uri
import android.util.Log
import android.view.Gravity
import android.view.SurfaceView
import android.view.TextureView
import android.view.View
import android.view.ViewGroup
import android.view.ViewTreeObserver
import android.widget.FrameLayout
import android.widget.ImageButton
import android.widget.ImageView
import androidx.lifecycle.LifecycleOwner
import androidx.media3.common.MediaItem
import androidx.media3.common.Player
import androidx.media3.exoplayer.ExoPlayer
import androidx.media3.ui.PlayerView
import com.ai.osmos.utils.ConfigManager
import com.ai.osmos.utils.Constants
import com.ai.osmos.utils.ViewUtils
import com.ai.osmos.utils.isVisibleInScreen
import com.ai.osmos.R
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.atomic.AtomicBoolean

/**
 * Created by sshendre on 03/04/25.
 * Project Name: OSMOS-Android-SDK
 * File Name: VideoLoader
 */
class VideoLoader(
    private val coroutineScope: CoroutineScope,
    private val adTracker: AdTracker
) {
    private lateinit var mButton: ImageButton
    private var isViewVisible = false
    private var wasPlaying = false
    private val playerMap: MutableMap<PlayerView, ExoPlayer> = ConcurrentHashMap()
    private val muteStateMap: MutableMap<ExoPlayer, Boolean> = ConcurrentHashMap()
    private val visibilityStateMap: MutableMap<ExoPlayer, Boolean> = ConcurrentHashMap()
    private val wasPlayingStateMap: MutableMap<ExoPlayer, Boolean> = ConcurrentHashMap()
    private val progressJobMap: MutableMap<PlayerView, Job> = ConcurrentHashMap()
    private val lifecycleObserverMap: MutableMap<ExoPlayer, VideoPlayerLifecycleObserver> = ConcurrentHashMap()
    private var onViewLoad: ((adMetadata: Map<String, Any>, String) -> Unit)? = null
    private val isReleased = AtomicBoolean(false)
    /**
     * Sets a listener to be notified when the ad view is fully loaded.
     *
     * @param listener Callback receiving the loaded [ImageAd] and its tracking ID.
     */
    fun setViewLoadListener(listener: (adMetadata: Map<String, Any>, String) -> Unit) {
        this.onViewLoad = { adMetadata, cliUbid ->
            listener(adMetadata, cliUbid)
        }
    }

    internal fun createVideoView(
        context: Context,
        videoAd: VideoAd,
        isClamped: Boolean,
        widthDp: Int,
        heightDp: Int,
        cliUbid: String,
        adClickListener: ((adMetadata: Map<String, Any>?) -> Unit)? = null

    ): FrameLayout {
        val widthPx = if (isClamped) widthDp else ViewUtils.dpToPx(context, widthDp)
        val heightPx = if (isClamped) heightDp else ViewUtils.dpToPx(context, heightDp)
        val exoPlayer: ExoPlayer = ExoPlayer.Builder(context).build()
        val playerView = buildPlayerView(context,widthPx, heightPx)
        
        // Initialize states for this specific player
        muteStateMap[exoPlayer] = true  // default: muted
        visibilityStateMap[exoPlayer] = false  // default: not visible
        wasPlayingStateMap[exoPlayer] = false  // default: not playing
        
        val muteButton = ImageButton(context).apply {
            val currentMuteState = muteStateMap[exoPlayer] ?: false
            setImageResource(if (currentMuteState) R.drawable.mute else R.drawable.unmute)
            setColorFilter(Color.BLACK, PorterDuff.Mode.SRC_IN)
            scaleType = ImageView.ScaleType.FIT_CENTER
            adjustViewBounds = true
            background = GradientDrawable().apply {
                shape = GradientDrawable.OVAL
                setStroke(3, Color.BLACK)
                setColor(Color.WHITE)
            }

            layoutParams = FrameLayout.LayoutParams(
                ViewUtils.dpToPx(context,24),
                ViewUtils.dpToPx(context,24),
                Gravity.END or Gravity.BOTTOM
            ).apply {
                bottomMargin = ViewUtils.dpToPx(context,50)

                marginEnd = ViewUtils.dpToPx(context,16)
                marginStart = ViewUtils.dpToPx(context,16)
            }
            visibility = View.GONE
            setOnClickListener { toggleMute(this, videoAd, cliUbid,exoPlayer) }
        }

        // Load video asynchronously
        loadVideoUsingUrl(context,videoAd, cliUbid, playerView,exoPlayer,muteButton)

        //Add Listeners
        addListeners(videoAd, cliUbid, playerView, adClickListener,exoPlayer)

        return buildContainer(context,widthPx, heightPx, playerView, muteButton)
    }

    internal fun createPIPVideoView(
        context: Context,
        videoAd: VideoAd,
        widthDp: Int,
        heightDp: Int,
        cliUbid: String,
        adClickListener: ((adMetadata: Map<String, Any>?) -> Unit)? = null,
        exoPlayer: ExoPlayer
    ): PlayerView {

        val playerView = buildPlayerView(context,widthDp, heightDp)

        // Initialize states for this specific PIP player
        muteStateMap[exoPlayer] = true  // default: muted for PIP
        visibilityStateMap[exoPlayer] = false  // default: not visible
        wasPlayingStateMap[exoPlayer] = false  // default: not playing

        // Load and play video
        loadVideoUsingUrl(context,videoAd, cliUbid, playerView,exoPlayer)

        addListeners(videoAd, cliUbid, playerView, adClickListener, exoPlayer)

        return playerView
    }

    internal fun createCarouselVideoView(
        context: Context,
        videoAd: VideoAd,
        widthDp: Int,
        heightDp: Int,
        cliUbid: String,
    ): FrameLayout {

        val widthPx = ViewUtils.dpToPx(context,widthDp)
        val heightPx = ViewUtils.dpToPx(context,heightDp)
        val exoPlayer: ExoPlayer = ExoPlayer.Builder(context).build()
        val playerView = buildPlayerView(context,widthPx, heightPx)
        
        // Initialize states for this specific carousel player
        muteStateMap[exoPlayer] = true  // default: muted
        visibilityStateMap[exoPlayer] = false  // default: not visible
        wasPlayingStateMap[exoPlayer] = false  // default: not playing
        
        val buttonSize = ViewUtils.dpToPx(context,36)
        val currentMuteState = muteStateMap[exoPlayer] ?: false
        mButton = ViewUtils.createStyledButton(
            context = context,
            iconRes = if (currentMuteState) R.drawable.mute else R.drawable.unmute,
            size = buttonSize,
            gravity = Gravity.BOTTOM or Gravity.RIGHT,
            marginBottom = ViewUtils.dpToPx(context,40),
            marginEnd = ViewUtils.dpToPx(context,10),
            onClick = {
                toggleMute(mButton, videoAd, cliUbid,exoPlayer)
            })
        mButton.visibility = View.GONE

        // Load video asynchronously
        loadVideoUsingUrl(context,videoAd, cliUbid, playerView,exoPlayer,mButton)

        playerView.viewTreeObserver.addOnScrollChangedListener {
            checkVisibilityAndHandlePlayback(playerView,exoPlayer)
        }

        playerView.viewTreeObserver.addOnPreDrawListener(object :
            ViewTreeObserver.OnPreDrawListener {
            override fun onPreDraw(): Boolean {
                if (playerView.isVisibleInScreen()) {
                    onViewLoad?.invoke(videoAd.toMap(),cliUbid)
                    adTracker.trackImpression(videoAd.uclid, cliUbid)
                    playerView.viewTreeObserver.removeOnPreDrawListener(this)
                }
                return true
            }
        })

        return buildContainer(context,widthPx, heightPx, playerView, mButton)
    }

    private fun buildPlayerView(context: Context,widthPx: Int, heightPx: Int): PlayerView {
        return PlayerView(context).apply {
            layoutParams = FrameLayout.LayoutParams(widthPx, heightPx, Gravity.CENTER)
            useController = false
            setShutterBackgroundColor(Color.TRANSPARENT)

        }
    }

    private fun buildContainer(
        context: Context,
        widthPx: Int,
        heightPx: Int,
        playerView: PlayerView,
        muteButton: View? = null
    ): FrameLayout {
        return FrameLayout(context).apply {
            layoutParams = ViewGroup.LayoutParams(widthPx, heightPx)
            addView(playerView, FrameLayout.LayoutParams(widthPx, heightPx).apply {
                gravity = Gravity.CENTER
            })
            muteButton?.let {
                addView(it, FrameLayout.LayoutParams(90, 90).apply {
                    gravity = Gravity.BOTTOM or Gravity.END
                    bottomMargin = 26
                })
            }
        }
    }


    private fun loadVideoUsingUrl(
        context: Context,
        videoAd: VideoAd,
        cliUbid: String,
        playerView: PlayerView,
        exoPlayer: ExoPlayer,
        muteButton: ImageButton? = null
    ) {
        val loadingJob = coroutineScope.launch(Dispatchers.IO) {  // Fetch video URL in background
            try {
                val videoUrl = videoAd.elements.value
                withContext(Dispatchers.Main) { // Switch to Main thread for ExoPlayer

                    exoPlayer.apply {
                        // Safest approach: Let ExoPlayer handle all format detection automatically
                        val mediaItem = MediaItem.Builder()
                            .setUri(videoUrl)
                            .build()
                        setMediaItem(mediaItem)
                        prepare()

                        addListener(createPlayerListener(videoAd,cliUbid,playerView,exoPlayer,muteButton))
                        val playerMuteState = muteStateMap[exoPlayer] ?: false
                        try {
                            volume = if (playerMuteState) 0f else 1f
                        } catch (e: IllegalStateException) {
                            Log.w("VideoLoader", "Cannot set initial volume on released player: ${e.message}")
                        }
                    }
                    playerMap[playerView] = exoPlayer
                    playerView.player = exoPlayer
                    exoPlayer?.playWhenReady = true
                    
                    // Attach lifecycle observer after player is added to map
                    if (context is LifecycleOwner) {
                        val observer = VideoPlayerLifecycleObserver(exoPlayer)
                        context.lifecycle.addObserver(observer)
                        lifecycleObserverMap[exoPlayer] = observer
                    }
                }

            } catch (e: Exception) {
                Log.e("VideoLoader", "Error loading video: ${e.message}", e)
            }
        }
        
        // Store loading job for cleanup - using a temporary key until progress tracking starts
        progressJobMap[playerView] = loadingJob

    }

    private fun addListeners(
        videoAd: VideoAd,
        cliUbid: String,
        playerView: PlayerView,
        adClickListener: ((adMetadata: Map<String, Any>?) -> Unit)?,
        exoPlayer: ExoPlayer
    ) {
        playerView.setOnClickListener {
            adTracker.trackAdClick(videoAd.uclid, cliUbid)
            adClickListener?.invoke(videoAd.toMap())
        }

        playerView.viewTreeObserver.addOnScrollChangedListener {
            checkVisibilityAndHandlePlayback(playerView,exoPlayer)
        }

        playerView.viewTreeObserver.addOnPreDrawListener(object :
            ViewTreeObserver.OnPreDrawListener {
            override fun onPreDraw(): Boolean {
                if (playerView.isVisibleInScreen()) {
                    onViewLoad?.invoke(videoAd.toMap(),cliUbid)
                    adTracker.trackImpression(videoAd.uclid, cliUbid)
                    playerView.viewTreeObserver.removeOnPreDrawListener(this)
                }
                return true
            }
        })
    }


    private fun createPlayerListener(
        videoAd: VideoAd,
        cliUbid: String,
        playerView: PlayerView,
        exoPlayer: ExoPlayer,
        muteButton: ImageButton? = null,
    ) = object : Player.Listener {

        override fun onPlaybackStateChanged(state: Int) {

            when (state) {
                Player.STATE_IDLE -> {}
                Player.STATE_BUFFERING -> {}
                Player.STATE_READY -> {
                    Log.d(
                        "VideoProgress",
                        "Progress: Start"
                    )
                    if (muteButton != null)
                        muteButton.visibility = View.VISIBLE

                    trackVideoProgressWithEvent(videoAd,cliUbid,playerView,exoPlayer)
                }

                Player.STATE_ENDED -> {
                    if (playerView.isVisibleInScreen()) {
                        val currentPosition = exoPlayer.currentPosition ?: 0L  // Current time (ms)
                        val totalDuration = exoPlayer.duration ?: 0L  // Total duration (ms)

                        Log.d(
                            "VideoProgress",
                            "Progress: End "
                        )

                        fireVideoProgressEvent(
                            videoAd.uclid,
                            cliUbid,
                            currentPosition,
                            totalDuration
                        )
                    }
                    exoPlayer?.seekTo(0)
                    exoPlayer?.playWhenReady = true
                }
            }
        }
    }


    private fun trackVideoProgressWithEvent(
        videoAd: VideoAd,
        cliUbid: String,
        playerView: PlayerView,
        exoPlayer: ExoPlayer
    ) {
        val job = coroutineScope.launch(Dispatchers.Main) {
            val eventProgressTime = videoAd.elements.video_progress_sec
                ?: ConfigManager.config.registerEvent?.videoProgressSec ?: Constants.VIDEO_PROGRESS_SEC_DEFAULT

            var lastFiredSec = -eventProgressTime!!

            while (exoPlayer != null) {
                if (exoPlayer?.isPlaying == true && playerView.isVisibleInScreen()) {
                    val currentPositionMs = exoPlayer?.currentPosition ?: 0L
                    val currentSec = (currentPositionMs / 1000).toInt()

                    if (currentSec >= lastFiredSec + eventProgressTime) {
                        fireVideoProgressEvent(
                            videoAd.uclid,
                            cliUbid,
                            currentPositionMs,
                            exoPlayer?.duration ?: 0L
                        )
                        lastFiredSec = currentSec
                    }
                }
                delay(1000) // Check every second
            }
        }
        
        // Cancel and replace any existing job (like loadingJob) for this playerView
        progressJobMap.remove(playerView)?.cancel()
        progressJobMap[playerView] = job
    }

    private fun fireVideoProgressEvent(
        uclid: String,
        cliUbid: String,
        currentPosition: Long,
        totalDuration: Long
    ) {
        adTracker.videoProgressEvent(
            uclid,
            cliUbid,
            (currentPosition / 1000).toInt(),
            (totalDuration / 1000).toInt()
        )
    }

    private fun toggleMute(
        button: ImageButton,
        videoAd: VideoAd,
        cliUbid: String,
        exoPlayer: ExoPlayer
    ) {
        // Get current mute state for this specific player, default to false (unmuted)
        val currentMuteState = muteStateMap[exoPlayer] ?: false
        val newMuteState = !currentMuteState
        
        // Update mute state for this specific player
        muteStateMap[exoPlayer] = newMuteState
        
        val actionType = if (newMuteState) "mute" else "unmute"
        exoPlayer?.volume = if (newMuteState) 0f else 1f
        button.setImageResource(if (newMuteState) R.drawable.mute else R.drawable.unmute)

        val currentPosition = exoPlayer?.currentPosition ?: 0L

        //Call video action event for mute and unmute
        adTracker.videoActionClick(
            videoAd.uclid,
            cliUbid,
            actionType,
            (currentPosition / 1000).toInt()
        )
    }

    private fun checkVisibilityAndHandlePlayback(playerView: PlayerView, exoPlayer: ExoPlayer) {
        val screenLocation = IntArray(2)
        playerView.getLocationOnScreen(screenLocation)

        val screenHeight = Resources.getSystem().displayMetrics.heightPixels
        val viewTop = screenLocation[1]
        val viewBottom = viewTop + playerView.height

        val isVisibleNow = viewTop < screenHeight && viewBottom > 0
        
        // Get per-player visibility state
        val wasVisible = visibilityStateMap[exoPlayer] ?: false
        val wasPlayerPlaying = wasPlayingStateMap[exoPlayer] ?: false

        if (isVisibleNow && !wasVisible) {
            // Video became visible
            visibilityStateMap[exoPlayer] = true
            if (wasPlayerPlaying) {
                val playerMuteState = muteStateMap[exoPlayer] ?: false
                try {
                    exoPlayer.volume = if (playerMuteState) 0f else 1f
                    exoPlayer.play()
                } catch (e: IllegalStateException) {
                    Log.w("VideoLoader", "Cannot control released player on visible: ${e.message}")
                    // Clean up state for released player
                    muteStateMap.remove(exoPlayer)
                    visibilityStateMap.remove(exoPlayer)
                    wasPlayingStateMap.remove(exoPlayer)
                }
            } else if (exoPlayer.playWhenReady && !exoPlayer.isPlaying) {
                val playerMuteState = muteStateMap[exoPlayer] ?: false
                try {
                    exoPlayer.volume = if (playerMuteState) 0f else 1f
                    wasPlayingStateMap[exoPlayer] = true
                } catch (e: IllegalStateException) {
                    Log.w("VideoLoader", "Cannot set volume on released player: ${e.message}")
                    muteStateMap.remove(exoPlayer)
                    visibilityStateMap.remove(exoPlayer)
                    wasPlayingStateMap.remove(exoPlayer)
                }
            }
        } else if (!isVisibleNow && wasVisible) {
            // Video went out of view
            visibilityStateMap[exoPlayer] = false
            if (exoPlayer.isPlaying == true) {
                try {
                    exoPlayer.pause()
                    // Don't change volume here - preserve individual mute state
                    // The volume should remain as per the player's mute state
                    wasPlayingStateMap[exoPlayer] = true
                } catch (e: IllegalStateException) {
                    Log.w("VideoLoader", "Cannot pause released player: ${e.message}")
                    // Clean up state for released player
                    muteStateMap.remove(exoPlayer)
                    visibilityStateMap.remove(exoPlayer)
                    wasPlayingStateMap.remove(exoPlayer)
                }
            }
        }
    }

    fun safeStopAndReleasePlayer(playerView: PlayerView?) {
        if (isReleased.get()) {
            // Bulk release already happened - don't proceed
            return
        }
        
        playerView?.let { pv ->
            try {
                pv.player?.apply {
                    playWhenReady = false
                    stop()
                    clearVideoSurface()
                }
            } catch (e: IllegalStateException) {
                // Player already released - continue with cleanup
            }
            releasePlayer(pv)
        }
    }

    fun releasePlayer(playerView: PlayerView?) {
        playerView?.let {
            try {
                progressJobMap.remove(it)?.cancel()
                val player = playerMap.remove(it)

                it.player = null
                player?.apply {
                    playWhenReady = false
                    stop()
                    clearVideoSurface()
                    setVideoSurfaceView(null)
                    setVideoTextureView(null)
                    
                    // Clean up all states for this player
                    muteStateMap.remove(this)
                    visibilityStateMap.remove(this)
                    wasPlayingStateMap.remove(this)
                    
                    // Remove lifecycle observer to prevent memory leak
                    lifecycleObserverMap.remove(this)?.let { observer ->
                        // We need context to remove observer, but it's not available here
                        // The observer will be cleaned up when Activity is destroyed
                        // This is a limitation of the current design
                    }
                    
                    release()
                }

            } catch (e: Exception) {
                Log.e("VideoLoader", "releasePlayer failed: ${e.message}")
            }
        }
    }


    fun pause(playerView: PlayerView) {
        val player = playerView.player
        player?.pause()
    }

    fun playExclusive(playerView: PlayerView) {
        // Pause all others
        for ((view, player) in playerMap) {
            if (view != playerView) {
                player.pause()
            }
        }
        // Play the selected one
        playerMap[playerView]?.play()
    }

    /**
     * Clear the view load listener to prevent memory leaks
     */
    private fun clearViewLoadListener() {
        onViewLoad = null
    }


    /**
     * Safe version of releaseAllPlayers that handles dead thread scenarios
     */
    fun safeReleaseAllPlayers() {
        if (isReleased.compareAndSet(false, true)) {
            try {
                releaseAllPlayers()
            } catch (e: IllegalStateException) {
                Log.w("VideoLoader", "Handler thread is dead, performing minimal cleanup")
                // Just clear the maps without UI operations
                try {
                    progressJobMap.clear()
                    playerMap.clear()
                    muteStateMap.clear()
                    visibilityStateMap.clear()
                    wasPlayingStateMap.clear()
                    lifecycleObserverMap.clear()
                    onViewLoad = null
                } catch (e2: Exception) {
                    Log.e("VideoLoader", "Error during minimal cleanup: ${e2.message}")
                }
            } catch (e: Exception) {
                Log.e("VideoLoader", "Error during safe release: ${e.message}")
            }
        }
    }

    private fun releaseAllPlayers() {
        try {
            // Cancel progress jobs with dead thread protection
            try {
                progressJobMap.values.forEach { it.cancel() }
                progressJobMap.clear()
            } catch (e: IllegalStateException) {
                progressJobMap.clear()
            } catch (e: Exception) {
                progressJobMap.clear()
            }

            clearViewLoadListener()

            // Clear player references with dead thread protection
            try {
                playerMap.keys.forEach { playerView ->
                    try {
                        playerView.player = null
                    } catch (e: IllegalStateException) {
                        // Handler thread might be dead, skip UI operation
                        Log.w("VideoLoader", "Handler thread is dead, skipping playerView.player = null")
                    } catch (e: Exception) {
                        Log.e("VideoLoader", "Error setting playerView.player = null: ${e.message}")
                    }
                }
            } catch (e: IllegalStateException) {
                Log.w("VideoLoader", "Handler thread is dead, skipping all playerView operations")
            } catch (e: Exception) {
                Log.e("VideoLoader", "Error iterating playerMap: ${e.message}")
            }

            val playersToRelease = playerMap.toMap()

            playersToRelease.forEach { (playerView, player) ->
                try {
                    playerView.player = null
                    player.playWhenReady = false
                    player.stop()
                    player.clearVideoSurface()
                    player.setVideoSurfaceView(null)
                    player.setVideoTextureView(null)
                    player.release()
                } catch (e: Exception) {
                    Log.e("VideoLoader", "Error releasing individual player: ${e.message}")
                }
            }
            playerMap.clear()
            muteStateMap.clear()  // Clean up all mute states
            visibilityStateMap.clear()  // Clean up all visibility states
            wasPlayingStateMap.clear()  // Clean up all playing states
            lifecycleObserverMap.clear()  // Clean up lifecycle observer references
        } catch (e: Exception) {
            Log.e("VideoLoader", "Error in releaseAllPlayers: ${e.message}")
        }
    }
}