package com.ai.osmos.EventSDK

import android.util.Log
import com.ai.osmos.network.WebConstants
import com.ai.osmos.network.WebService
import com.ai.osmos.utils.ConfigManager
import kotlinx.coroutines.CompletableDeferred
import org.json.JSONException

/**
 * Created by adeshmukh on 20/02/25.
 * Project Name: OSMOS-Android-SDK
 * File Name: RegisterEvent
 */

/**
 * Handles event tracking for different user actions such as product views,
 * add-to-cart, purchases, ad impressions, and ad clicks.
 */
public class RegisterEvent {

    // Instance of WebService to handle network requests
    private val webService = WebService()

    /**
     * Represents a product with relevant details for tracking events.
     *
     * @property sku_id Unique identifier of the product.
     * @property product_prices Optional: Price details of the product.
     * @property product_quantities Optional: Quantity details of the product.
     * @property merchant_id Optional: Identifier of the merchant selling the product.
     *
     *
     */
    data class Product(
        val sku_id: String,
        val product_prices: String?,
        val product_quantities: String?,
        val merchant_id: String? = null // Optional
    )


    /**
     * Processes event tracking by making an API call to the specified URL with event parameters.
     * This function constructs the full API URL using `WebConstants.buildUrl()` and sends
     * a POST/GET request via `webService.callApi()`. If debugging is enabled, the parameters
     * and response are logged for debugging purposes.
     *
     * @param urlPath The endpoint URL path where the event data will be sent.
     * @param params A map containing the event tracking parameters.
     * @return The API response as a Map<String,Any>.
     *
     */
    private suspend fun trackEventsProcessor(
        urlPath: String,
        params: Map<String, Any>
    ): Map<String, Any> {
        val deferredResponse = CompletableDeferred<Map<String,Any>>()

        if (ConfigManager.isDebugEnabled()) {
            // Log event parameters for debugging
            Log.d("OSMOS", "OSMOS = $params")
        }

        // Construct the API URL with the given parameters
        val apiUrl = WebConstants.buildUrl(urlPath, params)

        // Make the API call to track the event
        webService.callApi(apiUrl, "POST", null, requestBody = null) { response ->
            deferredResponse.complete(response) // Complete with the response
        }

        return deferredResponse.await() // Suspend until response is available
    }

    /**
     * Registers an ad click event for tracking user interactions with advertisements.
     * This function constructs a query parameter map containing the `uclid` and any additional parameters.
     * It then calls `track_events_processor` to send the event to the ad click tracking endpoint.
     *
     * @param cliUbid Unique identifier.
     * @param uclid Unique identifier for the ad click event.
     * @param extraParams Optional: Additional tracking parameters to be merged with the event data.
     * @return The API response as a Map<String,Any>.
     *
     * Example:
     * ```
     * val res = registerEvent.registerAdClickEvent(
     *             cliUbid = "c27b9ad197765dc9ba51e5b7fb9d5c43eb8f1f7b199367af54c74705424558dd",
     *             uclid = "2|haie5ibhks4fkd3a3a36oy9zp23i8ton|0.0001|1680792159968|2777510___Astore___Anet|BRAND_AD_TAG|c27b9ad197765dc9ba51e5b7fb9d5c43eb8f1f7b199367af54c74705424558dd|CPM|whitakers|522130|411094|437705|437705||15211|8816|",
     *             extraParams = mapOf("seller_id" to "whitakers")
     *         )
     * ```
     */
    public suspend fun registerAdClickEvent(
        cliUbid: String,
        uclid: String,
        extraParams: Map<String, Any>? = null
    ): Map<String, Any> {
        val mergedParams = mutableMapOf<String, Any>().apply {
            put("uclid", uclid)
            extraParams?.let { putAll(it) }
        }

        val queryParams = getQueryParam(
            cliUbid = cliUbid,
            eventName = "ad_click", // Used as event_name is not available for ad_click
            extraParams = mergedParams
        )

        try {
            val baseUrl = ConfigManager.config.registerEvent?.domainUri
                ?: throw JSONException("register event parameter")
            return track_events_processor("$baseUrl/aclick", queryParams)

        }
        catch(e: JSONException) {
            val error = """
            { 
                "status": false,
                "response": null,
                "error": {
                    "code": "ERROR_CONFIGURATION",
                    "description": "Register event parameters not configured at time of SDK initialization"
                }
            }
            """
            Log.e("OSMOS", "OSMOS:$error")
        }
        return emptyMap()
    }


    /**
     * Registers a generic tracking event.
     * This function constructs query parameters containing event details and sends them
     * to the tracking endpoint via `track_events_processor`.
     *
     * @param cliUbid Unique identifier
     * @param eventName The name of the event being tracked.
     * @param extraParams Additional parameters related to the event, providing more context.
     * @return The API response as a string.
     *
     */
    private suspend fun registerTrackingEvent(
        cliUbid: String,
        eventName: String,
        extraParams: Map<String, Any>
    ): Map<String, Any> {

        val queryParams = getQueryParam(
            cliUbid = cliUbid,
            eventName = eventName,
            extraParams = extraParams
        )
        try {
            val baseUrl = ConfigManager.config.registerEvent?.domainUri
                ?: throw JSONException("register event parameter")
            return  track_events_processor("$baseUrl/events", queryParams)

        }
        catch(e: JSONException) {
            val error = """
            {
                "status": false,
                "response": null,
                "error": {
                    "code": "ERROR_CONFIGURATION",
                    "description": "Register event parameters not configured at time of SDK initialization"
                }
            }
            """
            Log.e("OSMOS", "OSMOS:$error")
        }
        return emptyMap()
    }

    /**
     * Registers a "view product" event for tracking user interactions.
     * This function constructs an event payload and calls `registerViewProductEvent`
     * to log the "viewproduct" event with the necessary parameters.
     *
     * @param cliUbid Unique identifier.
     * @param skuId The unique identifier of the product being viewed.
     * @param productPrices Optional: Prices of the product, if applicable.
     * @param merchantId Optional: Identifier of the merchant selling the product.
     * @param extraParams Optional: Additional tracking parameters to be merged with event data.
     * @return The API response as a string.
     *
     * Example:
     * ```
     * val res = registerEvent.registerViewProductEvent(
     *             cliUbid = "dfbfee0554bf1de6dfe801dc32d91ab9f15c534ee396a00e7fcbc5930769d109",
     *             skuId = "2777510___Astore___Anet",
     *             productPrices = "85",
     *             merchantId = "whitakers",
     *             extraParams = mapOf("product_quantities" to "1",
     *                 "store_id" to "1")
     *         )
     * ```
     *
     */
    public suspend fun registerViewProductEvent(
        cliUbid: String,
        skuId: String,
        productPrices: String? = null,
        merchantId: String? = null,
        extraParams: Map<String, Any>? = null
    ): Map<String, Any> {
        val eventName = "viewproduct"

        val eventParams = mapOf(
            "sku_id" to skuId
        ) + listOfNotNull(
            productPrices?.let { "product_prices" to it },
            merchantId?.let { "seller_id" to it }
        ).toMap()

        return  registerTrackingEvent(cliUbid, eventName, addExtraParams(extraParams, eventParams))
    }

    /**
     * Registers an "add to cart" event for tracking user interactions.
     * This function constructs an event payload and calls `registerAdd2CartEvent`
     * to log the "funnel_add2cart" event with the necessary parameters.
     *
     * @param cliUbid Unique identifier.
     * @param skuId The unique identifier of the product being added to the cart.
     * @param productPrices Optional: The price of the product, if available.
     * @param merchantId Optional: The identifier of the merchant selling the product.
     * @param extraParams Optional: Additional tracking parameters to be merged with event data.
     * @return The API response as a string.
     *
     * Example:
     * ```
     * val res = registerEvent.registerAdd2CartEvent(
     *             cliUbid = "dfbfee0554bf1de6dfe801dc32d91ab9f15c534ee396a00e7fcbc5930769d109",
     *             skuId = "2777503___Astore___Anet",
     *             productPrices = "95",
     *             merchantId = "whitakers",
     *             extraParams = mapOf(
     *                 "product_quantities" to "1",
     *                 "store_id" to "1")
     *         )
     * ```
     *
     */
    public suspend fun registerAdd2CartEvent(
        cliUbid: String,
        skuId: String,
        productPrices: String? = null,
        merchantId: String? = null,
        extraParams: Map<String, Any>? = null
    ): Map<String, Any> {
        val eventName = "funnel_add2cart"

        val eventParams = mapOf(
            "sku_id" to skuId
        ) + listOfNotNull(
            productPrices?.let { "product_prices" to it },
            merchantId?.let { "seller_id" to it }
        ).toMap()

        return registerTrackingEvent(cliUbid, eventName,  addExtraParams(extraParams, eventParams))
    }

    /**
     * Registers a "purchase" event for tracking completed sales.
     * This function constructs an event payload by aggregating product details such as
     * SKU IDs, prices, quantities, and merchant IDs. It then calls `registerTrackingEvent`
     * to log the "funnel_salecomplete" event.
     *
     * @param cliUbid Unique identifier.
     * @param sales The total number of sales associated with the purchase.
     * @param orderId The unique identifier for the order (item identifier).
     * @param products A list of `Product` objects representing purchased items.
     * @param extraParams Optional: Additional tracking parameters to be merged with event data.
     * @return The API response as a string.
     *
     * Example:
     * ```
     * val res = registerEvent.registerPurchaseEvent(
     *             cliUbid = "dfbfee0554bf1de6dfe801dc32d91ab9f15c534ee396a00e7fcbc5930769d109",
     *             sales = 100,
     *             orderId = "11082669",
     *             products = listOf(
     *                 RegisterEvent.Product(
     *                     product_prices = "50",
     *                     product_quantities = "1",
     *                     sku_id = "2777510___Astore___Anet",
     *                     merchant_id = "whitakers"
     *                 ),
     *                 RegisterEvent.Product(
     *                     product_prices = "50",
     *                     product_quantities = "2",
     *                     sku_id = "2777503___Astore___Anet",
     *                     merchant_id = "whitakers"
     *                 )
     *             ),
     *             extraParams = mapOf(
     *                 "store_id" to "1",
     *                 "source" to "online"
     *             )
     *         )
     * ```
     */
    public suspend fun registerPurchaseEvent(
        cliUbid: String,
        sales: Int,
        orderId: String,
        products: List<Product>,
        extraParams: Map<String, Any>? = null
    ): Map<String, Any> {

        val eventName = "funnel_salecomplete"

        // Extract values from the product list
        val productPrices = products.mapNotNull { it.product_prices }.joinToString("|")
        val productQuantities = products.mapNotNull { it.product_quantities }.joinToString("|")
        val skuIds = products.map { it.sku_id }.joinToString("|")

        // Handle optional merchant_id
        val merchantIds = products.mapNotNull { it.merchant_id }
            .takeIf { it.isNotEmpty() }?.joinToString("|")

        // Prepare event params
        val eventParams = mutableMapOf(
            "sales" to sales,
            "item_identifier" to orderId,
            "product_prices" to productPrices,
            "product_quantities" to productQuantities,
            "sku_id" to skuIds
        )

        // Only add `seller_id` if there are valid merchant_ids
        merchantIds?.let { eventParams["seller_id"] = it }

        return registerTrackingEvent(cliUbid, eventName, addExtraParams(extraParams, eventParams))
    }

    /**
     * Registers an "ad impression" event for tracking ad views.
     * This function constructs an event payload and calls `registerTrackingEvent`
     * to log the "funnel_impression" event, capturing ad view interactions.
     *
     * @param cliUbid Unique identifier.
     * @param uclid Unique identifier for the ad impression.
     * @param position Optional: The position of the ad in the UI.
     * @param extraParams Optional: Additional tracking parameters to be merged with event data.
     * @return The API response as a string.
     *
     * Example:
     * ```
     * val res = registerEvent.registerAdImpressionEvent(
     *             cliUbid = "c27b9ad197765dc9ba51e5b7fb9d5c43eb8f1f7b199367af54c74705424558dd",
     *             uclid = "2|haie5ibhks4fkd3a3a36oy9zp23i8ton|0.0001|1680792159968|2777510___Astore___Anet|BRAND_AD_TAG|c27b9ad197765dc9ba51e5b7fb9d5c43eb8f1f7b199367af54c74705424558dd|CPM|whitakers|522130|411094|437705|437705||15211|8816|",
     *             position = 1,
     *             extraParams = mapOf()
     *         )
     * ```
     *
     */
    public suspend fun registerAdImpressionEvent(
        cliUbid: String,
        uclid: String,
        position: Int? = null,
        extraParams: Map<String, Any>? = null
    ): Map<String, Any> {
        val eventName = "funnel_impression"

        val eventParams = mutableMapOf<String, Any>(
            "uclid" to uclid
        ).apply {
            position?.let { put("position", it) } // Only add if not null
        }

        return registerTrackingEvent(cliUbid, eventName, addExtraParams(extraParams, eventParams))
    }

    /**
     * Registers a "video progress" event for tracking user engagement.
     * This function logs the "funnel_video_progress" event to track how much of a video was watched.
     *
     * @param cliUbid Unique identifier.
     * @param uclid Unique identifier for the video being tracked.
     * @param videoViewSec Number of seconds the video has been viewed.
     * @param videoDurationSec Total duration of the video in seconds.
     * @param extraParams Optional: Additional tracking parameters.
     * @return The API response as a string.
     *
     * Example:
     * ```
     * val res = registerEvent.registerVideoProgressEvent(
     *             cliUbid = "c27b9ad197765dc9ba51e5b7fb9d5c43eb8f1f7b199367af54c74705424558dd",
     *             uclid = "2",
     *             videoViewSec = 10,
     *             videoDurationSec = 10,
     *             extraParams = mapOf()
     *         )
     * ```
     *
     */
    public suspend fun registerVideoProgressEvent(
        cliUbid: String,
        uclid: String,
        videoViewSec: Int,
        videoDurationSec: Int,
        extraParams: Map<String, Any>? = null
    ): Map<String, Any> {

        val eventName = "funnel_video_progress"

        val eventParam = mutableMapOf(
            "uclid" to uclid,
            "video_view_sec" to videoViewSec,
            "video_duration_sec" to videoDurationSec
        )

        return registerTrackingEvent(cliUbid, eventName, addExtraParams(extraParams, eventParam))
    }

    /**
     * Registers a "video action" event to track user interactions with a video.
     * This function logs the "funnel_video_action" event, capturing user interactions
     * such as muting or unmuting a video at a specific timestamp.
     *
     * @param cliUbid Unique identifier.
     * @param uclid Unique identifier for the video being tracked.
     * @param videoViewSec The timestamp (in seconds) at which the action occurred.
     * @param actionType The type of action performed on the video (e.g., "mute", "unmute").
     * @param extraParams Optional: Additional tracking parameters to be merged with event data.
     * @return The API response as a string.
     *
     *
     * Example:
     * ```
     * val res = registerEvent.registerVideoActionEvent(
     *             cliUbid = "c27b9ad197765dc9ba51e5b7fb9d5c43eb8f1f7b199367af54c74705424558dd",
     *             uclid = "2",
     *             videoViewSec = 10,
     *             actionType = "mute",
     *             extraParams = mapOf()
     *         )
     * ```
     */
    public suspend fun registerVideoActionEvent(
        cliUbid: String,
        uclid: String,
        videoViewSec: Int,
        actionType: String, // e.g., "mute" or "unmute"
        extraParams: Map<String, Any>? = null
    ): Map<String, Any> {

        val eventName = "funnel_video_action"

        val eventParam = mutableMapOf(
            "uclid" to uclid,
            "video_view_sec" to videoViewSec,
            "action_type" to actionType
        )
        return registerTrackingEvent(cliUbid, eventName, addExtraParams(extraParams, eventParam))
    }

    /**
     * Processes event tracking by making an API call to the specified URL with event parameters.
     * This function constructs the full API URL using `WebConstants.buildUrl()` and sends
     * a POST/GET request via `webService.callApi()`. If debugging is enabled, the parameters
     * and response are logged for debugging purposes.
     *
     * @param url_path The endpoint URL path where the event data will be sent.
     * @param params A map containing the event tracking parameters.
     * @return The API response as a Map<String,Any>.
     *
     */
    @Deprecated(
        message = "Use trackEventsProcessor instead",
        replaceWith = ReplaceWith("trackEventsProcessor(urlPath, params)"),
        level = DeprecationLevel.WARNING
    )
    private suspend fun track_events_processor(
        url_path: String,
        params: Map<String, Any>
    ) : Map<String, Any> {
        return trackEventsProcessor(
            urlPath = url_path,
            params = params
        )
    }

    /**
     * Registers an ad click event for tracking user interactions with advertisements.
     * This function constructs a query parameter map containing the `uclid` and any additional parameters.
     * It then calls `track_events_processor` to send the event to the ad click tracking endpoint.
     *
     * @param cli_ubid Unique identifier.
     * @param uclid Unique identifier for the ad click event.
     * @param extra_params Optional: Additional tracking parameters to be merged with the event data.
     * @return The API response as a Map<String,Any>.
     *
     * Example:
     * ```
     * val res = registerEvent.register_adclick_event(
     *             cli_ubid = "c27b9ad197765dc9ba51e5b7fb9d5c43eb8f1f7b199367af54c74705424558dd",
     *             uclid = "2|haie5ibhks4fkd3a3a36oy9zp23i8ton|0.0001|1680792159968|2777510___Astore___Anet|BRAND_AD_TAG|c27b9ad197765dc9ba51e5b7fb9d5c43eb8f1f7b199367af54c74705424558dd|CPM|whitakers|522130|411094|437705|437705||15211|8816|",
     *             extra_params = mapOf("seller_id" to "whitakers")
     *         )
     * ```
     */
    @Deprecated(
        message = "Use registerAdClickEvent instead",
        replaceWith = ReplaceWith("registerAdClickEvent(cliUbid, uclid, extraParams)"),
        level = DeprecationLevel.WARNING
    )
    public suspend fun register_adclick_event(
        cli_ubid: String,
        uclid: String,
        extra_params: Map<String, Any>? = null
    ) : Map<String,Any> {
        return registerAdClickEvent(
            cliUbid = cli_ubid,
            uclid = uclid,
            extraParams = extra_params
        )
    }


    /**
     * Registers a generic tracking event.
     * This function constructs query parameters containing event details and sends them
     * to the tracking endpoint via `track_events_processor`.
     *
     * @param cli_ubid Unique identifier
     * @param event_name The name of the event being tracked.
     * @param extra_params Additional parameters related to the event, providing more context.
     * @return The API response as a string.
     *
     */
    @Deprecated(
        message = "Use registerTrackingEvent instead",
        replaceWith = ReplaceWith("registerTrackingEvent(cliUbid, eventName, extraParams)"),
        level = DeprecationLevel.WARNING
    )
    private suspend fun register_tracking_event(
        cli_ubid: String,
        event_name: String,
        extra_params: Map<String, Any>
    ) : Map<String,Any> {

        return registerTrackingEvent(
            cliUbid = cli_ubid,
            eventName = event_name,
            extraParams = extra_params
        )
    }

    /**
     * Registers a "view product" event for tracking user interactions.
     * This function constructs an event payload and calls `register_tracking_event`
     * to log the "viewproduct" event with the necessary parameters.
     *
     * @param cli_ubid Unique identifier.
     * @param sku_id The unique identifier of the product being viewed.
     * @param product_prices Optional: Prices of the product, if applicable.
     * @param merchant_id Optional: Identifier of the merchant selling the product.
     * @param extra_params Optional: Additional tracking parameters to be merged with event data.
     * @return The API response as a string.
     *
     * Example:
     * ```
     * val res = registerEvent.register_viewproduct_event(
     *             cli_ubid = "dfbfee0554bf1de6dfe801dc32d91ab9f15c534ee396a00e7fcbc5930769d109",
     *             sku_id = "2777510___Astore___Anet",
     *             product_prices = "85",
     *             merchant_id = "whitakers",
     *             extra_params = mapOf("product_quantities" to "1",
     *                 "store_id" to "1")
     *         )
     * ```
     *
     */
    @Deprecated(
        message = "Use registerViewProductEvent instead",
        replaceWith = ReplaceWith("registerViewProductEvent(cliUbid, skuId, productPrices, merchantId, extraParams)"),
        level = DeprecationLevel.WARNING
    )
    public suspend fun register_viewproduct_event(
        cli_ubid: String,
        sku_id: String,
        product_prices: String? = null, //optional parameter
        merchant_id: String? = null, //optional parameter
        extra_params: Map<String, Any>? = null //optional parameter
    ) : Map<String,Any>  {

        return registerViewProductEvent(
            cliUbid = cli_ubid,
            skuId = sku_id,
            productPrices = product_prices,
            merchantId = merchant_id,
            extraParams = extra_params
        )
    }

    /**
     * Registers an "add to cart" event for tracking user interactions.
     * This function constructs an event payload and calls `register_tracking_event`
     * to log the "funnel_add2cart" event with the necessary parameters.
     *
     * @param cli_ubid Unique identifier.
     * @param sku_id The unique identifier of the product being added to the cart.
     * @param product_prices Optional: The price of the product, if available.
     * @param merchant_id Optional: The identifier of the merchant selling the product.
     * @param extra_params Optional: Additional tracking parameters to be merged with event data.
     * @return The API response as a string.
     *
     * Example:
     * ```
     * val res = registerEvent.register_add2cart_event(
     *             cli_ubid = "dfbfee0554bf1de6dfe801dc32d91ab9f15c534ee396a00e7fcbc5930769d109",
     *             sku_id = "2777503___Astore___Anet",
     *             product_prices = "95",
     *             merchant_id = "whitakers",
     *             extra_params = mapOf(
     *                 "product_quantities" to "1",
     *                 "store_id" to "1")
     *         )
     * ```
     *
     */
    @Deprecated(
        message = "Use registerAdd2CartEvent instead",
        replaceWith = ReplaceWith("registerAdd2CartEvent(cliUbid, skuId, productPrices, merchantId, extraParams)"),
        level = DeprecationLevel.WARNING
    )
    public suspend fun register_add2cart_event(
        cli_ubid: String,
        sku_id: String,
        product_prices: String? = null, //Optional
        merchant_id: String? =null,  //Optional
        extra_params: Map<String, Any>? = null  //optional
    ) : Map<String,Any>  {

        return registerAdd2CartEvent(
            cliUbid = cli_ubid,
            skuId = sku_id,
            productPrices = product_prices,
            merchantId = merchant_id,
            extraParams = extra_params
        )
    }

    /**
     * Registers a "purchase" event for tracking completed sales.
     * This function constructs an event payload by aggregating product details such as
     * SKU IDs, prices, quantities, and merchant IDs. It then calls `register_tracking_event`
     * to log the "funnel_salecomplete" event.
     *
     * @param cli_ubid Unique identifier.
     * @param sales The total number of sales associated with the purchase.
     * @param order_id The unique identifier for the order (item identifier).
     * @param products A list of `Product` objects representing purchased items.
     * @param extra_params Optional: Additional tracking parameters to be merged with event data.
     * @return The API response as a string.
     *
     * Example:
     * ```
     * val res = registerEvent.register_purchase_event(
     *             cli_ubid = "dfbfee0554bf1de6dfe801dc32d91ab9f15c534ee396a00e7fcbc5930769d109",
     *             sales = 100,
     *             order_id = "11082669",
     *             products = listOf(
     *                 RegisterEvent.Product(
     *                     product_prices = "50",
     *                     product_quantities = "1",
     *                     sku_id = "2777510___Astore___Anet",
     *                     merchant_id = "whitakers"
     *                 ),
     *                 RegisterEvent.Product(
     *                     product_prices = "50",
     *                     product_quantities = "2",
     *                     sku_id = "2777503___Astore___Anet",
     *                     merchant_id = "whitakers"
     *                 )
     *             ),
     *             extra_params = mapOf(
     *                 "store_id" to "1",
     *                 "source" to "online"
     *             )
     *         )
     * ```
     */
    @Deprecated(
        message = "Use registerPurchaseEvent instead",
        replaceWith = ReplaceWith("registerPurchaseEvent(cliUbid, sales, orderId, products, extraParams)"),
        level = DeprecationLevel.WARNING
    )
    public suspend fun register_purchase_event(
        cli_ubid: String,
        sales: Int,
        order_id: String, // item_identifier
        products: List<Product>, // List of Product objects instead of Map
        extra_params: Map<String, Any>? = null
    ) : Map<String,Any> {

        return registerPurchaseEvent(
            cliUbid = cli_ubid,
            sales = sales,
            orderId = order_id,
            products = products,
            extraParams = extra_params
        )
    }

    /**
     * Registers an "ad impression" event for tracking ad views.
     * This function constructs an event payload and calls `register_tracking_event`
     * to log the "funnel_impression" event, capturing ad view interactions.
     *
     * @param cli_ubid Unique identifier.
     * @param uclid Unique identifier for the ad impression.
     * @param position Optional: The position of the ad in the UI.
     * @param extra_params Optional: Additional tracking parameters to be merged with event data.
     * @return The API response as a string.
     *
     * Example:
     * ```
     * val res = registerEvent.register_adimpression_event(
     *             cli_ubid = "c27b9ad197765dc9ba51e5b7fb9d5c43eb8f1f7b199367af54c74705424558dd",
     *             uclid = "2|haie5ibhks4fkd3a3a36oy9zp23i8ton|0.0001|1680792159968|2777510___Astore___Anet|BRAND_AD_TAG|c27b9ad197765dc9ba51e5b7fb9d5c43eb8f1f7b199367af54c74705424558dd|CPM|whitakers|522130|411094|437705|437705||15211|8816|",
     *             position = 1,
     *             extra_params = mapOf()
     *         )
     * ```
     *
     */
    @Deprecated(
        message = "Use registerAdImpressionEvent instead",
        replaceWith = ReplaceWith("registerAdImpressionEvent(cliUbid, uclid, position, extraParams)"),
        level = DeprecationLevel.WARNING
    )
    public suspend fun register_adimpression_event(
        cli_ubid: String,
        uclid: String,
        position: Int? = null, // Optional
        extra_params: Map<String, Any>? = null // Optional
    ) : Map<String,Any>  {

        return registerAdImpressionEvent(
            cliUbid = cli_ubid,
            uclid = uclid,
            position = position,
            extraParams = extra_params
        )
    }

    /**
     * Registers a "video progress" event for tracking user engagement.
     * This function logs the "funnel_video_progress" event to track how much of a video was watched.
     *
     * @param cli_ubid Unique identifier.
     * @param uclid Unique identifier for the video being tracked.
     * @param video_view_sec Number of seconds the video has been viewed.
     * @param video_duration_sec Total duration of the video in seconds.
     * @param extra_params Optional: Additional tracking parameters.
     * @return The API response as a string.
     *
     * Example:
     * ```
     * val res = registerEvent.register_video_progress_event(
     *             cli_ubid = "c27b9ad197765dc9ba51e5b7fb9d5c43eb8f1f7b199367af54c74705424558dd",
     *             uclid = "2",
     *             video_view_sec = 10,
     *             video_duration_sec = 10,
     *             extra_params = mapOf()
     *         )
     * ```
     *
     */
    @Deprecated(
        message = "Use registerVideoProgressEvent instead",
        replaceWith = ReplaceWith("registerVideoProgressEvent(cliUbid, uclid, videoViewSec, videoDurationSec, extraParams)"),
        level = DeprecationLevel.WARNING
    )
    public suspend fun register_video_progress_event(
        cli_ubid: String,
        uclid: String,
        video_view_sec: Int,
        video_duration_sec: Int,
        extra_params: Map<String, Any>? = null // Optional
    ) : Map<String,Any>  {

        return registerVideoProgressEvent(
            cliUbid = cli_ubid,
            uclid = uclid,
            videoViewSec = video_view_sec,
            videoDurationSec = video_duration_sec,
            extraParams = extra_params
        )
    }

    /**
     * Registers a "video action" event to track user interactions with a video.
     * This function logs the "funnel_video_action" event, capturing user interactions
     * such as muting or unmuting a video at a specific timestamp.
     *
     * @param cli_ubid Unique identifier.
     * @param uclid Unique identifier for the video being tracked.
     * @param video_view_sec The timestamp (in seconds) at which the action occurred.
     * @param action_type The type of action performed on the video (e.g., "mute", "unmute").
     * @param extra_params Optional: Additional tracking parameters to be merged with event data.
     * @return The API response as a string.
     *
     *
     * Example:
     * ```
     * val res = registerEvent.register_video_action_event(
     *             cli_ubid = "c27b9ad197765dc9ba51e5b7fb9d5c43eb8f1f7b199367af54c74705424558dd",
     *             uclid = "2",
     *             video_view_sec = 10,
     *             action_type = "mute",
     *             extra_params = mapOf()
     *         )
     * ```
     */
    @Deprecated(
        message = "Use registerVideoActionEvent instead",
        replaceWith = ReplaceWith("registerVideoActionEvent(cliUbid, uclid, videoViewSec, actionType, extraParams)"),
        level = DeprecationLevel.WARNING
    )
    public suspend fun register_video_action_event(
        cli_ubid: String,
        uclid: String,
        video_view_sec: Int,
        action_type: String, //mute or unmute
        extra_params: Map<String, Any>? = null // Optional
    ) : Map<String,Any>  {

        return registerVideoActionEvent(
            cliUbid = cli_ubid,
            uclid = uclid,
            videoViewSec = video_view_sec,
            actionType = action_type,
            extraParams = extra_params
        )
    }


    fun currentTime(): Long {
        return System.currentTimeMillis()
    }

    /**
     * Generates query parameters for API calls based on the event type.
     *
     * @param cli_ubid Unique device identifier.
     * @param event_name Name of the event.
     * @param extra_params Additional parameters for the event.
     * @return MutableMap containing all query parameters.
     */
    private fun getQueryParam(
        cliUbid: String,
        eventName: String,
        extraParams: Map<String, Any>
    ): MutableMap<String, Any> {

        val query_params = mutableMapOf<String, Any>(
            "client_id" to ConfigManager.getString("client_id").toString(),
            "cli_ubid" to cliUbid,
            "event_time" to currentTime()
        ).apply {
            if (eventName != "ad_click") {
                put("event_name", eventName) // Only include event_name if not "ad_click"
            }
        }

        val param_mappings = mapOf(
            "viewproduct" to listOf("product_prices", "sku_id", "seller_id", "product_quantities"),
            "funnel_add2cart" to listOf("product_prices", "sku_id", "seller_id", "product_quantities"),
            "funnel_salecomplete" to listOf("product_prices", "product_quantities", "sku_id", "seller_id", "sales", "item_identifier"),
            "funnel_impression" to listOf("uclid", "position"),
            "ad_click" to listOf("uclid","seller_id"),
            "funnel_video_progress" to listOf("uclid","video_view_sec","video_duration_sec"),
            "funnel_video_action" to listOf("uclid","video_view_sec","action_type")
        )

        for (key in param_mappings[eventName] ?: emptyList()) {
            if (key == "product_quantities" && key !in extraParams) {
                if (ConfigManager.isDebugEnabled()){
                    Log.d("OSMOS", "OSMOS: Default product_quantities `1` applied")
                }
                query_params[key] = "1"
                continue
            }

            extraParams[key]?.let { query_params[key] = it }
        }

        return query_params
    }

    /**
     * Merges extra parameters into the existing event parameters.
     *
     * @param baseParams The base parameters for the event.
     * @param newParams Additional parameters to be added.
     * @return A combined map of parameters.
     */
    private fun addExtraParams(
        baseParams: Map<String, Any>?,
        newParams: Map<String, Any>
    ): Map<String, Any> {
        return (baseParams ?: emptyMap()).toMutableMap().apply { putAll(newParams) }
    }
}