package com.ai.osmos

import com.ai.osmos.utils.ConfigManager
import android.content.Context
import android.util.Log
import com.ai.osmos.utils.Config

/**
 * Created by adeshmukh on 20/02/25.
 * Project Name: OSMOS-Android-SDK
 * File Name: OsmosSDK
 *
 * Singleton object representing the Osmos SDK
 */

public object OsmosSDK {

    internal var applicationContext: Context? = null

    /**
     * Initializes the Osmos SDK.
     *
     * @param context The application context required for initialization.
     * @param osmosConfig Resource ID of the configuration file (e.g., R.raw.osmos_config).
     *
     * Example:
     * ```
     * initialize(context, R.raw.osmos_config)
     * ```
     */
    fun initialize(context: Context, osmosConfig: Int) {
        applicationContext = context.applicationContext

        // Load the configuration file using ConfigManager
        ConfigManager.loadConfig(context, osmosConfig)
    }
    /**
     * Initializes the SDK using a config map (typically from API or dynamic source).
     *
     * @param context - The application context.
     * @param configMap - A map containing configuration data. Keys must match expected config structure.
     *
     * Example:
     * ```
     * val configMap = mapOf(
     *     "client_id" to "123456",
     *     "debug" to true,
     *     "display_ads" to mapOf(
     *         "domain_uri" to "https://example.com",
     *         "connection_timeout" to 500
     *     )
     * )
     * ```
     */
    fun initialize(context: Context, configMap: Map<String,Any>) {
        applicationContext = context.applicationContext
        ConfigManager.loadConfigFromMap(configMap)
    }

    fun initialize(context: Context, config: Config) {
        applicationContext = context.applicationContext
        ConfigManager.loadConfigFromData(context, config)
    }
    /**
     * Returns application context if initialized.
     * @return Application context or null
     */
    fun getContext(): Context? {
        if (applicationContext == null) {
            Log.e("OSMOS", "OsmosSDK is not initialized. Initialize the SDK before accessing context.")
        }
        return applicationContext
    }
}
