package com.ai.osmos.network

import android.util.Log
import com.ai.osmos.utils.ConfigManager
import com.ai.osmos.utils.Constants
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import okhttp3.*
import okhttp3.MediaType.Companion.toMediaType
import org.json.JSONException
import org.json.JSONObject
import java.io.IOException
import java.util.concurrent.TimeUnit

/**
 * Created by adeshmukh on 07/02/25.
 * Project Name: OSMOS-Android-SDK
 * File Name: WebService
 */

/**
 * Enum class to standardize error codes and messages within WebService.
 */
private enum class WebServiceError(val code: String, val description: String) {
    BAD_URL("ERROR_BAD_URL", "Invalid or malformed URL."),
    INVALID_REQUEST("ERROR_INVALID_REQUEST", "Improper request format or missing parameters."),
    BAD_RESPONSE("ERROR_BAD_RESPONSE", "Invalid or unprocessable server response."),
    BAD_STATUS("ERROR_BAD_STATUS", "Unsuccessful HTTP status code."),
    FAILED_TO_DECODE_RESPONSE("ERROR_FAILED_TO_DECODE_RESPONSE", "The application failed to decode or parse the server response.");

    companion object {
        fun getErrorResponse(error: WebServiceError, details: String? = null): Map<String,Any> {
            return mutableMapOf<String, Any>().apply {
                put("status", false) // Indicating failure
                put("response", JSONObject.NULL) // Ensuring response is null for errors
                put("error", mapOf(
                    "code" to error.code,
                    "data" to (details ?: error.description)
                ))
            }
        }
    }
}

/**
 * Handles API requests for fetching ads from a remote server using OkHttp.
 */
internal class WebService {

    companion object {
        private val JSON_MEDIA_TYPE = "application/json; charset=utf-8".toMediaType()
    }

    private val okHttpClient: OkHttpClient by lazy {
        val builder = OkHttpClient.Builder()
            .connectTimeout(Constants.CONNECT_TIMEOUT.toLong(), TimeUnit.MILLISECONDS)
            .readTimeout(Constants.READ_TIMEOUT.toLong(), TimeUnit.MILLISECONDS)
            .writeTimeout(Constants.READ_TIMEOUT.toLong(), TimeUnit.MILLISECONDS)
            .retryOnConnectionFailure(true)

        // Note: Manual logging is done in callApi method instead of using interceptor
        builder.build()
    }

    /**
     * Makes an API request using OkHttp.
     *
     * @param apiUrl The URL for the request.
     * @param methodType The HTTP request method ("GET", "POST").
     * @param requestBody The request body as a Map (converted to JSON).
     * @param requestHeaders The request headers as a Map.
     * @param callback The callback interface to handle API responses.
     */
    internal suspend fun callApi(
        apiUrl: String,
        methodType: String,
        requestHeaders: Map<String, Any>? = null,
        requestBody: Map<String, Any?>? = null,
        callback: WebServiceInterface
    ) {
        withContext(Dispatchers.IO) {
            try {
                if(ConfigManager.isDebugEnabled()) {
                    // Log URL as DEBUG
                    Log.d("OSMOS", "OSMOS: API URL = $apiUrl")
                    // Log method as DEBUG
                    Log.d("OSMOS", "Api call Method = $methodType")
                    // Log parameters as DEBUG
                    Log.d("OSMOS", "Api parameters = $requestBody")
                    // Log headers as DEBUG
                    Log.d("OSMOS", "Api headers = $requestHeaders")
                }
                
                val request = buildRequest(apiUrl, methodType, requestHeaders, requestBody)

                okHttpClient.newCall(request).enqueue(object : Callback {
                    override fun onFailure(call: Call, e: IOException) {
                        when (e) {
                            is java.net.MalformedURLException -> {
                                callback.didFinish(WebServiceError.getErrorResponse(WebServiceError.BAD_URL))
                            }
                            is java.net.SocketTimeoutException -> {
                                callback.didFinish(WebServiceError.getErrorResponse(WebServiceError.BAD_RESPONSE, "Request timeout"))
                            }
                            else -> {
                                callback.didFinish(WebServiceError.getErrorResponse(WebServiceError.BAD_RESPONSE, e.message))
                            }
                        }
                    }

                    override fun onResponse(call: Call, response: Response) {
                        response.use { res ->
                            try {
                                val responseCode = res.code
                                val responseBody = res.body?.string() ?: ""

                                // No Content (204, 205, 304) should be a success response with an empty data string
                                val noContentStatusCodes = setOf(204, 205, 304)

                                if (res.isSuccessful) {
                                    if (responseCode in noContentStatusCodes || responseBody.isEmpty()) {
                                        callback.didFinish(createResponse(true, responseCode, "", null))
                                    } else {
                                        callback.didFinish(createResponse(true, responseCode, responseBody, null))
                                    }
                                } else {
                                    callback.didFinish(WebServiceError.getErrorResponse(
                                        WebServiceError.BAD_STATUS,
                                        "HTTP $responseCode: ${res.message}"
                                    ))
                                }
                            } catch (e: JSONException) {
                                callback.didFinish(WebServiceError.getErrorResponse(WebServiceError.FAILED_TO_DECODE_RESPONSE))
                            } catch (e: Exception) {
                                callback.didFinish(WebServiceError.getErrorResponse(WebServiceError.BAD_RESPONSE, e.message))
                            }
                        }
                    }
                })
            } catch (e: Exception) {
                when (e) {
                    is IllegalArgumentException -> {
                        callback.didFinish(WebServiceError.getErrorResponse(WebServiceError.BAD_URL))
                    }
                    is JSONException -> {
                        callback.didFinish(WebServiceError.getErrorResponse(WebServiceError.INVALID_REQUEST))
                    }
                    else -> {
                        callback.didFinish(WebServiceError.getErrorResponse(WebServiceError.BAD_RESPONSE, e.message))
                    }
                }
            }
        }
    }

    /**
     * Builds an OkHttp Request object based on the provided parameters.
     */
    private fun buildRequest(
        apiUrl: String,
        methodType: String,
        requestHeaders: Map<String, Any>?,
        requestBody: Map<String, Any?>?
    ): Request {
        val requestBuilder = Request.Builder()
            .url(apiUrl)
            .addHeader("Accept", "application/json")

        // Add custom headers if provided
        requestHeaders?.forEach { (key, value) ->
            requestBuilder.addHeader(key, value.toString())
        }

        // Set request method and body
        when (methodType.uppercase()) {
            "GET" -> requestBuilder.get()
            "POST" -> {
                val body = if (!requestBody.isNullOrEmpty()) {
                    val jsonString = JSONObject(requestBody).toString()
                    RequestBody.create(JSON_MEDIA_TYPE, jsonString)
                } else {
                    RequestBody.create(JSON_MEDIA_TYPE, "{}")
                }
                requestBuilder.post(body)
            }
            "PUT" -> {
                val body = if (!requestBody.isNullOrEmpty()) {
                    val jsonString = JSONObject(requestBody).toString()
                    RequestBody.create(JSON_MEDIA_TYPE, jsonString)
                } else {
                    RequestBody.create(JSON_MEDIA_TYPE, "{}")
                }
                requestBuilder.put(body)
            }
            "DELETE" -> requestBuilder.delete()
            else -> throw IllegalArgumentException("Unsupported HTTP method: $methodType")
        }

        return requestBuilder.build()
    }

    /**
     * Creates a standardized response JSON.
     */
    private fun createResponse(status: Boolean, code: Int, data: String?, error: WebServiceError?): Map<String, Any> {
        return mutableMapOf<String, Any>().apply {
            put("status", status)

            put("response", mapOf(
                "code" to code,
                "data" to (data ?: "")
            ))

            put("error", error?.let {
                mapOf(
                    "code" to it.code,
                    "description" to it.description
                )
            } ?: JSONObject.NULL) // Keeping null handling for error
        }
    }
}