package com.ai.osmos.utils

import android.content.Context
import android.util.Log
import org.json.JSONException
import org.json.JSONObject

/**
 * Created by adeshmukh on 10/02/25.
 * Project Name: OSMOS-Android-SDK
 * File Name: ConfigManager
 */
object ConfigManager {
    private const val TAG = "ConfigManager"

    // Thread-safe fields using @Volatile
    @Volatile
    private var configData: JSONObject? = null

    @Volatile
    private var structuredConfig: Config? = null

    @Volatile
    private var isDebugMode: Boolean = false

    // Synchronization lock for load operations
    private val loadLock = Any()

    // Singleton config instance (read-only outside)
    val config: Config
        get() = structuredConfig ?: throw IllegalStateException("Config not loaded")


    /**
     * Loads configuration from a raw resource file.
     * Thread-safe method that prevents race conditions during initialization.
     * @param context - Application context.
     * @param resourceId - Resource ID of the config JSON file (e.g., R.raw.osmos_config).
     */
    fun loadConfig(context: Context, resourceId: Int) {
        synchronized(loadLock) {
            if (resourceId <= 0) {
                Log.e(TAG, "OSMOS: osmos_config.json not found in project root.")
                return
            }

            try {
                context.resources.openRawResource(resourceId).use { inputStream ->
                    val jsonString = inputStream.bufferedReader().use { it.readText() }
                    val newConfigData = JSONObject(jsonString)
                    val newDebugMode = newConfigData.optBoolean("debug", false)
                    val newStructuredConfig = parseStructuredConfig(newConfigData)

                    // Atomically update all fields to maintain consistency
                    configData = newConfigData
                    isDebugMode = newDebugMode
                    structuredConfig = newStructuredConfig
                }
            } catch (e: Exception) {
                logConfigError(e)
            }
        }
    }

    /**
     * Loads configuration directly from a JSON string.
     * Thread-safe method that prevents race conditions during initialization.
     * @param jsonString - JSON-formatted configuration string.
     */
    fun loadConfigFromString(jsonString: String) {
        synchronized(loadLock) {
            if (jsonString.isEmpty()) {
                Log.e(TAG, "OSMOS: Configuration string is empty.")
                // Clear configuration data when empty string is provided
                configData = null
                structuredConfig = null
                isDebugMode = false
                return
            }

            try {
                val newConfigData = JSONObject(jsonString)
                val newDebugMode = newConfigData.optBoolean("debug", false)
                val newStructuredConfig = parseStructuredConfig(newConfigData)

                // Atomically update all fields to maintain consistency
                configData = newConfigData
                isDebugMode = newDebugMode
                structuredConfig = newStructuredConfig
            } catch (e: Exception) {
                logConfigError(e)
            }
        }
    }

    /**
     * Loads configuration from a map (commonly passed from native or testing layers).
     * Thread-safe method that prevents race conditions during initialization.
     * @param configMap - Configuration map.
     */
    fun loadConfigFromMap(configMap: Map<String, Any>) {
        synchronized(loadLock) {
            if (configMap.isEmpty()) {
                Log.e(TAG, "OSMOS: Configuration map is empty.")
                // Clear configuration data when empty map is provided
                configData = null
                structuredConfig = null
                isDebugMode = false
                return
            }
            try {
                val newConfigData = JSONObject(configMap)
                val newDebugMode = newConfigData.optBoolean("debug", false)
                val newStructuredConfig = parseStructuredConfig(newConfigData)

                // Atomically update all fields to maintain consistency
                configData = newConfigData
                isDebugMode = newDebugMode
                structuredConfig = newStructuredConfig
            } catch (e: Exception) {
                logConfigError(e)
            }
        }
    }

    fun loadConfigFromData(context: Context, config: Config) {
        if (config == null) {
            Log.e(TAG, "OSMOS: Configuration map is empty.")
            return
        }
        try {
            isDebugMode = config.debug
            structuredConfig = config
        } catch (e: Exception) {
            logConfigError(e)
        }
    }
    /**
     * Logs configuration parsing error and example format.
     * Thread-safe method that handles error cleanup atomically.
     */
    private fun logConfigError(e: Exception) {
        val sampleJson = """
            {
              "client_id": "<osmos_client_id>",
              "debug": false,
              "register_event": {
                "domain_uri": "<osmos_register_event_domain>",
                "connection_timeout": 500
              }
            }
        """.trimIndent()

        Log.e(TAG, "OSMOS: Error reading or parsing config: ${e.message}")
        Log.e(TAG, "OSMOS: osmos_config.json should be in the following format: $sampleJson")
        
        // Atomically clear all configuration data
        synchronized(loadLock) {
            configData = null
            structuredConfig = null
            isDebugMode = false
        }
    }

    /**
     * Checks whether the config is loaded and available.
     * Thread-safe method using volatile field access.
     */
    fun isConfigLoaded(): Boolean = configData != null

    /**
     * Returns the debug mode flag.
     * Thread-safe method using volatile field access.
     */
    fun isDebugEnabled(): Boolean = isDebugMode

    /**
     * Returns the strongly typed structured configuration.
     * Thread-safe method that throws exception if config not loaded.
     */
    fun getStructuredConfig(): Config {
        return structuredConfig ?: throw IllegalStateException("Config not loaded.")
    }

    /**
     * Retrieves a string value from the JSON config.
     * Thread-safe method with synchronized access to configData.
     * @param key - The key to retrieve
     * @return String value or throws JSONException if key not found
     */
    fun getString(key: String): String {
        synchronized(loadLock) {
            return configData?.getString(key)
                ?: throw JSONException("Key '$key' not found in config")
        }
    }

    /**
     * Converts a raw JSONObject into a structured [Config] data class.
     * Handles optional blocks gracefully.
     */
    private fun parseStructuredConfig(jsonObject: JSONObject): Config {
        return Config(
            clientId = jsonObject.optString("client_id", ""),
            debug = jsonObject.optBoolean("debug", false),
            displayAds = jsonObject.optJSONObject("display_ads")?.let { parseAdConfigDisplay(it) },
            plaAds = jsonObject.optJSONObject("pla_ads")?.let { parseAdConfigPLA(it) },
            registerEvent = jsonObject.optJSONObject("register_event")?.let { parseAdConfigRegisterEvent(it) }
        )
    }


    /**
     * Converts a single ad configuration JSON block into an [AdConfig] object.
     */
    private fun parseAdConfigPLA(obj: JSONObject): PLAAdConfig {
        return PLAAdConfig(
            domainUri = obj.optString("domain_uri", ""),
            connectionTimeout = obj.optInt("connection_timeout", 500)
        )
    }

    private fun parseAdConfigRegisterEvent(obj: JSONObject): RegisterEventConfig {
        return RegisterEventConfig(
            domainUri = obj.optString("domain_uri", ""),
            connectionTimeout = obj.optInt("connection_timeout", 500),
            videoProgressSec = obj.optInt("video_progress_sec")
        )
    }

    private fun parseAdConfigDisplay(obj: JSONObject): DisplayAdConfig {
        return DisplayAdConfig(
            domainUri = obj.optString("domain_uri", ""),
            connectionTimeout = obj.optInt("connection_timeout", 500),
            closeButtonSec = obj.optInt("close_button_sec"),
            pipAdHeight = obj.optInt("pip_ad_height"),
            pipAdWidth = obj.optInt("pip_ad_width")
        )
    }
}
