package com.ai.osmos.utils
import android.content.Context
import android.graphics.Color
import android.graphics.PorterDuff
import android.graphics.drawable.GradientDrawable
import android.view.Gravity
import android.view.View
import android.widget.FrameLayout
import android.widget.ImageButton
import android.widget.ImageView
import android.widget.TextView
import com.ai.osmos.OsmosSDK

/**
 * Created by adeshmukh on 09/04/25.
 * Project Name: OSMOS-Android-SDK
 * File Name: ViewUtils
 */

/**
 * Utility object for common view-related helpers used across the SDK.
 */
object ViewUtils {

    /**
     * Returns a placeholder empty [View] to be shown when no ad is available or fails to load.
     *
     * This view is:
     * - A hidden [TextView] with empty text.
     * - Centered inside a [FrameLayout].
     * - Marked as [View.GONE] by default.
     *
     * @return A [View] instance representing an invisible empty view.
     */
    fun getEmptyView(context: Context): View {
        return TextView(context).apply {
            text = ""
            layoutParams = FrameLayout.LayoutParams(
                FrameLayout.LayoutParams.WRAP_CONTENT,
                FrameLayout.LayoutParams.WRAP_CONTENT
            ).apply {
                gravity = Gravity.CENTER
            }
            visibility = View.GONE
        }
    }

    fun createStyledButton(
        context: Context,
        iconRes: Int,
        size: Int,
        gravity: Int,
        marginStart: Int = 0,
        marginTop: Int = 0,
        marginEnd: Int = 0,
        marginBottom: Int = 0,
        onClick: (() -> Unit)? = null
    ): ImageButton {

        return ImageButton(context).apply {
            layoutParams = FrameLayout.LayoutParams(size, size).apply {
                this.gravity = gravity
                setMargins(marginStart, marginTop, marginEnd, marginBottom)
            }

            setImageResource(iconRes)
            setColorFilter(Color.BLACK, PorterDuff.Mode.SRC_IN)
            scaleType = ImageView.ScaleType.FIT_CENTER
            background = GradientDrawable().apply {
                shape = GradientDrawable.OVAL
                setStroke(3, Color.BLACK)
                setColor(Color.WHITE)
            }

            setOnClickListener { onClick?.invoke() }
        }
    }

    internal fun dpToPx(context: Context, dp: Int): Int {
        val density = context.resources.displayMetrics.density
        return (dp * density).toInt()
    }
}